/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

/*
 *   assignment to out via operator
 *
 *   Tests:
 *           single rate out / single rate in  (vector,vector,?in?)
 *                                             scalar out / scalar in
 *                                             port out   / scalar in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           single rate out / multi rate in  (vector,vector,?in?)
 *                                             scalar out / vector in
 *                                             port out   / vector in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           multi rate out / single rate in  (vector,vector, sc_core::sc_time,?in?)
 *           //delay is required due causality (sc_time argument)
 *                                             vector out / scalar in
 *                                             port out   / scalar in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 *           multi rate out / multi rate in  (vector,vector,?in?)
 *                                             vector out / vector in
 *                                             port out   / vector in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 */


#include <systemc-ams>
#include "test_utilities.h"

#include <string>
#include <fstream>
#include <sstream>

///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(ltf_zp_src)
{
  // single rate
  sca_tdf::sca_out<double> outp_const_1; // constant value (1.0)
  sca_tdf::sca_out<double> outp_ramp_1;  // ramp  y=t
  sca_tdf::sca_out<double> outp_t2_1;    // y=0.5*t*t
  sca_tdf::sca_out<double> outp_sin_1;
  sca_tdf::sca_out<double> outp_t3_1;    // y=0.5/3*t*t*t

  // multi rate
  sca_tdf::sca_out<double> outp_const_r;
  sca_tdf::sca_out<double> outp_ramp_r;
  sca_tdf::sca_out<double> outp_t2_r;
  sca_tdf::sca_out<double> outp_sin_r;
  sca_tdf::sca_out<double> outp_t3_r;

  // single rate
  sca_tdf::sca_out<double> outp0;
  sca_tdf::sca_out<double> outp1;
  sca_tdf::sca_out<double> outp2;
  sca_tdf::sca_out<double> outp3;
  sca_tdf::sca_out<double> outp4;

  // multi rate
  sca_tdf::sca_out<double> outpr0;
  sca_tdf::sca_out<double> outpr1;
  sca_tdf::sca_out<double> outpr2;
  sca_tdf::sca_out<double> outpr3;
  sca_tdf::sca_out<double> outpr4;

  struct params
  {
    int rate;
    sca_core::sca_time sample_time;
    double f_sin;

    params()
    {
      rate = 5;
      sample_time = sca_core::sca_time(1.0, sc_core::SC_US);
      f_sin = 13.3e3;
    }
  };

  void set_attributes()
  {
    set_timestep(p.sample_time);

    outp_const_r.set_rate(p.rate);
    outp_ramp_r.set_rate(p.rate);
    outp_t2_r.set_rate(p.rate);
    outp_sin_r.set_rate(p.rate);
    outp_t3_r.set_rate(p.rate);

    outpr0.set_rate(p.rate);
    outpr1.set_rate(p.rate);
    outpr2.set_rate(p.rate);
    outpr3.set_rate(p.rate);
    outpr4.set_rate(p.rate);
  }

  void initialize();
  void processing();

  ltf_zp_src( sc_core::sc_module_name nm, params pa = params() )
  : p(pa)
  {}

  params p;

private:

  sca_tdf::sca_ltf_zp  zp10, zp11, zp12, zp13, zp14;
  sca_tdf::sca_ltf_zp  zp20, zp21, zp22, zp23, zp24;
  sca_tdf::sca_ltf_zp  zp30, zp31, zp32, zp33, zp34;
  sca_tdf::sca_ltf_zp  zp40, zp41, zp42, zp43, zp44;

  sca_util::sca_vector<sca_util::sca_complex> Z1, P1;
  sca_util::sca_vector<sca_util::sca_complex> Z2, P2;
  sca_util::sca_vector<sca_util::sca_complex> Z3, P3;
  sca_util::sca_vector<sca_util::sca_complex> Z4, P4;
};


void ltf_zp_src::initialize()
{
  //Z1 / P1     // y=x

  // Z2
  P2(0) = 0.0;  // y=t*x - x=const=1.0

  // Z3
  P3(0) = 0.0;
  P3(1) = 0.0;  // y=0.5*t*t*x - x=const=1.0

  //Z4;
  P4(0) = 0.0;  // y=0.5/3.0*t*t*t*x - x=const=1.0
  P4(1) = 0.0;
  P4(2) = 0.0;
}

// time domain implementation
void ltf_zp_src::processing()
{
  double inp;
  inp = std::sin(2.0 * M_PI * p.f_sin * get_time().to_seconds());

  // output to scalar port sca_tdf::sca_out<double>
  // input double
  outp0 = zp10(Z1, P1, 1.0); // single rate
  outp1 = zp11(Z2, P2, 1.0);
  outp2 = zp12(Z3, P3, 1.0);
  outp3 = zp13(Z1, P1, inp);
  outp4 = zp14(Z4, P4, 1.0);

  // output to port sca_tdf::sca_out<double >
  outpr0 = zp20(Z1, P1, sc_core::sc_time(0.8, sc_core::SC_US), 1.0); // multi rate with 0.8us delay
  outpr1 = zp21(Z2, P2, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  outpr2 = zp22(Z3, P3, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  outpr3 = zp23(Z1, P1, sc_core::sc_time(0.8, sc_core::SC_US), inp);
  outpr4 = zp24(Z4, P4, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);

  double outv0, outv1, outv2, outv3, outv4;

  // output to double
  // input double
  outv0 = zp30(Z1, P1, 1.0);
  outv1 = zp31(Z2, P2, 1.0);
  outv2 = zp32(Z3, P3, 1.0);
  outv3 = zp33(Z1, P1, inp);
  outv4 = zp34(Z4, P4, 1.0);

  outp_const_1 = outv0;
  outp_ramp_1  = outv1;
  outp_t2_1    = outv2;
  outp_sin_1   = outv3;
  outp_t3_1    = outv4;

  sca_util::sca_vector<double> out_vec0(p.rate);
  sca_util::sca_vector<double> out_vec1(p.rate);
  sca_util::sca_vector<double> out_vec2(p.rate);
  sca_util::sca_vector<double> out_vec3(p.rate);
  sca_util::sca_vector<double> out_vec4(p.rate);

  // output to vector sca_util::sca_vector<double>
  // input double value
  out_vec0 = zp40(Z1, P1, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  out_vec1 = zp41(Z2, P2, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  out_vec2 = zp42(Z3, P3, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);
  out_vec3 = zp43(Z1, P1, sc_core::sc_time(0.8, sc_core::SC_US), inp);
  out_vec4 = zp44(Z4, P4, sc_core::sc_time(0.8, sc_core::SC_US), 1.0);

  for (long i = 0; i < p.rate; i++)
  {
    outp_const_r[i] = out_vec0(i);
    outp_ramp_r[i]  = out_vec1(i);
    outp_t2_r[i]    = out_vec2(i);
    outp_sin_r[i]   = out_vec3(i);
    outp_t3_r[i]    = out_vec4(i);
  }
}

////////////////////////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE (ltf_zp_test)
{
  // single rate
  sca_tdf::sca_in<double> in_const_1; // constant value (1.0)
  sca_tdf::sca_in<double> in_ramp_1;  // ramp  y=t
  sca_tdf::sca_in<double> in_t2_1;    // y=0.5*t*t
  sca_tdf::sca_in<double> in_sin_1;

  sca_tdf::sca_out<double> out_const_1;
  sca_tdf::sca_out<double> out_ramp_1;
  sca_tdf::sca_out<double> out_t2_1;
  sca_tdf::sca_out<double> out_sin_1;
  sca_tdf::sca_out<double> out_t3_1;

  sca_tdf::sca_out<double> out_const_3;
  sca_tdf::sca_out<double> out_ramp_3;
  sca_tdf::sca_out<double> out_t2_3;
  sca_tdf::sca_out<double> out_sin_3;
  sca_tdf::sca_out<double> out_t3_3;

  sca_tdf::sca_out<double> out_const_5;
  sca_tdf::sca_out<double> out_ramp_5;
  sca_tdf::sca_out<double> out_t2_5;
  sca_tdf::sca_out<double> out_sin_5;
  sca_tdf::sca_out<double> out_t3_5;

  sca_tdf::sca_out<double> out_const_7;
  sca_tdf::sca_out<double> out_ramp_7;
  sca_tdf::sca_out<double> out_t2_7;
  sca_tdf::sca_out<double> out_sin_7;
  sca_tdf::sca_out<double> out_t3_7;

  // multi rate
  sca_tdf::sca_in<double> in_const_r;
  sca_tdf::sca_in<double> in_ramp_r;
  sca_tdf::sca_in<double> in_t2_r;
  sca_tdf::sca_in<double> in_sin_r;

  // single rate
  sca_tdf::sca_in<double> in_10;
  sca_tdf::sca_in<double> in_11;
  sca_tdf::sca_in<double> in_12;
  sca_tdf::sca_in<double> in_13;

  sca_tdf::sca_out<double> out_20;
  sca_tdf::sca_out<double> out_21;
  sca_tdf::sca_out<double> out_22;
  sca_tdf::sca_out<double> out_23;
  sca_tdf::sca_out<double> out_24;

  sca_tdf::sca_out<double> out_40;
  sca_tdf::sca_out<double> out_41;
  sca_tdf::sca_out<double> out_42;
  sca_tdf::sca_out<double> out_43;
  sca_tdf::sca_out<double> out_44;

  sca_tdf::sca_out<double> out_60;
  sca_tdf::sca_out<double> out_61;
  sca_tdf::sca_out<double> out_62;
  sca_tdf::sca_out<double> out_63;
  sca_tdf::sca_out<double> out_64;

  sca_tdf::sca_out<double> out_80;
  sca_tdf::sca_out<double> out_81;
  sca_tdf::sca_out<double> out_82;
  sca_tdf::sca_out<double> out_83;
  sca_tdf::sca_out<double> out_84;

  // multi rate
  sca_tdf::sca_in<double> in_r_0;
  sca_tdf::sca_in<double> in_r_1;
  sca_tdf::sca_in<double> in_r_2;
  sca_tdf::sca_in<double> in_r_3;

  sca_tdf::sca_out<double> out_r_90;
  sca_tdf::sca_out<double> out_r_91;
  sca_tdf::sca_out<double> out_r_92;
  sca_tdf::sca_out<double> out_r_93;
  sca_tdf::sca_out<double> out_r_94;


  sca_tdf::sca_out<double> out_r_100;
  sca_tdf::sca_out<double> out_r_101;
  sca_tdf::sca_out<double> out_r_102;
  sca_tdf::sca_out<double> out_r_103;
  sca_tdf::sca_out<double> out_r_104;

  sca_tdf::sca_out<double> out_r_110;
  sca_tdf::sca_out<double> out_r_111;
  sca_tdf::sca_out<double> out_r_112;
  sca_tdf::sca_out<double> out_r_113;
  sca_tdf::sca_out<double> out_r_114;

  sca_tdf::sca_out<double> out_r_120;
  sca_tdf::sca_out<double> out_r_121;
  sca_tdf::sca_out<double> out_r_122;
  sca_tdf::sca_out<double> out_r_123;
  sca_tdf::sca_out<double> out_r_124;

  sca_tdf::sca_out<double> out_r_130;
  sca_tdf::sca_out<double> out_r_131;
  sca_tdf::sca_out<double> out_r_132;
  sca_tdf::sca_out<double> out_r_133;
  sca_tdf::sca_out<double> out_r_134;

  sca_tdf::sca_out<double> out_r_140;
  sca_tdf::sca_out<double> out_r_141;
  sca_tdf::sca_out<double> out_r_142;
  sca_tdf::sca_out<double> out_r_143;
  sca_tdf::sca_out<double> out_r_144;

  sca_tdf::sca_out<double> out_r_150;
  sca_tdf::sca_out<double> out_r_151;
  sca_tdf::sca_out<double> out_r_152;
  sca_tdf::sca_out<double> out_r_153;
  sca_tdf::sca_out<double> out_r_154;

  sca_tdf::sca_out<double> out_r_160;
  sca_tdf::sca_out<double> out_r_161;
  sca_tdf::sca_out<double> out_r_162;
  sca_tdf::sca_out<double> out_r_163;
  sca_tdf::sca_out<double> out_r_164;

  struct params // parameter
  {
    unsigned long rate, rate2;

    params() // default for parameter
    {
      rate = 5;
      rate2 = 3;
    }
  };

  void initialize();     // initialization
  void processing();     // time domain processing method
  void set_attributes();
  params p;

  // constructor
  ltf_zp_test(sc_core::sc_module_name nm, params pa = params()) : p(pa)
  {}

  // definition of local variables
private:

  sca_tdf::sca_ltf_zp zp10, zp11, zp12, zp13, zp14;
  sca_tdf::sca_ltf_zp zp20, zp21, zp22, zp23, zp24;
  sca_tdf::sca_ltf_zp zp30, zp31, zp32, zp33, zp34;
  sca_tdf::sca_ltf_zp zp40, zp41, zp42, zp43, zp44;
  sca_tdf::sca_ltf_zp zp50, zp51, zp52, zp53, zp54;
  sca_tdf::sca_ltf_zp zp60, zp61, zp62, zp63, zp64;
  sca_tdf::sca_ltf_zp zp70, zp71, zp72, zp73, zp74;
  sca_tdf::sca_ltf_zp zp80, zp81, zp82, zp83, zp84;
  sca_tdf::sca_ltf_zp zp90, zp91, zp92, zp93, zp94;
  sca_tdf::sca_ltf_zp zp100, zp101, zp102, zp103, zp104;
  sca_tdf::sca_ltf_zp zp110, zp111, zp112, zp113, zp114;
  sca_tdf::sca_ltf_zp zp120, zp121, zp122, zp123, zp124;
  sca_tdf::sca_ltf_zp zp130, zp131, zp132, zp133, zp134;
  sca_tdf::sca_ltf_zp zp140, zp141, zp142, zp143, zp144;
  sca_tdf::sca_ltf_zp zp150, zp151, zp152, zp153, zp154;
  sca_tdf::sca_ltf_zp zp160, zp161, zp162, zp163, zp164;
  sca_tdf::sca_ltf_zp zp_delay0, zp_delay1, zp_delay2, zp_delay3, zp_delay4;
  sca_util::sca_vector<sca_util::sca_complex> Z, P, Z2, P2;

  double cnt;

};

void ltf_zp_test::set_attributes()
{
  in_const_r.set_rate(p.rate);
  in_ramp_r.set_rate(p.rate);
  in_t2_r.set_rate(p.rate);
  in_sin_r.set_rate(p.rate);

  in_r_0.set_rate(p.rate);
  in_r_1.set_rate(p.rate);
  in_r_2.set_rate(p.rate);
  in_r_3.set_rate(p.rate);

  out_r_90.set_rate(p.rate);
  out_r_91.set_rate(p.rate);
  out_r_92.set_rate(p.rate);
  out_r_93.set_rate(p.rate);
  out_r_94.set_rate(p.rate);

  out_r_100.set_rate(p.rate);
  out_r_101.set_rate(p.rate);
  out_r_102.set_rate(p.rate);
  out_r_103.set_rate(p.rate);
  out_r_104.set_rate(p.rate);

  out_r_110.set_rate(p.rate);
  out_r_111.set_rate(p.rate);
  out_r_112.set_rate(p.rate);
  out_r_113.set_rate(p.rate);
  out_r_114.set_rate(p.rate);

  out_r_120.set_rate(p.rate);
  out_r_121.set_rate(p.rate);
  out_r_122.set_rate(p.rate);
  out_r_123.set_rate(p.rate);
  out_r_124.set_rate(p.rate);

  out_r_130.set_rate(p.rate2);
  out_r_131.set_rate(p.rate2);
  out_r_132.set_rate(p.rate2);
  out_r_133.set_rate(p.rate2);
  out_r_134.set_rate(p.rate2);

  out_r_140.set_rate(p.rate2);
  out_r_141.set_rate(p.rate2);
  out_r_142.set_rate(p.rate2);
  out_r_143.set_rate(p.rate2);
  out_r_144.set_rate(p.rate2);

  out_r_150.set_rate(p.rate2);
  out_r_151.set_rate(p.rate2);
  out_r_152.set_rate(p.rate2);
  out_r_153.set_rate(p.rate2);
  out_r_154.set_rate(p.rate2);

  out_r_160.set_rate(p.rate2);
  out_r_161.set_rate(p.rate2);
  out_r_162.set_rate(p.rate2);
  out_r_163.set_rate(p.rate2);
  out_r_164.set_rate(p.rate2);
}

void ltf_zp_test::initialize()
{
  // Z / P -> H(s) = 1.0

  P2(0) = 1.0;  // H(s)=1/s

  zp_delay0.set_max_delay((2 * in_ramp_r.get_rate() - 1) * in_ramp_r.get_timestep());
  zp_delay1.set_max_delay((2 * in_ramp_r.get_rate() - 1) * in_ramp_r.get_timestep());
  zp_delay2.set_max_delay((2 * in_ramp_r.get_rate() - 1) * in_ramp_r.get_timestep());
  zp_delay3.set_max_delay((2 * in_ramp_r.get_rate() - 1) * in_ramp_r.get_timestep());
  zp_delay4.set_max_delay((2 * in_ramp_r.get_rate() - 1) * in_ramp_r.get_timestep());
}

// time domain implementation
void ltf_zp_test::processing()
{
  double x0;
  double x1;
  double x2;
  double x3;

  x0 = in_const_1.read();
  x1 = in_ramp_1.read();
  x2 = in_t2_1.read();
  x3 = in_sin_1.read();

  ////////// single rate in / single rate out //////////////

  // double out / double in
  const double y10 = zp10(Z, P, x0);
  const double y11 = zp11(Z, P, x1);
  const double y12 = zp12(Z, P, x2);
  const double y13 = zp13(Z, P, x3);
  const double y14 = zp14(Z2, P2, x2);

  out_const_1 = y10;
  out_ramp_1  = y11;
  out_t2_1    = y12;
  out_sin_1   = y13;
  out_t3_1    = y14;

  // port out / double in
  out_20 = zp20(Z, P, x0);
  out_21 = zp21(Z, P, x1);
  out_22 = zp22(Z, P, x2);
  out_23 = zp23(Z, P, x3);
  out_24 = zp24(Z2, P2, x2);

  // double out / port in
  const double y30 = zp30(Z, P, in_10);
  const double y31 = zp31(Z, P, in_11);
  const double y32 = zp32(Z, P, in_12);
  const double y33 = zp33(Z, P, in_13);
  const double y34 = zp34(Z2, P2, in_12);

  out_const_3 = y30;
  out_ramp_3  = y31;
  out_t2_3    = y32;
  out_sin_3   = y33;
  out_t3_3    = y34;

  // port out / port in
  out_40 = zp40(Z, P, in_10);
  out_41 = zp41(Z, P, in_11);
  out_42 = zp42(Z, P, in_12);
  out_43 = zp43(Z, P, in_13);
  out_44 = zp44(Z2, P2, in_12);

  ///////// multi rate in (rate=5) / single rate out ////////////////

  // -800  -600  -400  -200  0   time stamps (ps) of vector elements

  sca_util::sca_vector<double> x_vec0(p.rate);
  sca_util::sca_vector<double> x_vec1(p.rate);
  sca_util::sca_vector<double> x_vec2(p.rate);
  sca_util::sca_vector<double> x_vec3(p.rate);
  sca_util::sca_vector<double> x_vec4(p.rate);

  sca_util::sca_vector<double> x_vec0_delayed(p.rate);
  sca_util::sca_vector<double> x_vec1_delayed(p.rate);
  sca_util::sca_vector<double> x_vec2_delayed(p.rate);
  sca_util::sca_vector<double> x_vec3_delayed(p.rate);
  sca_util::sca_vector<double> x_vec4_delayed(p.rate);

  // this assignment corresponds to a delay of -0.8us (negative!!!)
  // assign time 0 -> -800, 200 -> -600, ...
  for (unsigned long i = 0; i < p.rate; i++)
  {
    x_vec0(i) = in_const_r.read(i);
    x_vec1(i) = in_ramp_r.read(i);
    x_vec2(i) = in_t2_r.read(i);
    x_vec3(i) = in_sin_r.read(i);
    x_vec4(i) = in_t2_r.read(i);
  }

  // remove negative delay of -0.8us
  // init init init init 0  | 200 400 600 800 1us | ...
  x_vec0_delayed = zp_delay0(Z, P, (in_const_r.get_rate()-1) * in_const_r.get_timestep(), x_vec0);
  x_vec1_delayed = zp_delay1(Z, P, (in_ramp_r.get_rate()-1)  * in_ramp_r.get_timestep(), x_vec1);
  x_vec2_delayed = zp_delay2(Z, P, (in_t2_r.get_rate()-1)    * in_t2_r.get_timestep(),   x_vec2);
  x_vec3_delayed = zp_delay3(Z, P, (in_sin_r.get_rate()-1)   * in_sin_r.get_timestep(),  x_vec3);
  x_vec4_delayed = zp_delay4(Z, P, (in_t2_r.get_rate()-1)    * in_t2_r.get_timestep(),   x_vec4);

  // double out / vector in (rate=5)
  // use delayed vector, due the last vector value corresponds to the out port value
  const double y50 = zp50(Z, P, x_vec0_delayed);
  const double y51 = zp51(Z, P, x_vec1_delayed);
  const double y52 = zp52(Z, P, x_vec2_delayed);
  const double y53 = zp53(Z, P, x_vec3_delayed);
  const double y54 = zp54(Z2, P2, x_vec4_delayed);

  out_const_5 = y50;
  out_ramp_5  = y51;
  out_t2_5    = y52;
  out_sin_5   = y53;
  out_t3_5    = y54;

  ////////////////////////////////////////////////////////////////////////

  // port out / vector in
  // use delayed vector, due the last vector value corresponds to the out port value
  out_60 = zp60(Z, P, x_vec0_delayed);
  out_61 = zp61(Z, P, x_vec1_delayed);
  out_62 = zp62(Z, P, x_vec2_delayed);
  out_63 = zp63(Z, P, x_vec3_delayed);
  out_64 = zp64(Z2, P2, x_vec4_delayed);

  ///////////////////////////////////////////////////////////////////////

  // double out / port in (rate=5)
  const double y70 = zp70(Z, P, in_r_0);
  const double y71 = zp71(Z, P, in_r_1);
  const double y72 = zp72(Z, P, in_r_2);
  const double y73 = zp73(Z, P, in_r_3);
  const double y74 = zp74(Z2, P2, in_r_2);

  out_const_7 = y70;
  out_ramp_7  = y71;
  out_t2_7    = y72;
  out_sin_7   = y73;
  out_t3_7    = y74;

  //port out (single rate) / port in (rate=5)
  out_80 = zp80(Z, P, in_r_0);
  out_81 = zp81(Z, P, in_r_1);
  out_82 = zp82(Z, P, in_r_2);
  out_83 = zp83(Z, P, in_r_3);
  out_84 = zp84(Z2, P2, in_r_2);

  ////////// single rate in / multi rate out (rate=5) /////////////

  // vector out / double in
  sca_util::sca_vector<double> y90(p.rate);
  sca_util::sca_vector<double> y91(p.rate);
  sca_util::sca_vector<double> y92(p.rate);
  sca_util::sca_vector<double> y93(p.rate);
  sca_util::sca_vector<double> y94(p.rate);

  y90 = zp90(Z, P, x0);
  y91 = zp91(Z, P, x1);
  y92 = zp92(Z, P, x2);
  y93 = zp93(Z, P, x3);
  y94 = zp94(Z2, P2, x2);


  // this assignmet corresponds to a delay of 0.8us
  // -800 -> 0, -600 -> 200, -400 -> 400, -200 -> 600, 0 -> 800
  for (unsigned long i = 0; i < p.rate; i++)
  {
    out_r_90[i] = y90(i);
    out_r_91[i] = y91(i);
    out_r_92[i] = y92(i);
    out_r_93[i] = y93(i);
    out_r_94[i] = y94(i);
  }

  // overall delay 0.8us

  /////////////////////////////////////////////////////////////////////////

  // port out (rate=5) / double in
  // 0.8us delay to achieve causality
  out_r_100 = zp100(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), x0);
  out_r_101 = zp101(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), x1);
  out_r_102 = zp102(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), x2);
  out_r_103 = zp103(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), x3);
  out_r_104 = zp104(Z2, P2, sc_core::sc_time(0.8, sc_core::SC_US), x2);

  // overall delay 0.8us

  //////////////////////

  // vector out (rate=5) / port in (single rate)
  sca_util::sca_vector<double> y110(p.rate);
  sca_util::sca_vector<double> y111(p.rate);
  sca_util::sca_vector<double> y112(p.rate);
  sca_util::sca_vector<double> y113(p.rate);
  sca_util::sca_vector<double> y114(p.rate);

  y110 = zp110(Z, P, in_10);
  y111 = zp111(Z, P, in_11);
  y112 = zp112(Z, P, in_12);
  y113 = zp113(Z, P, in_13);
  y114 = zp114(Z2, P2, in_12);

  // this assignment corresponds to 0.8us delay
  // -800 -> 0, -600 -> 200, -400 -> 400, -200 -> 600, 0 -> 800
  for (unsigned long i = 0; i <p.rate; i++)
  {
    out_r_110[i] = y110(i);
    out_r_111[i] = y111(i);
    out_r_112[i] = y112(i);
    out_r_113[i] = y113(i);
    out_r_114[i] = y114(i);
  }

  // overall delay 0.8us

  //////////////////////////////////////////////////////////

  // port out (rate 5) / port in (single rate)
  // 0.8us delay to achieve causality
  out_r_120 = zp120(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), in_10);
  out_r_121 = zp121(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), in_11);
  out_r_122 = zp122(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), in_12);
  out_r_123 = zp123(Z, P, sc_core::sc_time(0.8, sc_core::SC_US), in_13);
  out_r_124 = zp124(Z2, P2, sc_core::sc_time(0.8, sc_core::SC_US), in_12);

  // overall delay 0.8us

  ////////// multi rate in / multi rate out /////////////

  // vector out (rate=3) / vector in (rate=5)
  sca_util::sca_vector<double> y130(p.rate2);
  sca_util::sca_vector<double> y131(p.rate2);
  sca_util::sca_vector<double> y132(p.rate2);
  sca_util::sca_vector<double> y133(p.rate2);
  sca_util::sca_vector<double> y134(p.rate2);

  // 0     1     2     3    4              sample index
  // 0    200   400   600  800             time of inport sample
  //-800 -600  -400  -200  0               time of vector sample
  // 0    200   400   600  800             time of delayed vector
  //    -666      -333     0               time of out vector
  //                       0   333  666    time of out port

  // we use the delayed vector to remove the -0.8us (negative) delay from
  // the inport assignment
  y130 = zp130(Z, P, x_vec0_delayed);
  y131 = zp131(Z, P, x_vec1_delayed);
  y132 = zp132(Z, P, x_vec2_delayed);
  y133 = zp133(Z, P, x_vec3_delayed);
  y134 = zp134(Z2, P2, x_vec4_delayed);

  // this assignment corresponds to a delay of 0.666us
  // -666 -> 0, -333 -> 333, 0 -> 666
  for (unsigned long i = 0; i < p.rate2; i++)
  {
    out_r_130[i] = y130(i);
    out_r_131[i] = y131(i);
    out_r_132[i] = y132(i);
    out_r_133[i] = y133(i);
    out_r_134[i] = y134(i);
  }

  //-> the overall delay will be 0.666us

  ///////////////////////////////////////////////////////////////////////////
  ///////////////////////////////////////////////////////////////////////////

  // 0     1     2     3    4              sample index
  // 0    200   400   600  800             time of inport sample
  //-800 -600  -400  -200  0               time of vector sample
  //                       0    333   666  time of outport sample

  // to achieve, that inport==outport we must delay the vector by 800ps
  // port out (rate=3) / vector in (rate=5)
  // a delay is required due the vector time ends at 0 and the port time at 1us/3*2
  out_r_140 = zp140(Z, P, sca_core::sca_time(0.8, sc_core::SC_US), x_vec0);
  out_r_141 = zp141(Z, P, sca_core::sca_time(0.8, sc_core::SC_US), x_vec1);
  out_r_142 = zp142(Z, P, sca_core::sca_time(0.8, sc_core::SC_US), x_vec2);
  out_r_143 = zp143(Z, P, sca_core::sca_time(0.8, sc_core::SC_US), x_vec3);
  out_r_144 = zp144(Z2, P2, sca_core::sca_time(0.8, sc_core::SC_US), x_vec2);

  ///////////////////////////////////////////////////////////

  // vector out (rate=3) / port in (rate=5)
  sca_util::sca_vector<double> y150(p.rate2);
  sca_util::sca_vector<double> y151(p.rate2);
  sca_util::sca_vector<double> y152(p.rate2);
  sca_util::sca_vector<double> y153(p.rate2);
  sca_util::sca_vector<double> y154(p.rate2);

  // inport time stamps:     0 200 400 600 800
  // out vector timestamps: -666 -333 0 (init init 0) | 333 666
  y150 = zp150(Z, P, in_r_0);
  y151 = zp151(Z, P, in_r_1);
  y152 = zp152(Z, P, in_r_2);
  y153 = zp153(Z, P, in_r_3);
  y154 = zp154(Z2, P2, in_r_2);

  //assignment corresponds to 0.666us delay
  // -666 -> 0, -333 -> 333, 0 -> 666
  for (unsigned long i = 0; i < p.rate2; i++)
  {
    out_r_150.write(y150(i), i);
    out_r_151.write(y151(i), i);
    out_r_152.write(y152(i), i);
    out_r_153.write(y153(i), i);
    out_r_154.write(y154(i), i);
  }

  // overall delay 0.666us

  ////////////////////////////////////////////////////////////////////

  // port out (rate=3) / port in (rate=5)
  out_r_160 = zp160(Z, P, in_r_0);
  out_r_161 = zp161(Z, P, in_r_1);
  out_r_162 = zp162(Z, P, in_r_2);
  out_r_163 = zp163(Z, P, in_r_3);
  out_r_164 = zp164(Z2, P2, in_r_2);
}

bool check_val(double val, double e_val, double rel_err, double abs_err)
{
  if ( (std::fabs(val - e_val) > abs_err) && (std::fabs(val - e_val) / e_val > rel_err))
  {
    std::cout << " Expect value: " << e_val << " read value: " << val << std::endl;
    return true;
  }

  return false;
}

int check_values(double time, double delay, double cval, double rampv, double t2_val, double sin_val,
                 double t3_val, double rel_err = 1e-10, double abs_err = 1e-10)
{
  double ctime = time-delay;
  double e_cval = 0.0;
  double e_rampv = 0.0;
  double e_t2_val = 0.0;
  double e_sin_val = 0.0;
  double e_t3_val = 0.0;

  if (ctime >= 0.0)
  {
    e_cval = 1.0;
    e_rampv = ctime;
    e_t2_val = 0.5 * ctime * ctime;
    e_sin_val = std::sin(2.0 * M_PI * 13.3e3 * ctime);
    e_t3_val = 0.5 / 3.0 * ctime * ctime * ctime;
  }
  else
  {
    return 0;
  }

  if (check_val(cval, e_cval, rel_err, abs_err))
  {
    return 1;
  }

  if (check_val(rampv, e_rampv, rel_err, abs_err))
  {
    return 2;
  }

  if (check_val(t2_val, e_t2_val, rel_err, abs_err))
  {
    return 3;
  }

  if (check_val(t3_val, e_t3_val, rel_err, abs_err))
  {
    return 5;
  }

  return 0;
}


bool check_file(std::string fname, const std::vector<double>& delays)
{
  std::ifstream fin;
  fin.open(fname.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  std::string line_in;
  unsigned long line_cnt = 0;
  if (!std::getline(fin, line_in))
  {
    std::ostringstream str;
    str << "Can't get header line from file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }
  line_cnt++;

  std::vector<std::string> names;
  std::istringstream nistr(line_in);

  std::string name;
  if (!(nistr >> name) && (name != "%time"))
  {
    std::ostringstream str;
    str << "Wrong header line in file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  while (nistr >> name)
  {
    names.push_back(name);
  }

  while (std::getline(fin, line_in))
  {
    std::istringstream istr(line_in);
    line_cnt++;

    double ctime;
    istr >> ctime;

    if (istr.fail())
    {
      std::ostringstream str;
      str << "Can't get time in file: " << fname << " line: " << line_cnt;
      SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
    }

    for (unsigned long i = 0; i < names.size(); i += 5)
    {
      double cval, rval, t2val, sinval, t3val;

      istr >> cval;
      istr >> rval;
      istr >> t2val;
      istr >> sinval;
      istr >> t3val;

      if (istr.fail())
      {
        std::ostringstream str;
        str << "Failed to read values from file: " << fname << " line: " << line_cnt;
        str << " for: " << names[i] << " " << names[i+1] << " "
            << names[i+2] << " " << names[i+3]<< " " << names[i+4];
        str << " at time: " << ctime;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }

      int err = check_values(ctime, delays[i / 5], cval, rval, t2val, sinval, t3val);
      if (err != 0)
      {
        std::ostringstream str;
        str << "Wrong value in file: " << fname << " line: " << line_cnt;
        str << " for signal: " << names[i+err-1] << " at time " << ctime;
        str << " delay: " << delays[i / 5];
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }
    }
  }

  return false;
}

int sc_main(int argn, char* argc[])
{
  sc_core::sc_set_time_resolution(1.0, sc_core::SC_FS);
  
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> s_out_const1, s_out_ramp1, s_out_t21, s_out_sin1, s_out_t31;
  sca_tdf::sca_signal<double> s_out_constr, s_out_rampr, s_out_t2r, s_out_sinr, s_out_t3r;

  sca_tdf::sca_signal<double> s_out_const_1, s_out_ramp_1, s_out_t2_1, s_out_sin_1, s_out_t3_1;
  sca_tdf::sca_signal<double> s_out_const_3, s_out_ramp_3, s_out_t2_3, s_out_sin_3, s_out_t3_3;
  sca_tdf::sca_signal<double> s_out_const_5, s_out_ramp_5, s_out_t2_5, s_out_sin_5, s_out_t3_5;
  sca_tdf::sca_signal<double> s_out_const_7, s_out_ramp_7, s_out_t2_7, s_out_sin_7, s_out_t3_7;

  sca_tdf::sca_signal<double> s_out_10, s_out_r_0;
  sca_tdf::sca_signal<double> s_out_11, s_out_r_1;
  sca_tdf::sca_signal<double> s_out_12, s_out_r_2;
  sca_tdf::sca_signal<double> s_out_13, s_out_r_3;
  sca_tdf::sca_signal<double> s_out_14, s_out_r_4;

  sca_tdf::sca_signal<double> s_out_20, s_out_40, s_out_60, s_out_80;
  sca_tdf::sca_signal<double> s_out_21, s_out_41, s_out_61, s_out_81;
  sca_tdf::sca_signal<double> s_out_22, s_out_42, s_out_62, s_out_82;
  sca_tdf::sca_signal<double> s_out_23, s_out_43, s_out_63, s_out_83;
  sca_tdf::sca_signal<double> s_out_24, s_out_44, s_out_64, s_out_84;

  sca_tdf::sca_signal<double> s_out_r90, s_out_r100, s_out_r110, s_out_r120;
  sca_tdf::sca_signal<double> s_out_r91, s_out_r101, s_out_r111, s_out_r121;
  sca_tdf::sca_signal<double> s_out_r92, s_out_r102, s_out_r112, s_out_r122;
  sca_tdf::sca_signal<double> s_out_r93, s_out_r103, s_out_r113, s_out_r123;
  sca_tdf::sca_signal<double> s_out_r94, s_out_r104, s_out_r114, s_out_r124;

  sca_tdf::sca_signal<double> s_out_r130, s_out_r140, s_out_r150, s_out_r160;
  sca_tdf::sca_signal<double> s_out_r131, s_out_r141, s_out_r151, s_out_r161;
  sca_tdf::sca_signal<double> s_out_r132, s_out_r142, s_out_r152, s_out_r162;
  sca_tdf::sca_signal<double> s_out_r133, s_out_r143, s_out_r153, s_out_r163;
  sca_tdf::sca_signal<double> s_out_r134, s_out_r144, s_out_r154, s_out_r164;

  // instantiate and connect components

  ltf_zp_src* i_src = new ltf_zp_src("i_src");
  i_src->outp_const_1(s_out_const1);
  i_src->outp_ramp_1(s_out_ramp1);
  i_src->outp_t2_1(s_out_t21);
  i_src->outp_sin_1(s_out_sin1);
  i_src->outp_t3_1(s_out_t31);

  i_src->outp_const_r(s_out_constr);
  i_src->outp_ramp_r(s_out_rampr);
  i_src->outp_t2_r(s_out_t2r);
  i_src->outp_sin_r(s_out_sinr);
  i_src->outp_t3_r(s_out_t3r);

  i_src->outp0(s_out_10);
  i_src->outp1(s_out_11);
  i_src->outp2(s_out_12);
  i_src->outp3(s_out_13);
  i_src->outp4(s_out_14);

  i_src->outpr0(s_out_r_0);
  i_src->outpr1(s_out_r_1);
  i_src->outpr2(s_out_r_2);
  i_src->outpr3(s_out_r_3);
  i_src->outpr4(s_out_r_4);

  ltf_zp_test* i_zp_test = new ltf_zp_test("i_zp_test");
  i_zp_test->in_const_1(s_out_const1);
  i_zp_test->in_ramp_1(s_out_ramp1);
  i_zp_test->in_t2_1(s_out_t21);
  i_zp_test->in_sin_1(s_out_sin1);

  i_zp_test->in_const_r(s_out_constr);
  i_zp_test->in_ramp_r(s_out_rampr);
  i_zp_test->in_t2_r(s_out_t2r);
  i_zp_test->in_sin_r(s_out_sinr);

  i_zp_test->in_10(s_out_10);
  i_zp_test->in_11(s_out_11);
  i_zp_test->in_12(s_out_12);
  i_zp_test->in_13(s_out_13);

  i_zp_test->in_r_0(s_out_r_0);
  i_zp_test->in_r_1(s_out_r_1);
  i_zp_test->in_r_2(s_out_r_2);
  i_zp_test->in_r_3(s_out_r_3);

  i_zp_test->out_const_1(s_out_const_1);
  i_zp_test->out_ramp_1(s_out_ramp_1);
  i_zp_test->out_t2_1(s_out_t2_1);
  i_zp_test->out_sin_1(s_out_sin_1);
  i_zp_test->out_t3_1(s_out_t3_1);

  i_zp_test->out_20(s_out_20);
  i_zp_test->out_21(s_out_21);
  i_zp_test->out_22(s_out_22);
  i_zp_test->out_23(s_out_23);
  i_zp_test->out_24(s_out_24);

  i_zp_test->out_const_3(s_out_const_3);
  i_zp_test->out_ramp_3(s_out_ramp_3);
  i_zp_test->out_t2_3(s_out_t2_3);
  i_zp_test->out_sin_3(s_out_sin_3);
  i_zp_test->out_t3_3(s_out_t3_3);

  i_zp_test->out_40(s_out_40);
  i_zp_test->out_41(s_out_41);
  i_zp_test->out_42(s_out_42);
  i_zp_test->out_43(s_out_43);
  i_zp_test->out_44(s_out_44);

  i_zp_test->out_const_5(s_out_const_5);
  i_zp_test->out_ramp_5(s_out_ramp_5);
  i_zp_test->out_t2_5(s_out_t2_5);
  i_zp_test->out_sin_5(s_out_sin_5);
  i_zp_test->out_t3_5(s_out_t3_5);

  i_zp_test->out_60(s_out_60);
  i_zp_test->out_61(s_out_61);
  i_zp_test->out_62(s_out_62);
  i_zp_test->out_63(s_out_63);
  i_zp_test->out_64(s_out_64);

  i_zp_test->out_const_7(s_out_const_7);
  i_zp_test->out_ramp_7(s_out_ramp_7);
  i_zp_test->out_t2_7(s_out_t2_7);
  i_zp_test->out_sin_7(s_out_sin_7);
  i_zp_test->out_t3_7(s_out_t3_7);

  i_zp_test->out_80(s_out_80);
  i_zp_test->out_81(s_out_81);
  i_zp_test->out_82(s_out_82);
  i_zp_test->out_83(s_out_83);
  i_zp_test->out_84(s_out_84);

  i_zp_test->out_r_90(s_out_r90);
  i_zp_test->out_r_91(s_out_r91);
  i_zp_test->out_r_92(s_out_r92);
  i_zp_test->out_r_93(s_out_r93);
  i_zp_test->out_r_94(s_out_r94);

  i_zp_test->out_r_100(s_out_r100);
  i_zp_test->out_r_101(s_out_r101);
  i_zp_test->out_r_102(s_out_r102);
  i_zp_test->out_r_103(s_out_r103);
  i_zp_test->out_r_104(s_out_r104);

  i_zp_test->out_r_110(s_out_r110);
  i_zp_test->out_r_111(s_out_r111);
  i_zp_test->out_r_112(s_out_r112);
  i_zp_test->out_r_113(s_out_r113);
  i_zp_test->out_r_114(s_out_r114);

  i_zp_test->out_r_120(s_out_r120);
  i_zp_test->out_r_121(s_out_r121);
  i_zp_test->out_r_122(s_out_r122);
  i_zp_test->out_r_123(s_out_r123);
  i_zp_test->out_r_124(s_out_r124);

  i_zp_test->out_r_130(s_out_r130);
  i_zp_test->out_r_131(s_out_r131);
  i_zp_test->out_r_132(s_out_r132);
  i_zp_test->out_r_133(s_out_r133);
  i_zp_test->out_r_134(s_out_r134);

  i_zp_test->out_r_140(s_out_r140);
  i_zp_test->out_r_141(s_out_r141);
  i_zp_test->out_r_142(s_out_r142);
  i_zp_test->out_r_143(s_out_r143);
  i_zp_test->out_r_144(s_out_r144);

  i_zp_test->out_r_150(s_out_r150);
  i_zp_test->out_r_151(s_out_r151);
  i_zp_test->out_r_152(s_out_r152);
  i_zp_test->out_r_153(s_out_r153);
  i_zp_test->out_r_154(s_out_r154);

  i_zp_test->out_r_160(s_out_r160);
  i_zp_test->out_r_161(s_out_r161);
  i_zp_test->out_r_162(s_out_r162);
  i_zp_test->out_r_163(s_out_r163);
  i_zp_test->out_r_164(s_out_r164);

  /////// Tracing -> for each rate a different file

  sca_util::sca_trace_file* tf1 = sca_util::sca_create_tabular_trace_file("tdf_zp_1_1.dat");

  sca_util::sca_trace(tf1, s_out_const1, "s_out_const1");
  sca_util::sca_trace(tf1, s_out_ramp1, "s_out_ramp1");
  sca_util::sca_trace(tf1, s_out_t21, "s_out_t21");
  sca_util::sca_trace(tf1, s_out_sin1, "s_out_sin1");
  sca_util::sca_trace(tf1, s_out_t31, "s_out_t31");

  sca_util::sca_trace(tf1, s_out_10, "s_out_10");
  sca_util::sca_trace(tf1, s_out_11, "s_out_11");
  sca_util::sca_trace(tf1, s_out_12, "s_out_12");
  sca_util::sca_trace(tf1, s_out_13, "s_out_13");
  sca_util::sca_trace(tf1, s_out_14, "s_out_14");

  sca_util::sca_trace(tf1, s_out_const_1, "s_out_const_1");
  sca_util::sca_trace(tf1, s_out_ramp_1, "s_out_ramp_1");
  sca_util::sca_trace(tf1, s_out_t2_1, "s_out_t2_1");
  sca_util::sca_trace(tf1, s_out_sin_1, "s_out_sin_1");
  sca_util::sca_trace(tf1, s_out_t3_1, "s_out_t3_1");

  sca_util::sca_trace(tf1, s_out_const_3, "s_out_const_3");
  sca_util::sca_trace(tf1, s_out_ramp_3, "s_out_ramp_3");
  sca_util::sca_trace(tf1, s_out_t2_3, "s_out_t2_3");
  sca_util::sca_trace(tf1, s_out_sin_3, "s_out_sin_3");
  sca_util::sca_trace(tf1, s_out_t3_3, "s_out_t3_3");

  sca_util::sca_trace(tf1, s_out_const_5, "s_out_const_5");
  sca_util::sca_trace(tf1, s_out_ramp_5, "s_out_ramp_5");
  sca_util::sca_trace(tf1, s_out_t2_5, "s_out_t2_5");
  sca_util::sca_trace(tf1, s_out_sin_5, "s_out_sin_5");
  sca_util::sca_trace(tf1, s_out_t3_5, "s_out_t3_5");

  sca_util::sca_trace(tf1, s_out_const_7, "s_out_const_7");
  sca_util::sca_trace(tf1, s_out_ramp_7, "s_out_ramp_7");
  sca_util::sca_trace(tf1, s_out_t2_7, "s_out_t2_7");
  sca_util::sca_trace(tf1, s_out_sin_7, "s_out_sin_7");
  sca_util::sca_trace(tf1, s_out_t3_7, "s_out_t3_7");

  sca_util::sca_trace(tf1, s_out_20, "s_out_20");
  sca_util::sca_trace(tf1, s_out_21, "s_out_21");
  sca_util::sca_trace(tf1, s_out_22, "s_out_22");
  sca_util::sca_trace(tf1, s_out_23, "s_out_23");
  sca_util::sca_trace(tf1, s_out_24, "s_out_24");

  sca_util::sca_trace(tf1, s_out_40, "s_out_40");
  sca_util::sca_trace(tf1, s_out_41, "s_out_41");
  sca_util::sca_trace(tf1, s_out_42, "s_out_42");
  sca_util::sca_trace(tf1, s_out_43, "s_out_43");
  sca_util::sca_trace(tf1, s_out_44, "s_out_44");

  sca_util::sca_trace(tf1, s_out_60, "s_out_60");
  sca_util::sca_trace(tf1, s_out_61, "s_out_61");
  sca_util::sca_trace(tf1, s_out_62, "s_out_62");
  sca_util::sca_trace(tf1, s_out_63, "s_out_63");
  sca_util::sca_trace(tf1, s_out_64, "s_out_64");

  sca_util::sca_trace(tf1, s_out_80, "s_out_80");
  sca_util::sca_trace(tf1, s_out_81, "s_out_81");
  sca_util::sca_trace(tf1, s_out_82, "s_out_82");
  sca_util::sca_trace(tf1, s_out_83, "s_out_83");
  sca_util::sca_trace(tf1, s_out_84, "s_out_84");

  sca_util::sca_trace_file* tfr = sca_util::sca_create_tabular_trace_file("tdf_zp_1_rate.dat");

  sca_util::sca_trace(tfr, s_out_constr, "s_out_constr");
  sca_util::sca_trace(tfr, s_out_rampr, "s_out_rampr");
  sca_util::sca_trace(tfr, s_out_t2r, "s_out_t2r");
  sca_util::sca_trace(tfr, s_out_sinr, "s_out_sinr");
  sca_util::sca_trace(tfr, s_out_t3r, "s_out_t3r");

  sca_util::sca_trace(tfr, s_out_r_0, "s_out_r_0");
  sca_util::sca_trace(tfr, s_out_r_1, "s_out_r_1");
  sca_util::sca_trace(tfr, s_out_r_2, "s_out_r_2");
  sca_util::sca_trace(tfr, s_out_r_3, "s_out_r_3");
  sca_util::sca_trace(tfr, s_out_r_4, "s_out_r_4");

  sca_util::sca_trace(tfr, s_out_r90, "s_out_r90");
  sca_util::sca_trace(tfr, s_out_r91, "s_out_r91");
  sca_util::sca_trace(tfr, s_out_r92, "s_out_r92");
  sca_util::sca_trace(tfr, s_out_r93, "s_out_r93");
  sca_util::sca_trace(tfr, s_out_r94, "s_out_r94");

  sca_util::sca_trace(tfr, s_out_r100, "s_out_r100");
  sca_util::sca_trace(tfr, s_out_r101, "s_out_r101");
  sca_util::sca_trace(tfr, s_out_r102, "s_out_r102");
  sca_util::sca_trace(tfr, s_out_r103, "s_out_r103");
  sca_util::sca_trace(tfr, s_out_r104, "s_out_r104");

  sca_util::sca_trace(tfr, s_out_r110, "s_out_r110");
  sca_util::sca_trace(tfr, s_out_r111, "s_out_r111");
  sca_util::sca_trace(tfr, s_out_r112, "s_out_r112");
  sca_util::sca_trace(tfr, s_out_r113, "s_out_r113");
  sca_util::sca_trace(tfr, s_out_r114, "s_out_r114");

  sca_util::sca_trace(tfr, s_out_r120, "s_out_r120");
  sca_util::sca_trace(tfr, s_out_r121, "s_out_r121");
  sca_util::sca_trace(tfr, s_out_r122, "s_out_r122");
  sca_util::sca_trace(tfr, s_out_r123, "s_out_r123");
  sca_util::sca_trace(tfr, s_out_r124, "s_out_r124");

  sca_util::sca_trace_file* tfr2 = sca_util::sca_create_tabular_trace_file("tdf_zp_1_rate2.dat");

  sca_util::sca_trace(tfr2, s_out_r130, "s_out_r130");
  sca_util::sca_trace(tfr2, s_out_r131, "s_out_r131");
  sca_util::sca_trace(tfr2, s_out_r132, "s_out_r132");
  sca_util::sca_trace(tfr2, s_out_r133, "s_out_r133");
  sca_util::sca_trace(tfr2, s_out_r134, "s_out_r134");

  sca_util::sca_trace(tfr2, s_out_r140, "s_out_r140");
  sca_util::sca_trace(tfr2, s_out_r141, "s_out_r141");
  sca_util::sca_trace(tfr2, s_out_r142, "s_out_r142");
  sca_util::sca_trace(tfr2, s_out_r143, "s_out_r143");
  sca_util::sca_trace(tfr2, s_out_r144, "s_out_r144");

  sca_util::sca_trace(tfr2, s_out_r150, "s_out_r150");
  sca_util::sca_trace(tfr2, s_out_r151, "s_out_r151");
  sca_util::sca_trace(tfr2, s_out_r152, "s_out_r152");
  sca_util::sca_trace(tfr2, s_out_r153, "s_out_r153");
  sca_util::sca_trace(tfr2, s_out_r154, "s_out_r154");

  sca_util::sca_trace(tfr2, s_out_r160, "s_out_r160");
  sca_util::sca_trace(tfr2, s_out_r161, "s_out_r161");
  sca_util::sca_trace(tfr2, s_out_r162, "s_out_r162");
  sca_util::sca_trace(tfr2, s_out_r163, "s_out_r163");
  sca_util::sca_trace(tfr2, s_out_r164, "s_out_r164");

  ////////////////////////////////////////////////

  sc_core::sc_start(2.0, sc_core::SC_MS); //start time domain simulation for 5ms

  sca_util::sca_close_tabular_trace_file(tf1);
  sca_util::sca_close_tabular_trace_file(tfr);
  sca_util::sca_close_tabular_trace_file(tfr2);

  std::vector<double> delays;
  delays.resize(20, 0.0);

  // delay is not shown, due we propagate the first value to zero -> thus at time
  // zero the constant is valid -> the ramp starts at zero (begin of integration with 1)
  check_file("tdf_zp_1_1.dat", delays);

  ///////////////////////////////////////////////////////////////////////////

  // all single rate in -> multi rate out have a delay in zp_test
  delays[2] = 0.8e-6;  // out_r_90  - out_r_94
  delays[3] = 0.8e-6;  // out_r_100 - out_r_104
  delays[4] = 0.8e-6;  // out_r_110 - out_r_114
  delays[5] = 0.8e-6;  // out_r_120 - out_r_124

  check_file("tdf_zp_1_rate.dat", delays);

  ///////////////////////////////////////////////////////////////////////

  delays.resize(0);
  delays.resize(16, 0.0);

  delays[0] = 2e-6 / 3.0;  //s_out_r130 - s_out_r134
  delays[2] = 2e-6 / 3.0;  //s_out_r150 - s_out_r154

  check_file("tdf_zp_1_rate2.dat", delays);

  if (sc_core::sc_report_handler::get_count(sc_core::SC_ERROR) <= 0)
  {
    SC_REPORT_INFO("ZP_CHECK", "Test was successful");
  }

  TEST_LABEL_END;

  delete i_src, i_zp_test;

  return 0;
}
