/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  circuit16.cpp  - eln regression test

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(circuit16)
{  
  // port declaration
  sca_tdf::sca_in<double> inp1;

  // node declaration
  sca_eln::sca_node n1;
  sca_eln::sca_node n2;
  sca_eln::sca_node_ref n3; // reference node
  sca_eln::sca_node n4;
  sca_eln::sca_node n5;
  sca_eln::sca_node_ref n6;
  sca_eln::sca_node n7;

  // component declaration
  sca_eln::sca_tdf_vsource *vin;
  sca_eln::sca_gyrator     *gyr;
  sca_eln::sca_r           *r1, *r2, *r3, *r4;

  struct params // parameter
  {
    double r1_val;
    double r2_val;
    double r3_val;
    double r4_val;

    params() // default for parameter
    {
      r1_val = 1.0e3;
      r2_val = 4.0e3;
      r3_val = 2.0e3;
      r4_val = 3.0e3;
    }
  };

  circuit16( sc_core::sc_module_name nm, params pa = params() ) 
  : inp1("inp1"), n1("n1"), n2("n2"), n3("n3"), n4("n4"), n5("n5"),
    n6("n6"), n7("n7"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~circuit16();

private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void circuit16::architecture()
{
  r1 = new sca_eln::sca_r("r1");
  r1->value = p.r1_val;
  r1->n(n1);
  r1->p(n7);

  r2 = new sca_eln::sca_r("r2");
  r2->value = p.r2_val;
  r2->n(n3);
  r2->p(n2);

  r3 = new sca_eln::sca_r("r3");
  r3->value = p.r3_val;
  r3->n(n4);
  r3->p(n3);

  r4 = new sca_eln::sca_r("r4");
  r4->value = p.r4_val;
  r4->n(n6);
  r4->p(n5);

  gyr = new sca_eln::sca_gyrator("gyr");
  gyr->n2(n4);
  gyr->p2(n2);
  gyr->p1(n1);
  gyr->n1(n5);
  gyr->g1 = 2.0;
  gyr->g2 = 3.0;

  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp1);
  vin->p(n7);
  vin->n(n6);
}

circuit16::~circuit16()
{
  delete r1, r2, r3, r4, gyr, vin;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..10" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double node7, node1, node2, node4, node5, ir1, ir2;
    str >> node7 >> node1 >> node2 >> node4 >> node5 >> ir1 >> ir2;

    if ( !test_util::check_value(2.0 * (node1 - node5), ir2, 1e-6) )
    {
      std::cerr.precision(15);
      std::cerr << "Error TD-check failed for verification of g1 at: " << time
                << " res: " << 2.0 * (node1 - node5) << " expect: " << ir2
                << " err: " << fabs(2.0 * (node1 - node5) - ir2) << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !test_util::check_value( ir1, (node2 - node4) * 3) )
    {
      std::cerr << "Error TD-check failed for verification of g2 at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;
  };

  if (test_passed_with_success) std::cerr  << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;

  test_util::const_src::params src_params;
  src_params.ampl = 10;

  test_util::const_src* src = new test_util::const_src("source", src_params);
  src->outp(signal1);

  circuit16* cir = new circuit16("circuit");
  cir->inp1(signal1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output16.dat");
  sca_util::sca_trace(tf, cir->n7, "node7");
  sca_util::sca_trace(tf, cir->n1, "node1");
  sca_util::sca_trace(tf, cir->n2, "node2");
  sca_util::sca_trace(tf, cir->n4, "node4");
  sca_util::sca_trace(tf, cir->n5, "node5");
  sca_util::sca_trace(tf, cir->r1, "ir1");
  sca_util::sca_trace(tf, cir->r2, "ir2");

  sc_core::sc_start(50, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output16.dat");

  TEST_LABEL_END;

  delete src, cir;
  
  return 0;
}
