/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc-ams>
#include "test_utilities.h"
#include <string>

SC_MODULE(test01)
{
  // port declaration
  sc_core::sc_in<double> inp1;

  // node declaration
  sca_eln::sca_node n_in;
  sca_eln::sca_node n_rr;
  sca_eln::sca_node_ref n_gnd;

  // component declaration
  sca_eln::sca_vsource   *src1;
  sca_eln::sca_de::sca_c *var_c1;
  sca_eln::sca_r         *r1;

  struct params // parameter
  {
    params() // default for parameter
    {
    }
  };

  test01( sc_core::sc_module_name nm, params pa = params() ) 
  : n_in("n_in"), n_rr("n_rr"), n_gnd("n_gnd"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~test01();
 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void test01::architecture()
{
  src1 = new sca_eln::sca_vsource("src1");
  src1->amplitude = 5.0;
  src1->frequency = 1.0e4;
  src1->offset = 0;
  src1->p(n_in);
  src1->n(n_gnd);

  r1 = new sca_eln::sca_r("r1");
  r1->p(n_in);
  r1->n(n_rr);
  r1->value = 1e3;

  var_c1 = new  sca_eln::sca_de::sca_c("var_c1");
  var_c1->p(n_rr);
  var_c1->n(n_gnd);
  var_c1->inp(inp1);
  var_c1->scale = 1.0e-6;
  var_c1->set_timestep(0.25, sc_core::SC_US);
}

test01::~test01()
{
  delete src1, r1, var_c1;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sc_core::sc_signal<double> signal1;

  test_util::sin_src_de::params src_params2;
  src_params2.ampl = 5;
  src_params2.freq = 500;
  src_params2.offset = 6;
  src_params2.sample_time = sc_core::sc_time(1, sc_core::SC_US);

  test_util::sin_src_de* src2 = new test_util::sin_src_de("source2", src_params2);
  src2->outp(signal1);

  test01* cir = new test01("circuit");
  cir->inp1(signal1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test03");
  sca_util::sca_trace(tf, cir->n_in, "n_in");
  sca_util::sca_trace(tf, cir->n_rr, "n_rr");

  sc_core::sc_start(8, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  test_util::check_results("test03", 2, "", 1e-10, 1e-6);

  TEST_LABEL_END;

  return 0;
}
