/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc-ams>
#include "test_utilities.h"


SC_MODULE(circuit18)
{  
  // port declaration
  sca_tdf::sca_in<double> inp1;

  // node declaration
  sca_eln::sca_node n1;
  sca_eln::sca_node n2;
  sca_eln::sca_node_ref n3; // reference node

  // component declaration
  sca_eln::sca_tdf_vsource *u0;
  sca_eln::sca_c           *c1;
  sca_eln::sca_r           *r1;

  struct params // parameter
  {
    double r_val;

    params() // default for parameter
    {
      r_val = 2e3;
    }
  };

  circuit18( sc_core::sc_module_name nm, params pa = params() ) : inp1("inp1"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~circuit18();

 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void circuit18::architecture()
{
  r1 = new sca_eln::sca_r("r1");
  r1->value = p.r_val;
  r1->n(n2);
  r1->p(n1);

  c1 = new sca_eln::sca_c("c1");
  c1->p(n2);
  c1->n(n3);
  c1->value = (1.0 / (2.0 * M_PI * p.r_val * 100.0));

  u0 = new sca_eln::sca_tdf_vsource("u0");
  u0->inp(inp1);
  u0->p(n1);
  u0->n(n3);
}

circuit18::~circuit18()
{
  delete r1, c1, u0;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;

  test_util::sin_src::params src_params;
  src_params.ampl = 10;
  src_params.freq = 100;
  src_params.sample_time = sc_core::sc_time(50, sc_core::SC_US);

  test_util::sin_src* src = new test_util::sin_src("source", src_params);
  src->outp(signal1);

  circuit18* cir = new circuit18("circuit");
  cir->inp1(signal1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test03.dat");
  sca_util::sca_trace(tf, cir-> n1, "node1");
  sca_util::sca_trace(tf, cir-> n2, "node2");
  sca_util::sca_trace(tf, cir-> n3, "node3");
  sca_util::sca_trace(tf, cir-> c1, "ic1");
  sca_util::sca_trace(tf, cir-> r1, "ir1");

  sc_core::sc_start(35, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  test_util::check_results("test03", 1e-12);

  TEST_LABEL_END;

  delete src, cir;
  
  return 0;
}
