/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example filter
 
#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> out; // output port

  sin_src( sc_core::sc_module_name nm, double ampl_= 1.0, double freq_ = 1.0e3,
           sca_core::sca_time Tm_ = sca_core::sca_time(0.125, sc_core::SC_MS) )
  : out("out"), ampl(ampl_), freq(freq_), Tm(Tm_)
  {}

  void set_attributes()
  {
    out.set_timestep(Tm);
  }

  void processing()
  {
    double t = get_time().to_seconds(); // actual time
    out.write(ampl * std::sin( 2.0 * M_PI * freq * t ) ) ;
  }

  void ac_processing()
  {
    sca_ac_analysis::sca_ac(out) = ampl;
  }

 private:
  double ampl; // amplitude
  double freq; // frequency
  sca_core::sca_time Tm; // module time step
};

SCA_TDF_MODULE(ltf_filter_ac)
{
  sca_tdf::sca_in<double>  in;
  sca_tdf::sca_out<double> out;

  ltf_filter_ac( sc_core::sc_module_name nm, double fc_, double h0_ = 1.0 )
  : in("in"), out("out"), fc(fc_), h0(h0_) {}

  void initialize()
  {
    num(0) = 1.0;
    den(0) = 1.0;
    den(1) = 1.0 / ( 2.0 * M_PI * fc );
  }

  void processing()    // time-domain implementation
  {
    out.write( ltf_nd( num, den, in.read(), h0 ) );
  }

  void ac_processing() // frequency-domain implementation
  {
    sca_ac_analysis::sca_ac(out) = sca_ac_analysis::sca_ac_ltf_nd( 
                                     num, den, sca_ac_analysis::sca_ac(in), h0 );
  }

 private:
  sca_tdf::sca_ltf_nd ltf_nd;            // Laplace transfer function
  sca_util::sca_vector<double> num, den; // numerator and denominator coefficients
  double fc; // 3dB cutoff frequency in Hz
  double h0; // DC gain
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;
  
  sin_src src("src");
    src.out(sig1);

  ltf_filter_ac lpf("lpf", 3.3e6);
    lpf.in(sig1);
    lpf.out(sig2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("lp_filter_td");
  
  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");
    
  sc_core::sc_start(10, sc_core::SC_MS);

  tf->reopen("lp_filter_ac");
   
  tf->set_mode(sca_util::sca_ac_format(sca_util::SCA_AC_MAG_RAD));
  
  sca_ac_analysis::sca_ac_start(1.0e3, 100.0e4, 4, sca_ac_analysis::SCA_LOG);

  sca_util::sca_close_tabular_trace_file(tf);

  std::cout << "Check time domain results..." << std::endl;
  test_util::check_results("lp_filter_td", 2, "ltf_filter_ac");

  std::cout << "Check frequency domain results..." << std::endl;
  test_util::check_results("lp_filter_ac", 4, "ltf_filter_ac");

  TEST_LABEL_END;
  
  return 0;
}

