/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  reinit_segfault.cpp -- Test case for segfault in
//      sca_synchronization_alg::sca_sync_objT::reinit() provoked by the
//      presence of TDF modules with more than two TDF output ports in clusters
//      that change rates or delays.

#include <iostream>
#include <systemc-ams>
#include "test_utilities.h"

// Counter module with three outputs.
class counter : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_out<double> out1;
  sca_tdf::sca_out<double> out2;
  sca_tdf::sca_out<double> out3;

  counter ( sc_core::sc_module_name nm,
            sca_core::sca_time Tm, double init_val,
            unsigned long Ro1, unsigned long Do1,
            unsigned long Ro2, unsigned long Do2,
            unsigned long Ro3, unsigned long Do3,
            sc_core::sc_time Tchange, unsigned long new_rate
            )
  : sca_tdf::sca_module(nm), out1("out1"), out2("out2"), out3("out3"),
    Tm_(Tm), init_val_(init_val), Ro1_(Ro1), Do1_(Do1), Ro2_(Ro2), Do2_(Do2),
    Ro3_(Ro3), Do3_(Do3), Tchange_(Tchange), new_rate_(new_rate)
  {}

  void set_attributes()
  {
    this->set_timestep(Tm_);
    out1.set_rate(Ro1_);
    out1.set_delay(Do1_);
    out2.set_rate(Ro2_);
    out2.set_delay(Do2_);
    out3.set_rate(Ro3_);
    out3.set_delay(Do3_);
    accept_attribute_changes();
    does_attribute_changes();
  }

  void initialize()
  {
    // init count
    count_ = init_val_;
    // init output 1 delays
    for (unsigned long i = 0; i < out1.get_delay(); i++)
    {
      int val = -1-(int)i;
      out1.initialize(val, i);
    }
    // init output 2 delays
    for (unsigned long i = 0; i < out2.get_delay(); i++)
    {
      int val = -1-(int)i;
      out2.initialize(val, i);
    }
    // init output 3 delays
    for (unsigned long i = 0; i < out3.get_delay(); i++)
    {
      int val = -1-(int)i;
      out3.initialize(val, i);
    }
  }

  void processing()
  {
    double val = count_++;
    // output 1 source value
    for (unsigned long i = 0; i < out1.get_rate(); i++)
    {
      out1.write(val, i);
    }
    // output 2 source value
    for (unsigned long i = 0; i < out2.get_rate(); i++)
    {
      out2.write(val, i);
    }
    // output 3 source value
    for (unsigned long i = 0; i < out3.get_rate(); i++)
    {
      out3.write(val, i);
    }
  }

  void change_attributes()
  {
    sc_core::sc_time curr_Tsim = sc_core::sc_time_stamp(); // curr Tsim
    if (curr_Tsim > Tchange_)
    {
      out1.set_rate(new_rate_);
      out2.set_rate(new_rate_);
      out3.set_rate(new_rate_);
    }
  }

 private:
  sca_core::sca_time Tm_;        // module time step
  double             init_val_;  // init counter value
  unsigned long      Ro1_;       // output 1 port rate
  unsigned long      Do1_;       // output 1 port delay
  unsigned long      Ro2_;       // output 2 port rate
  unsigned long      Do2_;       // output 2 port delay
  unsigned long      Ro3_;       // output 3 port rate
  unsigned long      Do3_;       // output 3 port delay
  sc_core::sc_time   Tchange_;   // time to change attributes
  unsigned long      new_rate_;  // new rate
  double             count_;     // variable to generate the output
};  // class counter


// Sink module
class sink : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_in<double> in;

  sink(sc_core::sc_module_name nm, unsigned long Ri, unsigned long Di)
  : sca_tdf::sca_module(nm), in("in"), Ri_(Ri), Di_(Di)
  {}

  void set_attributes()
  {
    in.set_rate(Ri_);
    in.set_delay(Di_);
    accept_attribute_changes();
  }

  void initialize()
  {
    // init input delay
    for (unsigned long i = 0; i < in.get_delay(); i++)
    {
      int val = (int)i - in.get_delay();
      in.initialize(val, i);
    }
  }

  void processing ()
  {
    for (unsigned long i = 0; i < in.get_rate(); i++)
    {
      in.read(i);
    }
  }

 private:
  unsigned long Ri_; // input rate
  unsigned long Di_; // input delay
};  // class sink


int sc_main(int argc, char* argv[])
{
  using sca_core::sca_time;

  // Time resolution
  sc_core::sc_set_time_resolution(1.0, sc_core::SC_NS);

  // Simulation stop time
  sc_core::sc_time tstop(20.0, sc_core::SC_US);

  // Signals
  sca_tdf::sca_signal<double> sig_src1;
  sca_tdf::sca_signal<double> sig_src2;
  sca_tdf::sca_signal<double> sig_src3;

  // Parameters
  sca_core::sca_time Tm(1.0, sc_core::SC_US);
  double init_src = 0;

  unsigned long srcRo1 = 1;
  unsigned long srcDo1 = 0;
  unsigned long srcRo2 = 1;
  unsigned long srcDo2 = 0;
  unsigned long srcRo3 = 1;
  unsigned long srcDo3 = 0;

  sc_core::sc_time Tchange(10.0, sc_core::SC_US);
  unsigned long newRo1 = 2;

  unsigned long sinkRi1 = 1;
  unsigned long sinkDi1 = 0;
  unsigned long sinkRi2 = 1;
  unsigned long sinkDi2 = 0;
  unsigned long sinkRi3 = 1;
  unsigned long sinkDi3 = 0;

  // Module instantiation
  counter counter_1("counter_1", Tm, init_src, srcRo1, srcDo1, srcRo2, srcDo2,
                    srcRo3, srcDo3, Tchange, newRo1);
  counter_1.out1(sig_src1);
  counter_1.out2(sig_src2);
  counter_1.out3(sig_src3);

  sink sink_1("sink_1", sinkRi1, sinkDi1);
  sink_1.in(sig_src1);

  sink sink_2("sink_2", sinkRi2, sinkDi2);
  sink_2.in(sig_src2);

  sink sink_3("sink_3", sinkRi3, sinkDi3);
  sink_3.in(sig_src3);

  // Tracing
  sca_util::sca_trace_file* tf;
  tf =  sca_util::sca_create_tabular_trace_file("tr_test_change_rate_src1");
  sca_util::sca_trace(tf, sig_src1, "sig_src1");

  sca_util::sca_trace_file* tf1;
  tf1 = sca_util::sca_create_tabular_trace_file("tr_test_change_rate_src2");
  sca_util::sca_trace(tf1, sig_src2, "sig_src2");

  sca_util::sca_trace_file* tf2;
  tf2 = sca_util::sca_create_tabular_trace_file("tr_test_change_rate_src3");
  sca_util::sca_trace(tf2, sig_src3, "sig_src3");

  // Simulation
  try
  {
    sc_core::sc_start(tstop);
  }
  catch (const std::exception& e)
  {
    std::cerr << e.what() << std::endl;
  }

  sca_util::sca_close_tabular_trace_file(tf);
  sca_util::sca_close_tabular_trace_file(tf1);
  sca_util::sca_close_tabular_trace_file(tf2);
  sc_core::sc_stop();

  TEST_LABEL_START;

  test_util::check_results("tr_test_change_rate_src1", 1, "reinit_segfault");

  test_util::check_results("tr_test_change_rate_src2", 1, "reinit_segfault");

  test_util::check_results("tr_test_change_rate_src3", 1, "reinit_segfault");

  TEST_LABEL_END;

  return sc_core::sc_report_handler::get_count(sc_core::SC_ERROR);
}

// vim: expandtab : tabstop=2 : softtabstop=2 : shiftwidth=2
