/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//----------------------------------------------------------------------
//  dtdf_multirate_tstep_change_bug.cpp --
//
//  SystemC-AMS 2.0.5_beta2 (rev. 1767) has a bug leading to corrupted
//  signal waveforms when the time step is changed in a multirate
//  cluster. This is demonstrated by the following test case:
//
//  +-----+     +-----+     +-----+
//  | src *--*->* ltf *---->* amp *---->
//  +-----+  |  +-----+     +-----+
//           |
//           |  +-----+
//           +->* fwd *---->
//              +-----+
//
//  src produces a constant sine signal of 1 kHz with amplitude 1. Its
//  initial time step is set to 50 us. At 20 ms, src sets its time step
//  to 100 us.
//
//  ltf applies a filter transfer function (fc = 500 Hz).
//
//  amp estimates the amplitude by averaging until 50 ms over 20
//  samples and then over 200 samples by setting its input rate from 20
//  to 200. At 80 ms, amp additionally sets the input time step to 10 us.
//
//  When examining, the traced signals, one can observe that the src
//  output signal gets corrupted once the time step is changed: The
//  signal isn't anymore sinusoidal and also the frequency changes
//  astonishingly. There seems to be some issue with the time stamp
//  calculation. This seems to get corrected by a change in the rates of
//  the cluster. After a new time step change, again signal corruption
//  can be observed. Note that the problem doesn't seem to be limited to
//  tracing as the forwarded signal by the block fwd gets also
//  corrupted.
//----------------------------------------------------------------------

#include <cmath>
#include <iostream>
#include <systemc-ams>
#include "test_utilities.h"

class tdf_source : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_out<double> out;

  tdf_source(sc_core::sc_module_name nm) : out("out"), tstep_changed_(false) {}

 protected:
  void set_attributes()
  {
    this->accept_attribute_changes();
    this->does_attribute_changes();
    this->set_timestep(50.0, sc_core::SC_US);
  }

  void processing()
  {
    out.write(std::sin(2.0 * M_PI * 1.0e3 * this->get_time().to_seconds()));
  }

  void change_attributes()
  {
    if (!tstep_changed_ && this->get_time().to_seconds() > 20.0e-3)
    {
      this->set_timestep(100.0, sc_core::SC_US);
      std::cout << this->name() << " @ " << this->get_time() << ": this->set_timestep(100.0 us)" << std::endl;
      tstep_changed_ = true;
    }
  }

 private:
  bool tstep_changed_;
};


class tdf_ltf_filter : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  tdf_ltf_filter(sc_core::sc_module_name nm) : in("in"), out("out") {
    // 1st-order filter with fc = 500 Hz
    num_(0) = 1.0;
    den_(0) = 1.0;
    den_(1) = 1.0 / (2.0 * M_PI * 500.0);
  }

 protected:
  void set_attributes()
  {
    this->accept_attribute_changes();
  }

  void processing()
  {
    out.write(ltf_(num_, den_, in.read()));
  }

 private:
  sca_util::sca_vector<double> num_, den_;
  sca_tdf::sca_ltf_nd ltf_;
};


class tdf_forward : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  tdf_forward(sc_core::sc_module_name nm) : in("in"), out("out") {}

 protected:
  void set_attributes()
  {
    this->accept_attribute_changes();
  }

  void processing()
  {
    out.write(in.read());
  }
};

class dtdf_amp_estimator : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  dtdf_amp_estimator(sc_core::sc_module_name nm) : in("in"), out("out"), tstep_changed_(false) {}

 protected:
  void set_attributes()
  {
    this->accept_attribute_changes();
    this->does_attribute_changes();
    in.set_rate(20);
  }

  void processing()
  {
    const unsigned long n = in.get_rate();
    double abs_sum = 0.0;
    for (unsigned long i = 0; i < n; ++i)
    {
      abs_sum += std::fabs(in.read(i));
    }
    double amp = abs_sum / n;
    out.write(amp);
  }

  void change_attributes()
  {
    if (this->get_time().to_seconds() > 50.0e-3 && in.get_rate() != 200)
    {
      in.set_rate(200);
      std::cout << this->name() << " @ " << this->get_time() << ": in.set_rate(200)" << std::endl;
    }
    if (!tstep_changed_ && this->get_time().to_seconds() > 80.0e-3)
    {
      in.set_timestep(10.0, sc_core::SC_US);
      std::cout << this->name() << " @ " << this->get_time() << ": in.set_timestep(10 us)" << std::endl;
      tstep_changed_ = true;
    }
  }

 private:
  bool tstep_changed_;
};


int sc_main(int argc, char* argv[])
{
  sca_tdf::sca_signal<double> sig_src("sig_src"), sig_ltf("sig_ltf"),
                              sig_fwd("sig_fwd"), sig_amp("sig_amp");

  tdf_source src("src");
  src.out(sig_src);

  tdf_ltf_filter ltf("ltf");
  ltf.in(sig_src);
  ltf.out(sig_ltf);

  tdf_forward fwd("fwd");
  fwd.in(sig_src);
  fwd.out(sig_fwd);

  dtdf_amp_estimator amp("amp");
  amp.in(sig_ltf);
  amp.out(sig_amp);

  sca_util::sca_trace_file* tfp_tab = sca_util::sca_create_tabular_trace_file("dtdf_multirate_tstep_change_bug");
  sca_util::sca_trace(tfp_tab, sig_src, sig_src.name());
  sca_util::sca_trace(tfp_tab, sig_fwd, sig_fwd.name());
  sca_util::sca_trace(tfp_tab, sig_ltf, sig_ltf.name());
  sca_util::sca_trace(tfp_tab, sig_amp, sig_amp.name());

  sc_core::sc_start(100.0, sc_core::SC_MS);

  sca_close_tabular_trace_file(tfp_tab);

  TEST_LABEL_START;

  test_util::check_results("dtdf_multirate_tstep_change_bug", 4, 1e-12);

  TEST_LABEL_END;

  return 0;
}
