/* Copyright
 * ========================================================================================
 * Project:		Accurate DRAM Model
 * Author:		Nan Li, KTH
 * ID:			test_gen.cpp, v1.2, 2010/12/04
 *
 * Description:	Test generator for accurate DRAM model
 *
 * ========================================================================================
 */

#include "test_gen.h"
#include <map>

void adm_test_gen::test() {
	while (true) {
		char ch = finTest.get();
		if (ch != '.') {
			/* flush the line */
			while (ch != '\n')
				ch = finTest.get();
			continue;
		}

		ch = finTest.get();
		if (ch == 'e') {
			/* ".e" indicated end of test file */
			break;
		} else if (ch == 'r') {
			/* ".r start_cycle address thread_id length" is read command */
			uint64 start;
			uint64 addr;
			int threadId;
			unsigned int len;

			finTest >> start >> hex >> addr >> dec >> threadId >> len;
			do ch = finTest.get(); while (ch != '\n');	/* flush the end of the line */

			while (cycle < start) wait();
			burstRead(addr, threadId, len, true);
		} else if (ch == 'w') {
			/* ".w start_cycle address thread_id length" is write command */
			uint64 start;
			uint64 addr;
			int threadId;
			unsigned int len;

			finTest >> start >> hex >> addr >> dec >> threadId >> len;
			do ch = finTest.get(); while (ch != '\n');	/* flush the end of the line */

			while (cycle < start) wait();
			burstWrite(addr, threadId, len, false, true);
		}
	}

	while (txnIdMap.size()) wait();

	/* write the summary into log file */

	foutLog << "ID,Type,Address,Length,Start,End,Latency,ThreadID\n";

	for (size_t i = 0; i < txnSummary.size(); ++i) {
		foutLog << txnSummary[i].id << ",";
		foutLog << (txnSummary[i].write ? "write" : "read") << ",";
		foutLog.setf(ios::showbase);
		foutLog << hex << txnSummary[i].address << dec << ",";
		foutLog << txnSummary[i].length << ",";
		foutLog << txnSummary[i].start << ",";
		foutLog << txnSummary[i].end << ",";
		foutLog << (txnSummary[i].end - txnSummary[i].start) << ",";
		foutLog << txnSummary[i].thread_id;
		foutLog << endl;
	}

//	double delay_sum = 0.0;
//	for (size_t i = 0; i < txnSummary.size(); ++i) {
//		delay_sum += (txnSummary[i].end - txnSummary[i].start);
//	}
//	foutLog << (delay_sum / txnSummary.size()) << endl;

	cout << "\nLog file generated at cycle " << cycle << "\n\n";

	sc_stop();
}

void adm_test_gen::handleResponse() {
	if (!rsp) return;

	static std::map<tlm::tlm_generic_payload *, unsigned int> burstLengthMap, burstCountMap;

	if (!burstLengthMap[rsp]) {
		/* first response */
		burstLengthMap[rsp] = calculateBurstLength(*rsp);
		burstCountMap[rsp] = 0;
	}

	if (rsp->get_command() == tlm::TLM_WRITE_COMMAND) {
		if (ocpInitPort.get_extension<ocpip::srmd>(*rsp)) {
			// cout << "[RSP @ " << sc_time_stamp() << "] [WRITE] BURST(" << burstLengthMap[rsp] << ") - SRMD : response received." << endl;
			ocpInitPort.release_transaction(rsp);
			burstLengthMap.erase(rsp);
			burstCountMap.erase(rsp);

			txnSummary[txnIdMap[rsp]].end = cycle;
			// cout << "[TXN " << txnSummary[txnIdMap[rsp]].id << "] [WRITE] [END @ " << cycle << "]" << endl;
			txnIdMap.erase(rsp);
		} else {
			// cout << "[RSP @ " << sc_time_stamp() << "] [WRITE] BURST(" << burstLengthMap[rsp] << ") : DATA[" << burstCountMap[rsp] << "] response received." << endl;
			if (burstCountMap[rsp] == burstLengthMap[rsp] - 1) {
				// cout << "[FIN @ " << sc_time_stamp() << "] [WRITE] BURST(" << burstLengthMap[rsp] << ") : finished." << endl;
				burstLengthMap.erase(rsp);
				burstCountMap.erase(rsp);
				ocpInitPort.release_transaction(rsp);
			}
		}

	} else {
		// Td data = *((Td *)rsp->get_data_ptr() + burstCountMap[rsp]);
		// cout << "[RSP @ " << sc_time_stamp() << "] [READ] BURST(" << burstLengthMap[rsp] << ") : DATA[" << burstCountMap[rsp] << "] = " << data << endl;

		if (burstCountMap[rsp] == burstLengthMap[rsp] - 1) {
			// cout << "[FIN @ " << sc_time_stamp() << "] [READ] BURST(" << burstLengthMap[rsp] << ") : finished." << endl;
			burstLengthMap.erase(rsp);
			burstCountMap.erase(rsp);
			ocpInitPort.release_transaction(rsp);

			txnSummary[txnIdMap[rsp]].end = cycle;
			// cout << "[TXN " << txnSummary[txnIdMap[rsp]].id << "] [READ] [END @ " << cycle << "]" << endl;
			txnIdMap.erase(rsp);
		}
	}

	if (burstCountMap.count(rsp))
		burstCountMap[rsp]++;
	rsp = 0;
}

void adm_test_gen::countCycle() {
	while (true) {
		wait(SC_ZERO_TIME);
		cycle++;

		wait();
	}
}

tlm::tlm_sync_enum adm_test_gen::nb_transport_bw(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) {
	if (ph == tlm::BEGIN_RESP) {
		rsp = &txn;
		ph = tlm::END_RESP;
		return tlm::TLM_UPDATED;
	} else if ((ph == tlm::END_REQ) || (ph == ocpip::END_DATA)) {
		req = 0;
	} else {
		sc_assert(false);
	}

	return tlm::TLM_ACCEPTED;
}

void adm_test_gen::burstWrite(uint64 address, unsigned int threadId, unsigned int length, bool posted, bool srmd) {
	tlm::tlm_generic_payload *txn = ocpInitPort.get_transaction();
	ocpInitPort.reserve_data_size(*txn, sizeof(Td) * length);
	txn -> set_address(address);
	txn -> set_byte_enable_ptr(NULL);
	txn -> set_streaming_width(sizeof(Td));
	txn -> set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);
	txn -> set_command(tlm::TLM_WRITE_COMMAND);

	/* set extension for posted */
	if (posted) ocpInitPort.validate_extension<ocpip::posted>(*txn);

	/* set extension for srmd */
	if (srmd) ocpInitPort.validate_extension<ocpip::srmd>(*txn);

	/* set extension burst_length */
	ocpip::burst_length *bLen;
	ocpInitPort.get_extension(bLen, *txn);
	bLen->value = length;
	ocpInitPort.validate_extension<ocpip::burst_length>(*txn);

	/* set extension thread_id */
	ocpip::thread_id *tId;
	ocpInitPort.get_extension(tId, *txn);
	tId->value = threadId;
	ocpInitPort.validate_extension<ocpip::thread_id>(*txn);

	if (srmd) {
		tlm::tlm_phase phase = tlm::BEGIN_REQ;
		sc_time time = SC_ZERO_TIME;
		req = txn;

		// cout << "[REQ @ " << sc_time_stamp() << "] [WRITE] BURST(" << length << ") - SRMD : request sent." << endl;
		ocpInitPort->nb_transport_fw(*req, phase, time);

		while (req) wait();

		/* keep track of transactions for log file */
		txn_info info;
		info.id = txnId++;
		info.write = true;
		info.address = address;
		info.length = length;
		info.start = cycle;
		info.thread_id = threadId;

		txnSummary.push_back(info);
		txnIdMap[txn] = info.id;
		// cout << "[TXN " << info.id << "] [WRITE] [START @ " << cycle << "] [ADDR 0x" << hex << address << dec << "] [THREAD " << threadId << "] [LEN " << length << "]" << endl;



		// cout << "[REQ @ " << sc_time_stamp() << "] [WRITE] BURST(" << length << ") - SRMD : request accepted." << endl;
	}

	/* send data */
	for (unsigned i = 0; i < length; ++i) {
		*((Td *)txn->get_data_ptr() + i) = 1234;	/* data does not matter */

		tlm::tlm_phase phase;
		if (srmd)
			phase = ocpip::BEGIN_DATA;
		else
			phase = tlm::BEGIN_REQ;
		sc_time time = SC_ZERO_TIME;

		req = txn;
		// cout << "[REQ @ " << sc_time_stamp() << "] [WRITE] BURST(" << length << ")";
		// if (srmd) cout << " - SRMD";
		// cout << " : DATA[" << i << "] = " << *((Td *)txn->get_data_ptr() + i) << endl;

		if (ocpInitPort->nb_transport_fw(*req, phase, time) == tlm::TLM_UPDATED) {
			req = 0;
			wait();
		} else {
			while (req) wait();
		}

		// cout << "[REQ @ " << sc_time_stamp() << "] [WRITE] BURST(" << length << ")";
		// if (srmd) cout << " - SRMD";
		// cout << " : data accepted." << endl;
	}

	if (posted)
		ocpInitPort.release_transaction(txn);
}

void adm_test_gen::burstRead(uint64 address, unsigned int threadId, unsigned int length, bool srmd) {
	tlm::tlm_generic_payload *txn = ocpInitPort.get_transaction();
	ocpInitPort.reserve_data_size(*txn, length * sizeof(Td));
	txn -> set_address(address);
	txn -> set_byte_enable_ptr(0);
	txn -> set_streaming_width(sizeof(Td));
	txn -> set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);
	txn -> set_command(tlm::TLM_READ_COMMAND);

	/* set extension for srmd */
	if (srmd) ocpInitPort.validate_extension<ocpip::srmd>(*txn);

	/* set extension burst_length */
	ocpip::burst_length *bLen;
	ocpInitPort.get_extension(bLen, *txn);
	bLen->value = length;
	ocpInitPort.validate_extension<ocpip::burst_length>(*txn);

	/* set extension thread_id */
	ocpip::thread_id *tId;
	ocpInitPort.get_extension(tId, *txn);
	tId->value = threadId;
	ocpInitPort.validate_extension<ocpip::thread_id>(*txn);

	if (srmd) {
		tlm::tlm_phase phase = tlm::BEGIN_REQ;
		sc_time time = SC_ZERO_TIME;
		req = txn;

		// cout << "[REQ @ " << sc_time_stamp() << "] [READ] BURST(" << length << ") - SRMD : request sent." << endl;
		ocpInitPort->nb_transport_fw(*req, phase, time);

		while (req) wait();

		/* keep track of transactions for log file */
		txn_info info;
		info.id = txnId++;
		info.write = false;
		info.address = address;
		info.length = length;
		info.start = cycle;
		info.thread_id = threadId;

		txnSummary.push_back(info);
		txnIdMap[txn] = info.id;
		// cout << "[TXN " << info.id << "] [READ] [START @ " << cycle << "] [ADDR 0x" << hex << address << dec << "] [THREAD " << threadId << "] [LEN " << length << "]" << endl;



		// cout << "[REQ @ " << sc_time_stamp() << "] [READ] BURST(" << length << ") - SRMD : request accepted." << endl;
	} else {
		for (unsigned i = 0; i < length; ++i) {
			tlm::tlm_phase phase = tlm::BEGIN_REQ;
			sc_time time = SC_ZERO_TIME;
			req = txn;

			// cout << "[REQ @ " << sc_time_stamp() << "] [READ] BURST(" << length << ") : DATA[" << i << "]" << endl;
			ocpInitPort->nb_transport_fw(*req, phase, time);

			while (req) wait();

			// cout << "[REQ @ " << sc_time_stamp() << "] [READ] BURST(" << length << ") : request accepted." << endl;
		}
	}
}

unsigned int adm_test_gen::calculateBurstLength(tlm::tlm_generic_payload &txn) {
	ocpip::burst_length *bLen;
	if (ocpip::extension_api::get_extension<ocpip::burst_length>(bLen, txn))
		return bLen->value;
	else {
		SC_REPORT_WARNING(SC_ID_WITHOUT_MESSAGE_, "burst_length extension is not used. Calculating burst length from data length.");
		return txn.get_data_length() / sizeof(Td);
	}
}
