/**
 *
 * @file task.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: task.cc 1916 2011-07-06 12:44:26Z lehton87 $
 *
 */

#include "task.hh"

#include <algorithm>
#include <exception>
#include <iomanip>

namespace sctg
{
   Task::Task(const boost::property_tree::ptree& pt,
	      Configuration& config)
      : 
      ResourceUser(pt, config),      
      _localBytesSent(0),
      _remoteBytesSent(0),
      _lastLatency(sc_core::SC_ZERO_TIME),
      _lastDelay(sc_core::SC_ZERO_TIME),
      _lastResponseTime(sc_core::SC_ZERO_TIME),
      _timesTriggered(0)
   {
      using boost::property_tree::ptree;
      
    
      // All tasks start in WAIT state
      _state = WAIT;
    
      
    
      // Read in all triggers
      for(ptree::const_iterator iter = pt.begin(); iter != pt.end(); ++iter)
      {
	 if((*iter).first == "trigger")
	 {
	    Trigger* trigger = new Trigger((*iter).second, config, this);
	    _triggers.push_back(trigger);
	 }
      }
    
      // Confirm that there are triggers for this task
      if(_triggers.size() == 0)
      {
	 std::string err = "Task " + _name + " has no triggers";
	 throw std::runtime_error(err.c_str());
      }
      _currentTrigger = 0;
   }  

   Task::~Task()
   {
      unsigned long int timesTriggered = 0;
      unsigned long int bytesSent = 0;
      unsigned long int cyclesExecuted = 0;
      unsigned long int bytesRead = 0;

      // Clean triggers
      for(unsigned int i = 0; i < _triggers.size(); ++i)
      {
	 timesTriggered += _triggers.at(i)->getTimesTriggered();
	 bytesSent      += _triggers.at(i)->getBytesSent();
	 cyclesExecuted += _triggers.at(i)->getCyclesExecuted();
	 bytesRead      +=  _triggers.at(i)->getBytesRead();
	 delete _triggers.at(i);
	 _triggers.at(i) = 0;
      }
      
      if(_config.getSummaryStream())
      {
	 **_config.getSummaryStream()
	    << "- Task " << std::setw(20) << std::left << getName() 
	    << " was triggered "  << std::setw(5) << std::right 
	    << timesTriggered << " times, sent " << std::setw(8)
	    << std::right << bytesSent << " bytes, read " <<
	    std::setw(8) << std::right << bytesRead << " and executed "
	    << std::setw(10) << std::right << cyclesExecuted
	    << " clock cycles during simulation" << std::endl;
      }
   }
   

   bool Task::hasInPort(unsigned long int id)
   {
      if(std::find(_inPorts.begin(), _inPorts.end(), id) != _inPorts.end())
      {	return true; }
      else
      { return false; }
   }

   bool Task::hasOutPort(unsigned long int id)
   {
      if(std::find(_outPorts.begin(), _outPorts.end(), id) != _outPorts.end())
      {	return true; }
      else
      { return false; }
   }

   OperationType Task::getOperation()
   {    
      return _triggers.at(_currentTrigger)->getOperation();
   }
  
   bool Task::isNewOperation()
   {
      return _triggers.at(_currentTrigger)->isNewOperation();
   }

   unsigned long int Task::getAmount()
   {
      return _triggers.at(_currentTrigger)->getAmount();
   }

   unsigned long int Task::getBurstAmount()
   {
      return _triggers.at(_currentTrigger)->getBurstAmount();
   }

   unsigned long int Task::getReadAmount()
   {
      return _triggers.at(_currentTrigger)->getReadAmount();
   }
  
   unsigned long int Task::getOutPort()
   {
      return _triggers.at(_currentTrigger)->getOutPort();
   }

   unsigned long int Task::getRespPort()
   {
      return _triggers.at(_currentTrigger)->getRespPort();
   }

   unsigned long int Task::consume(unsigned long int amount)
   {
      return _triggers.at(_currentTrigger)->consume(amount);
   }

   State Task::getState()
   {
      return _state;
   }

   void Task::receive(tgToken token)
   {
      // Check that packet was for this task
      if(!hasInPort(token.destination))
      {
	 std::string err = "Task " + getName() + 
	    " got token to a wrong destination";
	 throw std::runtime_error(err.c_str());
      }
      // Update receive time
      token.timeReceived = sc_core::sc_time_stamp();
      sc_core::sc_time lat = token.timeReceived - token.timeSent;
      _config.addTokenLatency(token.source, token.destination, lat);

      // Add packet to all triggers that has this port
      for(unsigned int i = 0; i < _triggers.size(); ++i)
      {
	 if(_triggers.at(i)->hasInPort(token.destination))
	 {
	    _triggers.at(i)->receive(token);
	 }
      }
      // Task's status might change
      if(_state == WAIT)
      {
	 // Task state is READY if any of triggers is active
	 for(unsigned int i = 0; i < _triggers.size(); ++i)
	 {
	    if(_triggers.at(i)->isActive())
	    {
	       changeState(READY);
	    }
	 }
      }
      else if(_state == FREE) // FREE task cannot receive tokens
      {
	 std::string err = "Task " + getName() + 
	    " was already FREE but received a token";
	 throw std::runtime_error(err.c_str());
      }
   }

   void Task::changeState(State newState)
   {
      // std::cout << "Task " << getName() << " changing state from "
      // 	      << stateToString(_state) << " to " 
      //      << stateToString(newState) << std::endl;

      if(_state == FREE)
      {
	 std::string err = "Task " + getName() + 
	    " was already FREE but tried to change its state";
	 throw std::runtime_error(err.c_str());
      }
      if(newState == FREE)
      {
	 _state = FREE;
      }
      else if(newState == WAIT || newState == READY)
      {
	 _state = WAIT;
	 // Task state is READY if any of its triggers is active
	 if(_triggers.at(_currentTrigger)->isActive())
	 {
	    _state = READY;
	 }
	 else
	 {
	    for(unsigned int i = _currentTrigger; 
		i < _triggers.size() + _currentTrigger; ++i)
	    {
	       if(_triggers.at(i % _triggers.size())->isActive())
	       {
		  _currentTrigger = i % _triggers.size();
		  _state = READY;
		  break;
	       }
	    }
	 }
      }
      else if(newState == RUN)
      {
	 _state = RUN;
	 if(isNewOperation())
	 {
	    _lastDelay = sc_core::sc_time_stamp() - 
	       _triggers.at(_currentTrigger)->getReceiveTime();
	 }
      }
      //std::cout << " new state is " << stateToString(_state) << std::endl;
   }

   unsigned long int Task::getBufferUsage() const
   {
      unsigned long int retval = 0;
      for(unsigned int i = 0; i < _triggers.size(); ++i)
      {
	 retval += _triggers.at(i)->getBufferUsage();
      }
      return retval;
   }

   unsigned long int Task::getTimesTriggered()
   {
      unsigned long int timesTriggered = 0;            
      for(unsigned int i = 0; i < _triggers.size(); ++i)
      {
	 timesTriggered += _triggers.at(i)->getTimesTriggered();
      }      
      return timesTriggered;
   }

   unsigned long int Task::getLocalBytesSent()
   {    
      return _localBytesSent;
   }

   void Task::addLocalBytesSent(unsigned long int b)
   {
      _localBytesSent += b;
   }

   unsigned long int Task::getRemoteBytesSent()
   {    
      return _remoteBytesSent;
   }

   void Task::addRemoteBytesSent(unsigned long int b)
   {
      _remoteBytesSent += b;
   }

   unsigned long int Task::getCyclesExecuted()
   {      
      unsigned long int cyclesExecuted = 0;
      for(unsigned int i = 0; i < _triggers.size(); ++i)
      {
	 cyclesExecuted += _triggers.at(i)->getCyclesExecuted();
      }    
      return cyclesExecuted;
   }

   const sc_core::sc_time& Task::getLastLatency() const
   {
      return _lastLatency;
   }

   const sc_core::sc_time& Task::getLastDelay() const
   {
      return _lastDelay;
   }

   const sc_core::sc_time& Task::getLastResponse() const
   {
      return _lastResponseTime;
   }

   void Task::setLastResponse(const sc_core::sc_time& resp)
   {
      _lastLatency      = resp - _lastDelay;
      _lastResponseTime = resp;
   }

   void Task::incTriggered(void)
   {
      ++_timesTriggered;
      
      _config.taskTriggered(_id, _timesTriggered);
   }
   

}


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
