/**
 * @file memory_model.hh
 * @author Lasse Lehtonen
 * 
 * @brief Integrates Accurate DRAM model to SCTG2
 *
 * 
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: amount.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef SCTG_MEMORY_MODEL_HH
#define SCTG_MEMORY_MODEL_HH

#include "buffer.hh"
#include "resource.hh"
#include "mem_area.hh"

#include "adm.h"

#include <systemc>

/** Bus width to memory
 */ 
#define MEM_BUSWIDTH 32

namespace sctg
{
   /** Connects Accurate DRAM model to TG
    *
    */
   class MemoryModel : public Resource
   {
   public:
      SC_HAS_PROCESS(MemoryModel);

      /** Constructor
       */
      MemoryModel(sc_core::sc_module_name name,
		  const boost::property_tree::ptree& pt,
		  sctg::Configuration& config);

      /** Destructor
       */
      virtual ~MemoryModel();

      /** Get the memory request size in bytes
       */
      unsigned long int getReqSize() const;

      /** Get measurement data
       */
      const sctg::PeMeasurements& getMeasurements();

   private:            

      /** updates measurements
       */
      void updateMeasurements();

      tlm::tlm_sync_enum nb_transport_bw(tlm::tlm_generic_payload& tx, 
					 tlm::tlm_phase& phase, 
					 sc_core::sc_time& time);

      void thread();

      void burstWrite(uint64 address, unsigned int threadId, 
		      unsigned int length, bool posted = false, 
		      bool srmd = false);
      void burstRead(uint64 address, unsigned int threadId, 
		     unsigned int length, bool srmd = false);

      void handleResponse();

      unsigned int calculateBurstLength(tlm::tlm_generic_payload &txn);

      void mapResourceUser(ResourceUser* task);

      typedef ocpip::ocp_data_class_unsigned<MEM_BUSWIDTH, 32>::DataType Dt;
      ocpip::ocp_master_socket_tl1<sizeof(Dt)*8>* ocpMaster_;
      
      adm_model<MEM_BUSWIDTH>*  dram_;
      sc_core::sc_clock*        clkgen_;
      sc_core::sc_in_clk        clk_;
      tlm::tlm_generic_payload* req;
      tlm::tlm_generic_payload* resp;
      std::map<tlm::tlm_generic_payload *, unsigned int> txnIdMap;
      sctg::Configuration&      config_;
      unsigned long int         id_;
      unsigned long int         packetSize_;
      sc_core::sc_time          cycleLength_;

      std::vector<MemArea*>     memAreas_;
      sc_core::sc_event         readDone;
      
      enum MemState {IDLE, EXECUTING, WRITING, SENDING, READING, CACHE_READ};
      MemState currentState_;

      sc_core::sc_time measureStart_;
      sc_core::sc_time measureEnd_;

      struct Resp
      {
	 unsigned long int dataLeft;
	 unsigned long int respLeft;
	 unsigned long int tokenId;
      };
      std::map<unsigned long int, Resp> respMap;

      unsigned long int reqSize_;

      PeMeasurements measurements_;
   };
}

#endif

// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:


