/**
 *
 * @file configuration.hh
 * @author Lasse Lehtonen
 *
 * ES: Purpose: Central data holder. Stores the application, 
 * ES: mapping, platform structure etc. Others can conveninetly request 
 * ES: information they need, e.g. "which PE executes task foo?"
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: configuration.hh 1947 2011-08-19 08:16:14Z ege $
 *
 */


#ifndef SCTG_CONFIGURATION_HH
#define SCTG_CONFIGURATION_HH

#include "noc_conf_if.hh"
#include "non_sc_conf_if.hh"

#include <boost/property_tree/ptree.hpp>
#include <boost/random.hpp>
#include <boost/optional.hpp>
#include <systemc>
#include <string>
#include <vector>
#include <map>
#include <iostream>
#include <fstream>
#include <queue>

namespace sctg
{
   class ResourceUser;
   class Resource;
   class ProcessingElement;
   class Task;
   class Buffer;
   class BufferInterface;

#ifdef SCTG_USE_EXECMON
   class TcpServerIf;
#endif

   /** Keeps hold of connections between everything and measurements
    */
   class Configuration : public NocConfIf
#ifdef SCTG_USE_EXECMON
		       , public NonScConfIf
#endif
   {
   public :
    
      /** Random number generators for real and integer numbers
       */
      typedef boost::lagged_fibonacci19937 RealEngineType; 
      typedef boost::mt19937               IntEngineType;

      /** Constructor
       */
      Configuration(const boost::property_tree::ptree& pt,
		    std::string& saveDir);

      /** Destructor
       */
      ~Configuration();
    

      /** Time
       */
      const sc_core::sc_time& getSimLength() const;
      const sc_core::sc_time& getSimResolution() const;
      const sc_core::sc_time& getMeasurementInterval() const;


      /** Returns processing element libs filename
       */
      const std::string& getPeLibFile() const;

      /** Random numbers and a unique id
       */
      RealEngineType& getRealRNGEngine();
      IntEngineType& getIntRNGEngine();
      const unsigned long int getSeed() const;
      double random();
      unsigned int getTokenId();

      /** Task connections
       */
      void addPortConnection(unsigned long int source,
			     unsigned long int destination);
      unsigned long int getDestination(unsigned long int source);
      unsigned long int getSource(unsigned long int dest);


      /** Resources and theirs users
       */
      void addResourceMap(unsigned long int id, Resource* pe);
      void addResourceUserMap(unsigned long int id, ResourceUser* task);
      void addResourceUserToResourceMap(unsigned long int id, Resource* pe);
      void addResourceUserToInPortMap(unsigned long int id, ResourceUser* task);

      Resource* getResource(unsigned long int id);
      ResourceUser* getResourceUser(unsigned long int id);
      Resource* getResourceByResourceUser(unsigned long int id);
      ResourceUser* getResourceUserByInPort(unsigned long int id);


      /** Buffers
       */
      void addBufferToResource(Buffer* buffer, unsigned long int pe);
      Buffer* getBufferByResource(unsigned long int pe);
      BufferInterface* getBufferIf(unsigned long int agent);


      /** Grouping and mapping
       */
      void addTaskToGroup(unsigned long int task,
			  unsigned long int group);
      void addGroupToPe(unsigned long int group,
			unsigned long int pe);
      unsigned long int getGroup(unsigned long int task);
      unsigned long int getPeByGroup(unsigned long int group);


      /** Noc
       */
      void setNocClass(std::string& s);
      void setNocType(std::string& s);
      void setNocSubType(std::string& s);
      const std::string& getNocClass();
      const std::string& getNocType();
      const std::string& getNocSubType();
      /** Returns a reference to a property tree rooting to <noc> tag
       */
      const std::map<std::string, int>& getGenerics();



      /** Get resource/group/task maps' begin and end
       */
      std::map<unsigned long int, Resource*>::iterator getResourceBegin();
      std::map<unsigned long int, Resource*>::iterator getResourceEnd();
      std::map<unsigned long int, unsigned long int>::iterator getGroupPeBegin();
      std::map<unsigned long int, unsigned long int>::iterator getGroupPeEnd();
      std::map<unsigned long int, unsigned long int>::iterator getTaskGroupBegin();
      std::map<unsigned long int, unsigned long int>::iterator getTaskGroupEnd();

      /** Set/request whether Execution Monitor is used or not
       */
      void useExecMon(bool use);
      bool useExecMon();

#ifdef SCTG_USE_EXECMON
      /** Sets pointer to TCP server
       */
      void setTcpServer(TcpServerIf* server);

      /** Return pointer to TCP server
       */
      TcpServerIf* getTcpServer();
#endif



      /** Returns ostreams for logging
       */
      boost::optional<std::ostream*>& getPacketStream();
      boost::optional<std::ostream*>& getTokenStream();
      boost::optional<std::ostream*>& getAppStream();
      boost::optional<std::ostream*>& getPeStream();
      boost::optional<std::ostream*>& getMemStream();
      boost::optional<std::ostream*>& getSummaryStream();
      boost::optional<std::ostream*>& getExecMonStream();
      

      /** Return all cost functions and variables
       */
      std::vector<std::string>& getCostFunctions();
      std::map<std::string, double>& getCostVariables();

      /** Tokens
       */
      void addTokenLatency(unsigned long int src,
			   unsigned long int dst,
			   sc_core::sc_time& latency);
      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
      getTokenLatency();
      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
      getTokenLatencyMax();
      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
      getTokenLatencyMin();
      std::map<unsigned long int, std::map<unsigned long int, unsigned long int> >&
      getTokenCount();

      /** Paths
       */
      std::map<std::string, sc_core::sc_time>&  getTotPathLat();
      std::map<std::string, sc_core::sc_time>&  getMaxPathLat();
      std::map<std::string, sc_core::sc_time>&  getMinPathLat();
      std::map<std::string, unsigned long int>& getPathCount();

      /** Sets send and receive time to port
       */
      void addSendTime(unsigned long int port);
      void addReceiveTime(unsigned long int port);


      /** Set how many times task has triggered
       */
      void taskTriggered(unsigned long int task, unsigned long int times);


      
   private:

      /** Creates sc_time_unit from string
       */
      const sc_core::sc_time_unit parseUnit(std::string unit);

      sc_core::sc_time         _simResolution;
      sc_core::sc_time         _simLength;    
      std::string              _peLibFile;
      std::vector<std::string> _costFunctions;
      // RNG engines 
      RealEngineType          _realEngine;
      IntEngineType           _intEngine;
      unsigned long int       _seed;

      unsigned int       _nextTokenId;

      sc_core::sc_time _measurementInterval;

      bool _useExecMon;

      boost::uniform_real<double>  _random;
    
      std::map<unsigned long int, unsigned long int> _portConnections;
      std::map<unsigned long int, unsigned long int> _revPortConnections;
      std::map<unsigned long int, Resource*> _resourceMap;
      std::map<unsigned long int, Resource*> _resourceResourceUserMap;
      std::map<unsigned long int, ResourceUser*> _resourceUserMap;
      std::map<unsigned long int, ResourceUser*> _resourceUserInPortMap;
      std::map<unsigned long int, Buffer*> _resourceBufferMap;
      std::map<unsigned long int, unsigned long int> _taskGroupMap;
      std::map<unsigned long int, unsigned long int> _groupPeMap;

      boost::optional<std::ostream*> _pktOutFile;
      boost::optional<std::ostream*> _tokenOutFile;
      boost::optional<std::ostream*> _peOutFile;
      boost::optional<std::ostream*> _memOutFile;
      boost::optional<std::ostream*> _appOutFile;
      boost::optional<std::ostream*> _summaryOutFile;
      boost::optional<std::ostream*> _execMonOutFile;

      std::string _nocClass;
      std::string _nocType;
      std::string _nocSubType;

      std::map<std::string, int> _generics;

      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >
      _tokenLatency;
      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >
      _tokenLatencyMax;
      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >
      _tokenLatencyMin;

      std::map<unsigned long int, std::map<unsigned long int, unsigned long int> >
      _tokenCount;

      std::multimap<unsigned long int, unsigned long int>             _pathsFwd;
      // Above:       source port,      destination port
      std::multimap<unsigned long int, unsigned long int>             _pathsRev;
      // Above:      destination port,     source port

      // Below: string "src_dst", ...
      std::map<std::string, std::queue<sc_core::sc_time> > _portSendTime;

      std::map<std::string, sc_core::sc_time>  _totPathLat;
      std::map<std::string, sc_core::sc_time>  _maxPathLat;
      std::map<std::string, sc_core::sc_time>  _minPathLat;
      std::map<std::string, unsigned long int> _pathCount;

      std::multimap<unsigned long int, unsigned long int> _taskTriggerTimes;

      std::map<std::string, double> _costVariables;

#ifdef SCTG_USE_EXECMON
      TcpServerIf* _serverIf;
#endif

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
