/**
 *
 * @file buffer.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: buffer.hh 1916 2011-07-06 12:44:26Z lehton87 $
 *
 */


#ifndef SCTG_BUFFER_HH
#define SCTG_BUFFER_HH

#include "configuration.hh"
#include "buffer_if.hh"
#include "tg_packet.hh"
#include "tg_token.hh"
#include "common.hh"

#include <systemc>
#include <queue>
#include <map>

namespace sctg
{
   class Configuration;

   /** Acts as memory model between resources and network
    *
    * Buffer creates transaction queues between resources and their
    * network interfaces for both incoming and outgoing ports
    *
    */
   class Buffer : public BufferInterface
   {
   public:
    
      /** Constructor
       *
       * @param [in] rxBufferSize Size of the receive buffer in bytes
       * @param [in] txBufferSize Size of the transfer buffer in bytes
       * @param [in] config Reference to Configuration object
       */
      Buffer(unsigned long int rxBufferSize,
	     unsigned long int txBufferSize,
	     sctg::Configuration& config);

      /** Destructor
       */
      virtual ~Buffer();

      /** Informs buffer to pass packets to resource instead of tokens
       */
      void usePackets(bool use);

      //
      // Interface from network to PE
      //

      /** Puts packet to buffer
       * 
       * Place received packet to receive queue (from NI to resource)
       *
       * @param [in] packet Packet from network interface
       *
       * Throws exception if there is not enough space for packet
       */
      virtual void rxPutPacket(tgPacket* packet);

      /** Returns amount of bytes left in receive buffer
       */
      virtual unsigned long int rxSpaceLeft();
    
      /** True if buffer has complete token
       */
      bool rxTokenAvailable();

      /** True if buffer has received packets
       */
      bool rxPacketAvailable();

      /** True if buffer has received cache packets
       */
      bool rxCacheAvailable();

      /** Gets packet from buffer
       */
      tgToken rxGetToken();

      /** Gets packet from buffer
       */
      tgPacket* rxGetPacket();

      /** Gets cache packet from buffer
       */
      tgPacket* rxGetCache();

      /** Returns event which notifies when rx buffer has been read by PE
       */
      virtual sc_core::sc_event* rxGetReadEvent();

      /** Returns event which indicates that buffer has a full token available
       */
      sc_core::sc_event* rxGetTokenAvailableEvent();

      /** Returns event which indicates that buffer has a packet available
       */
      sc_core::sc_event* rxGetPacketAvailableEvent();


      /** Returns event which indicates that buffer has a cache packet available
       */
      sc_core::sc_event* rxGetCacheAvailableEvent();

      //
      // Interface from PE to network
      //    

      /** Put packet to buffer
       */
      void txPutPacket(tgPacket* packet);

      /** Get available space in transmit buffer in bytes
       */
      unsigned long int txSpaceLeft();

      /** True if transmit buffer has packet(s) to send through network
       */
      virtual bool txPacketAvailable();

      /** Returns packet to be sent
       */
      virtual tgPacket* txGetPacket();

      /** Returns event that indicates when buffer has packets to send
       */
      virtual sc_core::sc_event* txGetPacketAvailableEvent();

      /** Returns event which tells when network has read from tx buffer
       */
      sc_core::sc_event* txGetReadEvent();

      //
      // Conf interface
      //

      /** Tells buffer to expect a token
       */
      void expectToken(tgToken token);

      /** Remove token from rx size
       */
      void removeToken(unsigned long int size);
    
      /** Returns measurement information
       */
      const BufferMeasurements& getMeasurements();

      /** Add amount of bytes sent inside PE
       */
      void addInternalBytes(unsigned long int bytes);
      
      /** Reserve space from receive buffer
       */
      void rxReserveSpace(unsigned long int bytes);
      /** Free space from receive buffer
       */
      void rxFreeSpace(unsigned long int bytes);
      /** Reserve space from transfer buffer
       */
      void txReserveSpace(unsigned long int bytes);
      /** Free space from transfer buffer
       */
      void txFreeSpace(unsigned long int bytes);

      /** Add latencies for unfinished tokens (from creation to the
       * end of simulation)
       */
      void updateUnfinishedTokensLatency();

   private:

      sc_core::sc_event _rxReadEvent;
      sc_core::sc_event _rxTokenEvent;
      sc_core::sc_event _rxPacketEvent;
      sc_core::sc_event _rxCacheEvent;
      sc_core::sc_event _txReadEvent;
      sc_core::sc_event _txPacketEvent;

      unsigned long int _rxBufferSize;
      unsigned long int _txBufferSize;

      unsigned long int _rxBufferUsed;
      unsigned long int _txBufferUsed;

      std::queue<tgPacket*> _txQueue;
      std::map<unsigned long int, tgToken> _expectedTokens;
      std::map<unsigned long int, unsigned long int> _incomingData;
      std::queue<tgToken> _receivedTokens;
      std::queue<tgPacket*> _receivedPackets;
      std::queue<tgPacket*> _receivedCaches;

      BufferMeasurements _measurements;

      sctg::Configuration& _config;

      bool                 _usePackets;

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
