/**
 *
 * @file addr_lut.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: addr_lut.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_SC_RTL_1_ADDR_LUT_HH
#define ASEBT_SC_RTL_1_ADDR_LUT_HH

#include "addr_lut_pkg.hh"

#include <systemc>

namespace asebt
{
namespace sc_rtl_1
{

   template <int in_addr_w_g  = 32,
	     int out_addr_w_g = 36,
	     int cmp_high_g   = 31,
	     int cmp_low_g    = 0,
	     int net_type_g   = 1,
	     int lut_en_g     = 1
	     >
   class addr_lut : public sc_core::sc_module
   {
   public:
      sc_core::sc_in<sc_dt::sc_lv<in_addr_w_g> >   addr_in;
      sc_core::sc_out<sc_dt::sc_lv<out_addr_w_g> > addr_out;
  
      SC_HAS_PROCESS(addr_lut);
  
      addr_lut(sc_core::sc_module_name name)
	 : sc_core::sc_module(name),
	   addr_in("addr_in"),
	   addr_out("addr_out")
      {

	 int x;
	 int y;
	 int z;

	 for(unsigned int i = 0; i < n_addr_ranges_c; ++i)
	 {
	    x = num_table_c[net_type_g][i].x;
	    y = num_table_c[net_type_g][i].y;
	    z = num_table_c[net_type_g][i].z;
	    switch(net_type_g)
	    {
	       case 1:
		  assert(out_addr_w_c%2 == 0 && "out_addr_w_c mus be even");
		  res_addr_table_c[i] =
		     (sc_dt::sc_uint<out_addr_w_c>(x+y*(2^(out_addr_w_c/2))));
		  break;
	       case 2:
		  res_addr_table_c[i] =
		     (sc_dt::sc_uint<out_addr_w_c>(x+y+z));
		  break;
	       case 3:
		  res_addr_table_c[i] =
		     (sc_dt::sc_uint<out_addr_w_c>(z));
		  break;
	       default:
		  assert(false && "net_type not supported (hibi etc)");
	    }
	 }

	 //
	 in_range_high_c = in_addr_w_g < out_addr_w_g ? 
	    in_addr_w_g : out_addr_w_g;

	 SC_METHOD(method);
	 sensitive << addr_in;
    
	 addr_out.initialize(0);
      }

      virtual ~addr_lut()
      {
      }

      void method()
      {
	 //addr_out.write(0);
    
    
	 if(lut_en_g == 0)
	 {
	    addr_out.write(addr_in.read().range(in_range_high_c, 0));
	 }
	 else
	 {
	    for(unsigned int i = 0; i < n_addr_ranges_c; ++i)
	    {
	       if(addr_in.read().range(cmp_high_g, cmp_low_g) ==
		  addr_table_c[i].in_addr.range(cmp_high_g, cmp_low_g))
	       {
		  addr_out.write(res_addr_table_c[i].range(out_addr_w_g-1,0));
	       }
	    }
	 }
      }
 
  
   private:

      res_addr_array res_addr_table_c;
      unsigned int   in_range_high_c;

  
      number_record get_num(sc_dt::sc_lv<addr_w_c> addr, unsigned int net_type)
      {
	 number_record ret_val = {-1, -1, -1};

	 for(unsigned int i = 0; i < n_addr_ranges_c; ++i)
	 {
	    if((addr & addr_table_c[i].mask) == addr_table_c[i].in_addr)
	    {
	       ret_val = num_table_c[net_type][i];
	    }
	 }
  
	 return ret_val;
      }



   };

}
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
