/**
 *
 * @file mesh_2d_with_pkt_codec_sc_bfm.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: mesh_2d_with_pkt_codec_sc_bfm.hh 1508 2011-01-11 14:23:50Z lehton87 $
 *
 */

#ifndef MESH_2D_SC_2_MESH_2d_WITH_PKT_CODEC_SC_BFM_HH
#define MESH_2D_SC_2_MESH_2d_WITH_PKT_CODEC_SC_BFM_HH

#include "mesh_2d_with_pkt_codec_sc_top.hh"
#include "buffer_if.hh"
#include "tg_packet.hh"
#include "noc_conf_if.hh"

#include <systemc>
#include <vector>
#include <queue>
#include <iostream>
#include <stdexcept>

namespace asebt
{
   namespace mesh_2d_sc_2
   {
         
      template <int n_ag_g, int rows_g, int cols_g, int stfw_en_g, int data_width_g,
		int addr_width_g, int packet_length_g, int tx_len_width_g,
		int timeout_g, int fill_packet_g, int lut_en_g, int net_type_g,
		int len_flit_en_g, int oaddr_flit_en_g, int status_en_g,
		int fifo_depth_g, int mesh_freq_g, int ip_freq_g >
      class mesh_2d_with_pkt_codec_sc_bfm : public sc_core::sc_module
      {
      public:
  
	 SC_HAS_PROCESS(mesh_2d_with_pkt_codec_sc_bfm);

	 //* Constructor
	 mesh_2d_with_pkt_codec_sc_bfm(sc_core::sc_module_name name,
				       sctg::NocConfIf* confIf)
	    : sc_module(name),
	      _confIf(confIf),
	      rst_n("rst_n")	
	 {

	    clk_ip = new sc_core::sc_clock("clk_ip", 
					   sc_core::sc_time(1000000000/ip_freq_g, sc_core::SC_NS));
	    clk_noc = new sc_core::sc_clock("clk_noc", 
					    sc_core::sc_time(1000000000/mesh_freq_g, sc_core::SC_NS));

	    mesh = new mesh_2d_with_pkt_codec_sc_top< 
	       n_ag_g,          // Num of agents
	       stfw_en_g,       // store and forward enable (1 = enabled)
	       data_width_g,    // data width in bits
	       addr_width_g,    // address width in bits
	       packet_length_g, // packet length (words = header+payload)
	       tx_len_width_g,  // width of tx_len line
	       timeout_g,       // timeout, How many cycles to wait for packet completion
	       fill_packet_g,   // fill packet, 1=fill with dummy data, 0=don't
	       lut_en_g,        // lut enable
	       net_type_g,      // net type, 0=mesh, 1=octagon
	       len_flit_en_g,   // len flit enable
	       oaddr_flit_en_g, // Whether to send original address
	       status_en_g,     // status enable
	       fifo_depth_g,    // fifo depth
	       mesh_freq_g,     // mesh frequency
	       ip_freq_g,       // IP frequency
	       rows_g,          // Number of rows in mesh
	       cols_g           // Number of columns in mesh
	       >("Mesh");
   
	    // Bind ports
	    mesh->clk_net(*clk_noc);
	    mesh->clk_ip(*clk_ip);
	    mesh->rst_n(rst_n);
	    for(unsigned int i = 0; i < n_ag_g; ++i)
	    {
	       mesh->rx_av_in[i](rx_av[i]);
	       mesh->rx_data_in[i](rx_data[i]);
	       mesh->rx_we_in[i](rx_we[i]);
	       mesh->rx_txlen_in[i](rx_txlen[i]);
	       mesh->rx_full_out[i](rx_full[i]);
	       mesh->rx_empty_out[i](rx_empty[i]);
	       mesh->tx_av_out[i](tx_av[i]);
	       mesh->tx_data_out[i](tx_data[i]);
	       mesh->tx_re_in[i](tx_re[i]);
	       mesh->tx_empty_out[i](tx_empty[i]);   

	       rx_av[i] = false;
	       rx_data[i] = "0";
	       rx_we[i] = false;
	       rx_txlen[i] = "0";
	       tx_re[i] = false;
	    }

	    // Initial values
	    rst_n.write(false);
      
	    SC_THREAD(thread);
      
	    for(unsigned int i = 0; i < n_ag_g; ++i)
	    {
	       sc_spawn(sc_bind(&mesh_2d_with_pkt_codec_sc_bfm::sender, this, i));
	       sc_spawn(sc_bind(&mesh_2d_with_pkt_codec_sc_bfm::receiver, this, i));
	    }
   
	 }

	 //* Destructor
	 ~mesh_2d_with_pkt_codec_sc_bfm()
	 {
	    delete clk_ip;
	    delete clk_noc;
	    delete mesh;
	 }

      private:

	 void sender(unsigned int agent)
	 {   
	    wait(reset);

	    sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);

	    if(!buffer) 
	    {return;}
      
	    tgPacket* packet = 0;
            
	    while(true)
	    {
	       // Wait for packets to send
	       if(!buffer->txPacketAvailable())
	       {
		  sc_core::wait(*(buffer->txGetPacketAvailableEvent()));
	       }
	       // Read packet from buffer
	       packet = buffer->txGetPacket();
	       sc_core::wait(clk_ip->posedge_event());
	       // Send address - set address valid
	       rx_av[agent] = true;
	       // Put first data (is address) to bus
	       /*std::cout << "Agent " << agent << " sending to "
		 << sc_dt::sc_uint<data_width_g>(packet->address)
		 << std::endl;;*/
	       rx_data[agent] = sc_dt::sc_uint<data_width_g>(packet->address);
	       // Set transfer length	 
	       rx_txlen[agent] = sc_dt::sc_uint<tx_len_width_g>(packet->size/4);
	       // Set write enable
	       rx_we[agent] = true;
	       // Wait until NoC is ready to read
	       do { sc_core::wait(clk_ip->posedge_event()); }
	       while(rx_full[agent].read() == true);
	       // Negate address valid
	       rx_av[agent] = false;
	       // Send data
	       for(unsigned int i = 0; i < packet->size/4; ++i)
	       {
		  // Set data to line
		  if(i == 0)
		  {
		     // Send first byte actually, fill rest with dummy data
		     rx_data[agent]
			= sc_dt::sc_uint<data_width_g>
			(*reinterpret_cast<int*>(packet->data));
		     /*std::cout << "Agent " << agent << " sending data "
		       << sc_dt::sc_uint<data_width_g>
		       (*reinterpret_cast<int*>(packet->data))
		       << std::endl;*/
		  }
		  else
		  {
		     rx_data[agent]
			= sc_dt::sc_uint<data_width_g>(agent);
		     /*std::cout << "Agent " << agent << " sending data "
		       << sc_dt::sc_uint<data_width_g>(agent)
		       << std::endl;*/
		  }

		  // Wait clock cycle or more if NoC can't read it
		  do { sc_core::wait(clk_ip->posedge_event()); }
		  while(rx_full[agent].read() == true);
	       }
	       // Negate write enable
	       rx_we[agent] = false;
	       // Set data bus to zero (not necessary, for easier debugging in sim)
	       rx_data[agent] = sc_dt::sc_uint<data_width_g>(0);
	       // Set tx len lines to zeros (not necessary)
	       rx_txlen[agent] = sc_dt::sc_uint<tx_len_width_g>(0);
	       // Delete packet
	       delete [] packet->data;
	       delete packet; packet = 0;      
	    }
	 }

	 void receiver(unsigned int agent)
	 {
	    wait(reset);

	    sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);
	    tgPacket* packet = 0;

	    if(!buffer) 
	    {return;}

	    while(true)
	    {
	       sc_core::wait(clk_ip->posedge_event());

	       while(rst_n.read() == false)
		  sc_core::wait(clk_ip->posedge_event());

	       if(tx_empty[agent].read() == false)
	       {
		  // Set read enable
		  tx_re[agent] = true;
		  // Check whether it's address or data
		  if(tx_av[agent].read() == true)
		  {
		     /*
		       std::cout << "Agent " << agent << " getting addr "
		       << tx_data[agent].read().to_string() 
		       << " at " << sc_core::sc_time_stamp() << std::endl;
		     */
		     if(packet == 0)
		     {
			packet = new tgPacket;
			packet->address =
			   tx_data[agent].read().to_ulong();
			packet->size = 0;
			packet->data = 0;
		     }
		     else
		     {
			unsigned long int addr = tx_data[agent].read().to_ulong();
			if(addr != packet->address)
			{
			   std::cout << "MESH DBG1" << std::endl;
			   throw std::runtime_error
			      ("mesh_receiver: implement continuous receive!");
			}
		     }

		  }
		  else
		  {
		     /*
		       std::cout << "Agent " << agent << " getting data "
		       << tx_data[agent].read().to_string() 
		       << " at " << sc_core::sc_time_stamp() << std::endl;
		     */
		     if(packet == 0)
		     {
			std::cout << "MESH DBG2" << std::endl;
			throw std::runtime_error
			   ("mesh_receiver: got data without address");
		     }
		     if(packet->size == 0)
		     {
			packet->data = new unsigned char[sizeof(unsigned int)];
			unsigned int data = tx_data[agent].read().to_uint();
			/*std::cout << "Agent " << agent << " getting packet id "
			  << data << std::endl;*/

			*reinterpret_cast<unsigned int*>(packet->data) = data;
	       
		     }
		     packet->size += data_width_g/8;
		  }	 
	       }
	       else
	       {
	 
		  // Set read enable low
		  tx_re[agent] = false;
	 
		  // If we have packet put it to buffer
		  if(packet != 0)
		  {
		     // Wait until packet fits to agent's buffer
		     while(buffer->rxSpaceLeft() < packet->size)
		     {	       
			wait(*(buffer->rxGetReadEvent()));
		     }
		     buffer->rxPutPacket(packet);
		     packet = 0;
		     sc_core::wait(clk_ip->posedge_event());
		  }
	       }
	    }
	 }

	 void thread()
	 {      
	    // Handles reset only
	    sc_core::wait(1, sc_core::SC_NS);
	    rst_n.write(true);
	    reset.notify();
	 }

	 sc_core::sc_event reset;

	 sctg::NocConfIf* _confIf;
 
	 sc_core::sc_clock* clk_ip;
	 sc_core::sc_clock* clk_noc;
	 sc_core::sc_signal<bool> rst_n;
      
	 sc_core::sc_signal<bool>               rx_av[n_ag_g]; 
	 sc_core::sc_signal<sc_dt::sc_bv<data_width_g> >   rx_data[n_ag_g];
	 sc_core::sc_signal<bool>               rx_we[n_ag_g];
	 sc_core::sc_signal<sc_dt::sc_bv<tx_len_width_g> > rx_txlen[n_ag_g];
	 sc_core::sc_signal<bool>               rx_full[n_ag_g];
	 sc_core::sc_signal<bool>               rx_empty[n_ag_g];
	 sc_core::sc_signal<bool>               tx_av[n_ag_g];
	 sc_core::sc_signal<sc_dt::sc_bv<data_width_g> >   tx_data[n_ag_g];
	 sc_core::sc_signal<bool>               tx_re[n_ag_g];
	 sc_core::sc_signal<bool>               tx_empty[n_ag_g];
      
	 mesh_2d_with_pkt_codec_sc_top< 
	    n_ag_g,          // Num of agents
	    stfw_en_g,       // store and forward enable (1 = enabled)
	    data_width_g,    // data width in bits
	    addr_width_g,    // address width in bits
	    packet_length_g, // packet length (words = header+payload)
	    tx_len_width_g,  // width of tx_len line
	    timeout_g,       // timeout, How many cycles to wait for packet completion
	    fill_packet_g,   // fill packet, 1=fill with dummy data, 0=don't
	    lut_en_g,        // lut enable ??
	    net_type_g,      // net type, 0=mesh, 1=octagon
	    len_flit_en_g,   // len flit enable ??
	    oaddr_flit_en_g, // Whether to send original address (means what??)
	    status_en_g,     // status enable ??
	    fifo_depth_g,    // fifo depth ??
	    mesh_freq_g,     // mesh frequency (in Hz??)
	    ip_freq_g,       // IP frequency (in Hz??)
	    rows_g,          // Number of rows in mesh
	    cols_g           // Number of columns in mesh
	    >* mesh;

 
      };

   }
}


#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
