/**
 * File:    ServiceVertexRenderer.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 6.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.service;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Stroke;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import javax.swing.border.Border;

import org.jgraph.JGraph;
import org.jgraph.graph.CellView;
import org.jgraph.graph.GraphConstants;
import org.jgraph.graph.VertexRenderer;
import org.jgraph.graph.VertexView;


/**
 * ServiceVertexRenderer handles drawing of service vertices.
 */
public class ServiceVertexRenderer extends VertexRenderer {
	private Color bgColor;
	private Color fgColor;
	private Color borderColor;
	private int borderWidth;
	private Font font;
	private String label;

	/**
	 * Creates a new ServiceVertexRenderer
	 */
	public ServiceVertexRenderer() {
		super();
	}
	
	
	/* (non-Javadoc)
	 * @see org.jgraph.graph.CellViewRenderer#getRendererComponent(org.jgraph.JGraph, org.jgraph.graph.CellView, boolean, boolean, boolean)
	 */
	public Component getRendererComponent(JGraph graph, CellView cellView, boolean sel, boolean focus, boolean prev) {
		Component comp = super.getRendererComponent(graph, cellView, sel, focus, prev);

		bgColor = GraphConstants.getBackground(cellView.getAllAttributes());
		fgColor = GraphConstants.getForeground(cellView.getAllAttributes());
		borderColor = GraphConstants.getBorderColor(cellView.getAllAttributes());
		font = GraphConstants.getFont(cellView.getAllAttributes());
		Border b = GraphConstants.getBorder(cellView.getAllAttributes());
		Insets insets = b.getBorderInsets(comp);
		borderWidth = insets.left;
		label = cellView.getCell().toString();
		
		return comp;
	}

	
	/* (non-Javadoc)
	 * @see org.jgraph.graph.VertexRenderer#getPerimeterPoint(org.jgraph.graph.VertexView, java.awt.geom.Point2D, java.awt.geom.Point2D)
	 */
	public Point2D getPerimeterPoint(VertexView vView, Point2D src, Point2D dest) {
		Rectangle2D bounds = vView.getBounds();

		double semiX = bounds.getWidth()/2;
		double semiY = bounds.getHeight()/2;
		double centerX = bounds.getCenterX();
		double centerY = bounds.getCenterY();
		
		double biasedDestX = dest.getX()-centerX;
		double biasedDestY = -1*(dest.getY()-centerY); // -1 for changing the direction of y-axis

		double thetaRad = Math.atan2(biasedDestY, biasedDestX);
		
		if (thetaRad < 0) {
			thetaRad = 2*Math.PI+thetaRad;
		}
		
		double biasedEllipseX = semiX*Math.cos(thetaRad);
		double biasedEllipseY = semiY*Math.sin(thetaRad);
		
		double ellipseX = biasedEllipseX+centerX;
		double ellipseY = -1*(biasedEllipseY-centerY);
		
		Point2D p = new Point2D.Double(ellipseX, ellipseY);	
		return p;
	}
	
	
	protected void paintComponent(Graphics g) {
		Graphics2D g2 = (Graphics2D)g.create();

		Dimension size = getSize();

		Insets insets = getInsets();
		int x = insets.left;
		int y = insets.top;
		int width = (int)size.getWidth()-insets.left-insets.right;
		int height = (int)size.getHeight()-insets.top-insets.bottom;
		
		// draw background
		g2.setColor(bgColor);
        g2.fillOval(x, y, width, height);

        // draw border
		g2.setColor(borderColor);
		Stroke stroke = new BasicStroke(borderWidth);
		g2.setStroke(stroke);
        g2.drawOval(x, y, width, height);

        // draw label
		g2.setColor(fgColor);
		g2.setFont(font);
		
		FontMetrics fm = g2.getFontMetrics();
		Rectangle2D bounds = fm.getStringBounds(label, g2);
		int descent = fm.getMaxDescent();
		
		int labelX = (int)(x+width/2-bounds.getWidth()/2);
		int labelY = (int)(y+height/2+bounds.getHeight()/2-descent);
		g2.drawString(label, labelX, labelY);

		g2.dispose();
	}
	
	
	protected void paintBorder(Graphics g) {
		// do nothing
	}
}
