/**
 * File:    ProcessIcon.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 8.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.ctrl;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.Rectangle2D;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import fi.cpu.Settings;
import fi.cpu.data.Process;
import fi.cpu.data.Thread;
import fi.cpu.ui.MainWindow;


/**
 * The ProcessIcon class represents a process in the
 * control view.
 */
public class ProcessIcon extends JComponent implements Comparable<ProcessIcon> {
	public enum State {NORMAL, MOVED, REMOVED}

    public static final int ICON_WIDTH;
    public static final int ICON_HEIGHT;
    public static final int FONT_SIZE;
    public static final Color NORMAL_COLOR;
    public static final Color MOVED_COLOR;
    public static final Color REMOVED_COLOR;
    public static final Color TEXT_COLOR;
    public static final Color MOVE_COLOR;
    public static final boolean SQUARE_SHAPE;
    
    protected ProcessIcon self;
    protected Process process;
    protected Point dragOffset;
    protected State state;

    
    static {
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_DEFAULT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_DEFAULT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_DEFAULT_COLOR_BLUE") );
        NORMAL_COLOR = new Color(red, green, blue);
        
        red = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_MOVED_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_MOVED_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_MOVED_COLOR_BLUE") );
        MOVED_COLOR = new Color(red, green, blue);
        
        red = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_REMOVED_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_REMOVED_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_REMOVED_COLOR_BLUE") );
        REMOVED_COLOR = new Color(red, green, blue);

        red = Integer.parseInt( Settings.getAttributeValue("PROCESS_TEXT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESS_TEXT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESS_TEXT_COLOR_BLUE") );
        TEXT_COLOR  = new Color(red, green, blue);

        red = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_MOVE_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_MOVE_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESS_BACKGROUND_MOVE_COLOR_BLUE") );
        MOVE_COLOR = new Color(red, green, blue);
        
        ICON_WIDTH = Integer.parseInt( Settings.getAttributeValue("PROCESS_WIDTH") );
        ICON_HEIGHT = Integer.parseInt( Settings.getAttributeValue("PROCESS_HEIGHT") );
        FONT_SIZE = Integer.parseInt( Settings.getAttributeValue("PROCESS_TEXT_FONT") );
        SQUARE_SHAPE = Boolean.parseBoolean( Settings.getAttributeValue("PROCESS_SHAPE_SQUARE") );
    }
    
    
    
    /**
     * Creates a new ProcessIcon
     * @param process The process model object.
     */
    public ProcessIcon(Process process) {
    	super();
    	this.self = this;
    	this.process = process;
    	this.dragOffset = new Point(0, 0);
    	
    	state = State.NORMAL;
        
        setSize(ICON_WIDTH, ICON_HEIGHT);
        setOpaque(false);

        initColors();

        //Add listeners
        MyMouseInputListener listener = new MyMouseInputListener();
        addMouseListener(listener);        
    }


	/**
     * Initializes painting colors.
     */
    protected void initColors() {
    	switch (state) {
		case NORMAL:
		{
			setBackground(NORMAL_COLOR);
			break;
		}
		case MOVED:
		{
			setBackground(MOVED_COLOR);
			break;
		}
		case REMOVED:
		{
			setBackground(REMOVED_COLOR);
			break;
		}
		default:
			break;
		}
    	repaint();
    }

    
    /**
     * Returns the process model object.
	 */
	public Process getProcess() {
		return process;
	}

	
	/**
	 * @return Returns the state.
	 */
	public State getState() {
		return state;
	}


	/**
	 * @param state The state to set.
	 */
	public void setState(State state) {
		this.state = state;
		initColors();
	}


	/**
	 * Returns the offset of the drag location from the
	 * component's upper left corner.
	 */
	public Point getDragOffset() {
		return dragOffset;
	}
	
    
    public void paint(Graphics g) {
        super.paint( g );

        // Save graphics state
        Color oldColor = g.getColor();
        Font oldFont = g.getFont();
        
        // set border color
        g.setColor( Color.BLACK );
        String name = process.getName();
        
        if ( !SQUARE_SHAPE ) {
        	// Draw an ellipse with border color
            g.fillArc( 0, 0, ICON_WIDTH, ICON_HEIGHT, 0, 360 );

            // Draw a smaller ellipse with background color
            g.setColor( getBackground() );
            g.fillArc( 2, 2, ICON_WIDTH - 4, ICON_HEIGHT - 4, 0, 360 );

            // Draw text label
            g.setColor( TEXT_COLOR );

            Font newFont = new Font( oldFont.getName(), oldFont.getStyle(), FONT_SIZE );
            g.setFont( newFont );
            Rectangle2D stringArea = g.getFontMetrics().getStringBounds(name, g);

            int startXpoint = ICON_WIDTH / 2 - (int) stringArea.getWidth() / 2 + 3;
            int startYpoint = ICON_HEIGHT / 2 + (int) stringArea.getHeight() / 2 - 3;

            g.drawString(name, startXpoint, startYpoint);

        } else {
        	// Draw a rectangle with border color
            g.fillRect( 0, 0, ICON_WIDTH, ICON_HEIGHT );

        	// Draw a smaller rectangle with background color
            g.setColor( getBackground() );
            g.fillRect( 2, 2, ICON_WIDTH - 4, ICON_HEIGHT - 4 );

            // Draw text label
            g.setColor( TEXT_COLOR );

            Font newFont = new Font( oldFont.getName(), oldFont.getStyle(), FONT_SIZE );
            g.setFont( newFont );
            Rectangle2D stringArea = g.getFontMetrics().getStringBounds(name, g);

            int startXpoint = ICON_WIDTH / 2 - (int) stringArea.getWidth() / 2 + 3;
            int startYpoint = ICON_HEIGHT / 2 + (int) stringArea.getHeight() / 2 - 3;

            g.drawString( name, startXpoint, startYpoint );
        }
        
        // Restore graphics state
        g.setColor(oldColor);
        g.setFont(oldFont);
    }

    
	/**
	 * Implements the Compararable-interface
	 */
    public int compareTo(ProcessIcon o) {
    	return process.compareTo(o.process);
    }

    
    /**
     * Listener for MouseInputEvents
     */
    private class MyMouseInputListener implements MouseListener {
    	public void mouseClicked( MouseEvent e ) {
    		// do nothing
    	}
    	
    	public void mousePressed( MouseEvent e ) {
    		// What's the offset of the mouse location
    		// from the component's origin?
    		dragOffset = new Point(e.getPoint());
    		MainWindow.getInstance().setDraggedProcess(self);
    	}

    	public void mouseReleased( MouseEvent e ) {
    		MainWindow.getInstance().setDraggedProcess(null);
    		dragOffset = new Point(0, 0);
    		
			// Check if we are transferring the process to some other thread.
    		Point old = self.getLocation();    		
    		Point newPoint = new Point();
    		newPoint.x = old.x + (e.getPoint().x);
    		newPoint.y = old.y + (e.getPoint().y);

			newPoint = SwingUtilities.convertPoint( self.getParent(), newPoint,
					MainWindow.getInstance() );
			
			Component component = SwingUtilities.getDeepestComponentAt(
					MainWindow.getInstance(), newPoint.x, newPoint.y );
			
			Container destContainer = SwingUtilities.getAncestorOfClass(ThreadPanel.class, component);
			Container srcContainer = SwingUtilities.getAncestorOfClass(ThreadPanel.class, self);

			if (destContainer == null || srcContainer == null) {
				return;
			}
			
			if (destContainer == srcContainer) {
				// The new location is also inside the same thread.
				
			} else {
				// The new location is not inside the same thread.
				// We are transferring the process to another thread
				ThreadPanel destThreadPanel = (ThreadPanel)destContainer;
				ThreadPanel srcThreadPanel = (ThreadPanel)srcContainer;
				Thread destThread = destThreadPanel.getThread();
				
				if ( !MainWindow.getInstance().isAutoActive() ) {
					// We can't transfer again a process that has already been
					// transferred away, but not yet activated.
					if (state == State.REMOVED) {
						JOptionPane.showMessageDialog(destThreadPanel,
								MainWindow.bundle.getString("PROCESS_MOVE_ERROR_MSG"),
								MainWindow.bundle.getString("PROCESS_MOVE_ERROR_TITLE"),
								JOptionPane.INFORMATION_MESSAGE);
						MainWindow.getInstance().repaint();
						return;
					}
					
					// The activation can be done after user's approval.				
					ProcessIcon p = destThreadPanel.getProcessIcon(process.getId());
					if (p != null) {
						// The destination thread already contains a process with
						// same id, meaning that we are transferring a moved
						// process back (canceling the previous move).
						destThreadPanel.removeProcessIcon(p);
						srcThreadPanel.removeProcessIcon(self);						
						self.setState(State.NORMAL);
						destThreadPanel.addProcessIcon(self);
						
					}  else if (state == State.MOVED ) {
						// Moving the process to another destination
						// without activating in previous.
						srcThreadPanel.removeProcessIcon(self);
						destThreadPanel.addProcessIcon(self);
						
					} else {
						// Moving normally
						ProcessIcon shadowIcon = new ProcessIcon(process);
						shadowIcon.setState(State.MOVED);
						self.setState(State.REMOVED);
						destThreadPanel.addProcessIcon(shadowIcon);
					}
					
				} else {
					// AutoActive. Remove the process from the source thread and
					// add it to the destination.
					MainWindow.getInstance().getConfiguration().getPeModel().moveNode(process, destThread);
					MainWindow.getInstance().sendMappingString();
				}
			}
    	}
    	
    	public void mouseEntered( MouseEvent e ) {
    		// do nothing
    	}
    	
    	public void mouseExited( MouseEvent e ) {
    		// do nothing
    	}    	
    }
}
