/**
 * File:    PePanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 8.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.ctrl;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.time.Millisecond;

import fi.cpu.Settings;
import fi.cpu.data.Chart;
import fi.cpu.data.Configuration;
import fi.cpu.data.ModelNode;
import fi.cpu.data.ProcessingElement;
import fi.cpu.data.ProcessingElementType;
import fi.cpu.data.Thread;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;
import fi.cpu.ui.ChartCreationDialog;
import fi.cpu.ui.MainWindow;
import fi.cpu.ui.graph.GraphPanel;


/**
 * The PePanel represents a processing element in the
 * control view.
 */
public class PePanel extends JPanel implements Comparable<PePanel> {
    public static final Color TEXT_BACKGROUND;
    public static final Color TEXT_COLOR;
    public static final Color BG_COLOR = Color.WHITE;
    public static final Color BORDER_COLOR = Color.BLACK;
    public static final int FONT_SIZE;
    private static final String ADD_GRAPH = "ADD_GRAPH";
    private static final String SHOW_GRAPH = "SHOW_GRAPH";
    private static final String CLEAR_PROCESSOR = "CLEAR_PROCESSOR";
    
    protected PePanel self;
    protected Configuration config;
    protected JPopupMenu popupMenu;
    protected ProcessingElement pe;
    private JLabel nameLabel;
    private JPanel threadPanelContainer;
    private JPanel placeHolder;
    private MyActionListener actionListener;
    private MyThreadListener threadListener;
    private MyProcessingElementListener peListener;
    private int rowCount = 0;
    protected Map<String, ThreadPanel> threadPanels;
    protected Map<String, GraphPanel> chartPanels;
    
    
    static {
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_BLUE") );
        TEXT_COLOR = new Color(red, green, blue);
 
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_BLUE") );
        TEXT_BACKGROUND = new Color(red, green, blue);
        
        FONT_SIZE = Integer.parseInt(Settings.getAttributeValue("PROCESSOR_TEXT_FONT"));
    }

    
    /**
     * Creates a new PePanel.
     * @param pe            The processing element model object.
     */
    public PePanel(ProcessingElement pe) {
    	super();
    	this.self = this;
        this.pe = pe;

        config = MainWindow.getInstance().getConfiguration();
        actionListener = new MyActionListener(null);
        threadListener = new MyThreadListener();
        peListener = new MyProcessingElementListener();
        threadPanels = new HashMap<String, ThreadPanel>();
        chartPanels = new HashMap<String, GraphPanel>();
        
        setBackground(TEXT_BACKGROUND);
        setBorder(BorderFactory.createLineBorder(BORDER_COLOR));
        setLayout(new GridBagLayout());

        addMouseListener(new MyMouseListener());
        config.getPeModel().addModelListener(new MyModelListener());

        // Create container panel for threads
        threadPanelContainer = new JPanel();
        threadPanelContainer.setLayout(new BoxLayout(threadPanelContainer, BoxLayout.Y_AXIS));
        threadPanelContainer.setPreferredSize(new Dimension(250, 140));

        // Create pe name label
        nameLabel = new JLabel(pe.getName());
        Font currentFont = nameLabel.getFont();
        Font newFont = new Font(currentFont.getName(), currentFont.getStyle(), FONT_SIZE);
        nameLabel.setBackground(TEXT_BACKGROUND);
        nameLabel.setForeground(TEXT_COLOR);
        nameLabel.setFont(newFont);
        
        rowCount = 0;
        
        // Add name label to view
        add(nameLabel, new GridBagConstraints(
        		0, rowCount++, GridBagConstraints.REMAINDER, 1,
        		0.0, 0.0, GridBagConstraints.NORTHWEST, GridBagConstraints.NONE,
        		new Insets(2, 2, 2, 2), 0, 0));

        // If cpu, add a button for clearing processes
        if (pe.getType() == ProcessingElementType.CPU) {
            JButton clearB = new JButton(MainWindow.bundle.getString("CLEAR_PROCESSOR"));
            clearB.setActionCommand(CLEAR_PROCESSOR);
            clearB.addActionListener(actionListener);
            add(clearB, new GridBagConstraints(
            		0, rowCount++, GridBagConstraints.REMAINDER, 1,
            		0.0, 0.0, GridBagConstraints.NORTHWEST, GridBagConstraints.NONE,
            		new Insets(0, 2, 2, 2), 0, 0));
        }
        
        // Add thread container panel
        add(threadPanelContainer, new GridBagConstraints(
        		0, rowCount++, GridBagConstraints.REMAINDER, 1,
        		1.0, 0.5, GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));

        // Add a placeholder for chart location
        placeHolder = new JPanel();
        placeHolder.setBackground( TEXT_BACKGROUND );
        add(placeHolder, new GridBagConstraints(
        		0, rowCount, GridBagConstraints.REMAINDER, 1,
        		1.0, 0.5, GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));
        
        setProcessingElement(pe);
    }

    
	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		
		nameLabel.setText(pe.getName());
		
		threadPanels.clear();
		
        Iterator<ModelNode> tIter = config.getPeModel().getChildren(pe).iterator();

        while (tIter.hasNext()) {
        	ModelNode node = tIter.next();
        	if (!(node instanceof Thread)) {
        		continue;
        	}
        	Thread t = (Thread) node;
        	t.addModelNodeListener(threadListener);
        	threadPanels.put(t.getId(), new ThreadPanel(t));
        }
        drawThreadPanels();
        
        Set<Map.Entry<String, GraphPanel>> chartPanelSet = chartPanels.entrySet();
        for (Map.Entry<String, GraphPanel> chartPanelEntry : chartPanelSet) {
        	chartPanelEntry.getValue().addMarker();
        }
	}

	
	/**
	 * Initializes chart panels.
	 */
	public void initChartPanels() {
		//chartPanels.clear();
		
        Iterator<Chart> chartIter = pe.getCharts().iterator();
        while (chartIter.hasNext()) {
        	Chart chart = chartIter.next();
        	if (!chartPanels.containsKey(chart.getId())) {
        		GraphPanel panel = new GraphPanel();
            	panel.setChart(chart);
            	chartPanels.put(chart.getId(), panel);
        	}
        }
        
        String visibleChart = pe.getVisibleChart();
        
        if (chartPanels.containsKey(visibleChart)) {
        	showChartPanel(visibleChart);
        } else if (visibleChart != null && chartPanels.size() > 0) {
        	chartIter = pe.getCharts().iterator();
        	Chart c = chartIter.next();
        	showChartPanel(c.getId());
        } else {
        	showChartPanel((String)null);        	
        }
        initPopupmenu();		
	}
	
    
    /**
     * Initializes the popup menu.
     */
    private void initPopupmenu() {
        // Create/clear the popup menu.
        if( popupMenu == null ) {
            popupMenu = new JPopupMenu();
        } else {
            popupMenu.removeAll();        	
        }
        
        // Item for adding new graph
        JMenuItem menuItem = new JMenuItem( MainWindow.bundle.getString(ADD_GRAPH) );
        menuItem.setActionCommand(ADD_GRAPH);
        menuItem.addActionListener(actionListener);
        popupMenu.add( menuItem );
                
        // Items for selecting added graphs
        if(chartPanels.size() > 0) {
            popupMenu.addSeparator();
            
            Iterator<String> idIter = chartPanels.keySet().iterator();
            while(idIter.hasNext()) {
            	String id = idIter.next();
            	GraphPanel panel = chartPanels.get(id);

                menuItem = new JMenuItem(panel.getChartModel().getTitle());
                menuItem.setActionCommand(SHOW_GRAPH);
                menuItem.addActionListener(new MyActionListener(id));
                popupMenu.add( menuItem );
            }
        }
    }
    
    
    /**
	 * @param pe The processing element to set.
	 */
	public void setProcessingElement(ProcessingElement pe) {
		this.pe.removeModelNodeListener(peListener);
		pe.setCharts(this.pe.getCharts());
		pe.setVisibleChart(this.pe.getVisibleChart());
		this.pe = pe;
        this.pe.addModelNodeListener(peListener);
		initPanel();
		initChartPanels();
	}


	/**
	 * @return Returns the processing element model object.
	 */
	public ProcessingElement getProcessingElement() {
		return pe;
	}
    
	
    /**
     * @return Returns a ThreadPanel corresponding to given process id.
     */
    public ThreadPanel getThreadPanel(String id) {
    	return threadPanels.get(id);
    }

    
    /**
     * Commits changes in process locations.
     */
    public void commitProcessMoves() {
    	Iterator<ThreadPanel> iter = threadPanels.values().iterator();
    	while (iter.hasNext()) {
    		ThreadPanel panel = iter.next();
    		panel.commitProcessMoves();
    	}
    }

    
    /**
     * @return Returns a chart panel corresponding to given id.
     */
    public GraphPanel getChartPanel(String id) {
    	return chartPanels.get(id);
    }

    
    /**
     * Adds the chart panel.
     * @param panel
     */
    protected void addChartPanel(GraphPanel panel) {
    	if (panel == null) {
    		return;
    	}
    	chartPanels.put(panel.getChartModel().getId(), panel);
    	initPopupmenu();
    	// Show the added chart
    	showChartPanel(panel);
    }
    

    /**
     * Removes the chart panel.
     * @param panel
     */
    protected void removeChartPanel(GraphPanel panel) {
    	if (panel == null) {
    		return;
    	}
    	String id = panel.getChartModel().getId();
    	chartPanels.remove(id);
    	initPopupmenu();

    	// If the removed chart was visible, show some other chart
        if (pe.getVisibleChart().equals(id) && chartPanels.size() > 0) {
        	Iterator<Chart> chartIter = pe.getCharts().iterator();
        	Chart c = chartIter.next();
        	showChartPanel(c.getId());
        }
    }

    
    /**
     * Shows a chart.
     * @param chartID The id of the chart to be shown.
     */
    protected void showChartPanel(String chartID) {
        GraphPanel panel = chartPanels.get(chartID);
        showChartPanel(panel);
    }
    
    
    /**
     * Shows a chart panel.
     * @param panel The panel to be shown.
     */
    protected void showChartPanel(GraphPanel panel) {
        remove(rowCount);
        if(panel == null) {
            add(placeHolder, new GridBagConstraints(
            		0, rowCount, GridBagConstraints.REMAINDER, 1,
            		1.0, 0.5, GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH,
            		new Insets(0, 0, 0, 0), 0, 0));
        	pe.setVisibleChart(null);
        	
        } else {
        	add(panel, new GridBagConstraints(
        			0, rowCount, GridBagConstraints.REMAINDER, 1,
        			1.0, 0.5, GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH,
        			new Insets(0, 0, 0, 0), 0, 0));
        	pe.setVisibleChart(panel.getChartModel().getId());
        }        
        revalidate();
        repaint();
    }

	
    /**
     * Draws thread panels.
     */
    protected void drawThreadPanels() {
    	// Remove old panels
        threadPanelContainer.removeAll();
        
		// Sort panels
		ArrayList<ThreadPanel> pList = new ArrayList<ThreadPanel>(threadPanels.values());
		Collections.sort(pList);

		// Add new panels
        Iterator<ThreadPanel> iter = pList.iterator();
        while(iter.hasNext()) {
        	ThreadPanel panel = iter.next();
        	threadPanelContainer.add(panel);
        }
        
        revalidate();
        repaint();
    }

    
	/**
	 * Implements the Compararable-interface
	 */
    public int compareTo(PePanel o) {
    	return pe.compareTo(o.pe);
    }
    
    
    /**
     * Listener for MouseEvents.
     */
    private class MyMouseListener extends MouseAdapter {
        public void mouseClicked(MouseEvent e) {
            if(SwingUtilities.isRightMouseButton(e)) {
                popupMenu.show( self, e.getX(), e.getY() );
            }
        }
    }
    
    
    /**
     * Listener for ActionEvents.
     */
    private class MyActionListener implements ActionListener {
    	private String chartId;
    	
    	public MyActionListener(String chartId) {
    		this.chartId = chartId;
    	}
    	
    	public void actionPerformed(ActionEvent e) {
    		String cmd = e.getActionCommand();

    		if(cmd.equals(ADD_GRAPH)) {
            	ChartCreationDialog.getInstance().setCPUspecifig(true);
    			Chart chart = ChartCreationDialog.getInstance().showDialog();
    			if (chart != null) {
    				// modify the model, which then fires CHART_ADDED-event
    				pe.setChart(chart.getId(), chart);
    			}

    		} else if(cmd.equals(SHOW_GRAPH)) {
                showChartPanel(chartId);

    		} else if(cmd.equals(CLEAR_PROCESSOR)) {
    			config.clearProsessingElement(pe);
    			MainWindow.getInstance().sendMappingString();
            }    		
    	}
    }
    
    
    /**
     * Listener for ProcessingElementEvents.
     */
    private class MyProcessingElementListener implements ModelNodeListener {
    	public void modelNodeChanged(ModelNodeEvent e) {
    		int eventId = e.getEventId();
    		Object idObj = e.getEventObject();

    		if (eventId == ProcessingElement.CHART_ADDED_EVENT) {
    			if (idObj instanceof String) {
    				String id = (String) idObj;
    				Chart c = pe.getChart(id);
    				if (c != null) {
    					GraphPanel panel = new GraphPanel();
    					panel.setChart(c);
    					addChartPanel(panel);
    				}
    			}
    			
    		} else if (eventId == ProcessingElement.CHART_REMOVED_EVENT) {
    			if (idObj instanceof String) {
    				String id = (String) idObj;
    				GraphPanel panel = chartPanels.get(id);
    				if (panel != null) {
    					removeChartPanel(panel);
    				}
    			}
    		}
    	}
    }
    
    
    /**
     * Listener for ThreadEvents from contained threads.
     */
    private class MyThreadListener implements ModelNodeListener {
    	public void modelNodeChanged(ModelNodeEvent e) {
    		int eventId = e.getEventId();
    		if (eventId == Thread.PRIORITY_CHANGED_EVENT) {
        		drawThreadPanels();    			
    		}
    	}
    }
    
	
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		if (parent != null && parent.equals(pe)) {
    			// PE's children changed.
    			initPanel();
    		}
    	}
    	public void nodeRemoved(ModelEvent e) {
    		ModelNode node = e.getNode();
    		ModelNode oldParent = e.getOldParent();
    		
    		if (node != null && node.equals(pe)) {
    			// PE was removed.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		} else if (oldParent != null && oldParent.equals(pe)) {
    			// PE's children changed.
    			initPanel();
    		}
    	}
    	public void nodeMoved(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		ModelNode oldParent = e.getOldParent();
    		
    		if ((parent != null && parent.equals(pe))
    				|| (oldParent != null && oldParent.equals(pe))) {
    			// PE's children changed.
    			initPanel();    			
    		}
    	}
    	public void structureChanged(ModelEvent e) {
    		ModelNode node = e.getNode();
    		
    		if (node != null && node.equals(pe)) {
    			// Structure changed starting from this node.
    			initPanel();
    		} else if (config.getPeModel().isChildOf(pe, node)) {
    			// PE is part of the changed structure.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		}
    	}
    }
}
