/*
 * CVS-tietoja:
 * $Author: perttul5 $ 
 * $Date: 2008-05-20 17:23:56 +0300 (ti, 20 touko 2008) $ 
 * $Revision: 3514 $
 *
 * Created on 5.11.2004
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *
 */
package fi.cpu.handler;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.SocketException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Vector;
import java.util.logging.Logger;

import fi.cpu.Settings;
import fi.cpu.data.ChartDataLogger;
import fi.cpu.data.Configuration;
import fi.cpu.data.Connection;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Process;
import fi.cpu.data.Service;
import fi.cpu.data.ServiceActivationInfo;
import fi.cpu.data.TraceDataLogger;
import fi.cpu.ui.MainWindow;
import fi.cpu.ui.xml.UIConfigIO;
import fi.cpu.ui.xml.XML_Source_FPGA;
import fi.cpu.xml.bind.measurement.GraphType;
import fi.cpu.xml.bind.measurement.MeasurementType;
import fi.cpu.xml.bind.measurement.ProcessExecType;
import fi.cpu.xml.bind.measurement.SignalType;


/**
 * @author jsikio
 * 
 * Class that know how to parse data from serial port
 *  
 */
public abstract class DataHandler extends java.lang.Thread implements Handler
{
    /**
     * Static logger
     */
    public static Logger logger = null;

    static
    {
        logger = Logger.getLogger( "fi.sensor" + "." + "DataHandler" );
    }

    protected boolean useDB_ = false;

    protected BufferedReader bufferedReader_ = null;

    protected InputStream byteReader_ = null;

    protected BufferedWriter bufferedWriter_ = null;

    protected OutputStream byteWriter_ = null;

    private boolean writingBytes_ = false;

    private byte[] bytesToWrite_;

    private int writeOffset_ = 0;

    private boolean handleData_ = false;

    private boolean run_ = true;

    //Tnne pusuriin luetaan kaikki data
    protected Vector<String> dataBuffer_ = new Vector<String>();

    //Debug objects
    private Map debugRows_ = Collections.synchronizedMap( new TreeMap() );

    private BufferedWriter debugFileWriter_ = null;

    protected DataHandler()
    {
    }

    private long startTime_ = -1;

    public void readData()
    {
        try
        {
            String tmp;
            
            while ( (tmp = bufferedReader_.readLine()) != null )
            {
                dataBuffer_.add( tmp );
                break;
            }
        }
        catch ( SocketException exception )
        {
            System.out.println("Socket closed");
            System.exit(1);
        }
        catch ( IOException exception )
        {
            exception.printStackTrace();
        }
        catch(Exception e )
        {
            e.printStackTrace();
        }
    }

    /**
     * Write data to serial port. First write the message, then new line and
     * flush the stream.
     * 
     * @param message
     * @throws IOException
     */
    public void writeToDestination( String message ) throws IOException
    {
        long sleepTime = Long.parseLong( Settings.getAttributeValue( "SLEEP_TIME_WHILE_WRITING" ) );
                
        if( bufferedWriter_ != null )
        {
            /*  Deprecated ?
        	//Write start message
            String startMessage = "AAA";
            
            for( int i=0; i < startMessage.length(); i++ )
            {
                bufferedWriter_.write( startMessage.charAt(i) );
                bufferedWriter_.flush();
                try
                {
                    java.lang.Thread.sleep( sleepTime );
                }
                catch ( InterruptedException e )
                {
                    e.printStackTrace();
                }
            }
            */
            
            //Write message
            for( int i=0; i < message.length(); i++ )
            {
                bufferedWriter_.write( message.charAt(i) );
                bufferedWriter_.flush();
                try
                {
                    java.lang.Thread.sleep( sleepTime );
                }
                catch ( InterruptedException e )
                {
                    e.printStackTrace();
                }
                
            }
            bufferedWriter_.newLine();
            bufferedWriter_.flush();

        }
    }

    public void run()
    {

        System.out.println( "Aloitetaan" );

        while ( run_ )
        {
            if ( dataBuffer_.size() == 0 )
            {
                try
                {
                    //Dataa ei ole
                    super.sleep( 500 );
                }
                catch ( InterruptedException e )
                {
                    e.printStackTrace();
                }
            }
            else
            {
               //System.out.println( "Ksitelln dataa" );

                try
                {
                    String data = (String) dataBuffer_.remove( 0 );

                    handleData( data );
                }
                catch ( Exception e )
                {
                    e.printStackTrace();
                }
            }

        }

        System.out.println( "The end" );

    }

    /**
     * @param data
     */
    private void handleData( String data )
    {
        data = data.trim();
        
        //System.out.println( "Data: " + data );

        if ( data.length() == 0 )
        {
            return;
        }

        XML_Source_FPGA fpga = new XML_Source_FPGA();
        Configuration config = MainWindow.getInstance().getConfiguration();

        if ( data.startsWith( "<configuration>" ) )
        {
        	UIConfigIO.readConfigurationString(config, MainWindow.getInstance(), data);
        	MainWindow.getInstance().configurationChanged();
        }
        else if ( data.startsWith( "<measurement" ) )
        {
            try
            {
                MeasurementType me = fpga.getMeasurement( data );
                
                MainWindow.getInstance().setSimulationTime(me.getTime());

                // handle graph measurements
                List graphMeasurementList = me.getGraph();

                for ( int i = 0; i < graphMeasurementList.size(); i++ )
                {
                	
                	Object listItem = graphMeasurementList.get(i);
                	
                	if (listItem instanceof GraphType)
                	{
                		GraphType graph = (GraphType)listItem;
                		
                		// Add value to chart
                		if ( graph.getCpuId() != -1 ) {
                			MainWindow.getInstance().addValueToChart(
                					""+graph.getCpuId(), graph.getId(), graph.getValue());
                		} else {
                			MainWindow.getInstance().addValueToChart(
                					null, graph.getId(), graph.getValue());
                		}
                	}
                }
                
                // handle signal measurements
                ServiceActivationInfo serviceActivations =
                    new ServiceActivationInfo();
                List signalMeasurementList = me.getSignal();
                
                for ( int i = 0; i < signalMeasurementList.size(); i++ )
                {
                	Object listItem = signalMeasurementList.get(i);
                	
                	if (listItem instanceof SignalType)
                	{
                		SignalType signal = (SignalType)listItem;
                		int sourcePid = signal.getSourcePid();
                		int destPid = signal.getDestPid();
                		int value = signal.getValue();
                		
                		MainWindow.getInstance().getSignalsPanel().getTableModel().setValue(sourcePid, destPid, value);
                		
                		MainWindow.getInstance().getProcessesPanel().addSignalCount(sourcePid, destPid, value);
                        
                        if (sourcePid == destPid) {
                            continue;
                        }
                        
                        // Find services (if any) that contain the processes
                        Service srcService = null;
                        Service destService = null;
                        
                        List<ModelNode> processNodes =
                            config.getServiceModel().getNodesOfType(Process.class);
                        Iterator<ModelNode> nodeIter = processNodes.iterator();
                        
                        while (nodeIter.hasNext()) {
                            ModelNode node = nodeIter.next();                            
                            if (node instanceof Process) {
                                Process p = (Process) node;
                                
                                // Is this the source process?
                                if (p.getId() == sourcePid) {
                                    ModelNode parentNode = p.getParentNode();
                                    if (parentNode instanceof Service) {
                                        srcService = (Service) parentNode;
                                    }
                                }
                                
                                // Is this the destination process?
                                if (p.getId() == destPid) {
                                    ModelNode parentNode = p.getParentNode();
                                    if (parentNode instanceof Service) {
                                        destService = (Service) parentNode;
                                    }
                                }

                                if (srcService != null && destService != null) {
                                    // both services were found
                                    break;
                                }
                            }
                        }
                        
                        // Check that services were found and that they are
                        // the same
                        if (srcService == null
                                || destService == null
                                || srcService == destService) {
                            continue;
                        }
                        
                        // Find the connection between services
                        List<ModelNode> connectionNodes =
                            config.getServiceModel().getNodesOfType(Connection.class);
                        nodeIter = connectionNodes.iterator();
                        
                        while (nodeIter.hasNext()) {
                            ModelNode node = nodeIter.next();
                            
                            if (node instanceof Connection) {
                                Connection c = (Connection) node;
                                
                                int destId = c.getDestService();
                                int srcId = c.getSrcService();
                                int destServId = destService.getId();
                                int srcServId = srcService.getId();
                                
                                // if connection's endpoints match with signal's
                                // endpoints, we have found the connection that should
                                // be activated
                                if (destId == destServId && srcId == srcServId) {
                                    c.setSignalValue(sourcePid, destPid, value);
                                    
                                    serviceActivations.addActivatedItem(
                                            ServiceActivationInfo.CONNECTION,
                                            c.getId(),
                                            c.getSignalValuesSum());
                                    break;
                                }
                            }
                        }
                	}
                }
                
                // handle process execution measurements
                List pExecMeasurementsList = me.getProcessExec();                
                for ( int i = 0; i < pExecMeasurementsList.size(); i++ )
                {
                	Object listItem = pExecMeasurementsList.get(i);
                	
                	if (listItem instanceof ProcessExecType)
                	{
                		ProcessExecType pExec = (ProcessExecType)listItem;
                		Integer pid = pExec.getPid();
                		
                		// for all processes in pe model
                		List<ModelNode> processNodes =
                            config.getPeModel().getNodesOfType(Process.class);
                		Iterator<ModelNode> nodeIter = processNodes.iterator();
                		
                		while (nodeIter.hasNext()) {
                			ModelNode node = nodeIter.next();
                			
                			if (node instanceof Process) {
								Process p = (Process) node;
            					if (pid == p.getId()) {
            						p.addExecDataPoint(pExec.getCount(),
            										   pExec.getExecTime(),
            								           pExec.getLocalCommTime(),
            								           pExec.getLocalCommBytes(),
            								           pExec.getRemoteCommTime(),
            								           pExec.getRemoteCommBytes(),
            								           pExec.getSignalQueue(),
            								           pExec.getResponseTime(),
            								           pExec.getLatency());
            						break;
            					}
							}
                		}
                		
                		// for all processes in service model
                		processNodes = config.getServiceModel().getNodesOfType(Process.class);
                		nodeIter = processNodes.iterator();
                		
                		while (nodeIter.hasNext()) {
                			ModelNode node = nodeIter.next();
                			
                			if (node instanceof Process) {
								Process p = (Process) node;
            					if (pid == p.getId()) {
            						p.addExecDataPoint(pExec.getCount(),
 										   pExec.getExecTime(),
 								           pExec.getLocalCommTime(),
 								           pExec.getLocalCommBytes(),
 								           pExec.getRemoteCommTime(),
 								           pExec.getRemoteCommBytes(),
 								           pExec.getSignalQueue(),
 								           pExec.getResponseTime(),
 								           pExec.getLatency());
                                    
                                    // add service to be activated
                                    ModelNode parentNode = p.getParentNode();
                                    if (parentNode instanceof Service) {
                                        Service s = (Service) parentNode;
                                        serviceActivations.addActivatedItem(
                                                ServiceActivationInfo.SERVICE,
                                                s.getId(),
                                                (int)s.getTotExecTime());
                                    }
            						break;
            					}
							}
                		}
                	}
                }
                
                MainWindow.getInstance().getServicePanel().setActivationStates(serviceActivations);
            }
            
            catch ( Exception exp )
            {
                exp.printStackTrace();
            }
        }
        else if ( data.startsWith( "<log>" ) )
        {
        	// handle start of execution trace message
        	if (TraceDataLogger.getInstance().getStatus() == TraceDataLogger.TracingStatus.STOPPED) {
        		//nothing to do actually
        	}
        }
        else if ( data.startsWith( "<transition" ) || data.startsWith( "<start" ) )
        {
        	// handle execution trace log message
        	if (TraceDataLogger.getInstance().getStatus() == TraceDataLogger.TracingStatus.TRACING
        		|| TraceDataLogger.getInstance().getStatus() == TraceDataLogger.TracingStatus.WAITING_FOR_STOP) {
	        	try {
	        		TraceDataLogger.getInstance().logTraceData(data);
		        } catch (IOException ioe) {
		    		String eMsg = TraceDataLogger.getInstance().getErrorMessage();
		    		if (eMsg != null) {
		    			MainWindow.getInstance().stopTracing(eMsg);
		    		} else {
		    			MainWindow.getInstance().stopTracing(ioe.getMessage());
		    		}
		    	}
        	}
        }
        else if ( data.startsWith( "</log>" ) )
        {
        	// handle stop of execution trace message
        	if (TraceDataLogger.getInstance().getStatus() == TraceDataLogger.TracingStatus.TRACING
            		|| TraceDataLogger.getInstance().getStatus() == TraceDataLogger.TracingStatus.WAITING_FOR_STOP) {
        		TraceDataLogger.getInstance().traceReady();
        	}
        }
        else if ( data.startsWith( "<usecase ready>" ) )
        {
        	MainWindow.getInstance().stopTracing(null);
        }
        else if ( data.startsWith( "<sync/>") )
        {
        	MainWindow.getInstance().getProcessesPanel().sync();
        }
        else
        {
            MainWindow.getInstance().addDebugMessage( data );
        }
    }

    /**
     * Writes bytes to serial port. This is asynchronous. TODO: timeout handling
     * 
     * @param bytes
     *                   bytes to write, bytes.length must be >1
     * @return
     * @throws IOException
     */
    private synchronized void writeBytes( byte[] bytes ) throws IOException
    {

        bytesToWrite_ = bytes;
        writingBytes_ = true;
        writeOffset_ = 0;
        logger.info( "start writing..." );

        byte[] tmp = new byte[1];

        // lets write first byte, the echo is checked in readAndHandleData
        // method
        // and next bytes are transmitted in there also.
        byteWriter_.write( bytes, 0, 1 );
        logger.info( "Wrote byte: " + bytesToWrite_[0] );
    }
}
