/**
 * File:    Thread.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 23.2.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.tree.TreeNode;

import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;


/**
 * Class for thread information.
 */
public class Thread extends ModelNode implements Comparable<Thread> {
	public static final int DATA_CHANGED_EVENT = 1;
	public static final int PRIORITY_CHANGED_EVENT = 2;
	
	private String id;
	private String name;
	private int priority;
	private int execCount;
	private double avgExecTime;
	private double totExecTime;
	private MyProcessNodeListener processListener;
	
	/**
	 * Creates a new thread.
	 */
	public Thread(String id, String name, int priority) {
		super(name);
		this.id = id;
		this.name = name;
		this.priority = priority;
		
		processListener = new MyProcessNodeListener();
	}
	

	/**
	 * Returns the name.
	 */
	public String getName() {
		return name;
	}


	/**
	 * Sets the name.
	 */
	public void setName(String name) {
		this.name = name;
	}


	/**
	 * Returns the id.
	 */
	public String getId() {
		return id;
	}


	/**
	 * Returns the priority.
	 */
	public int getPriority() {
		return priority;
	}
	
	
	/**
	 * Sets the priority.
	 */
	public void setPriority(int p) {
		priority = p;
		fireModelNodeChanged(new ModelNodeEvent(this, PRIORITY_CHANGED_EVENT, new Integer(p)));
	}

	
	/**
	 * Returns the average execution time.
	 */
	public double getAvgExecTime() {
		return avgExecTime;
	}


	/**
	 * Returns the execution count.
	 */
	public int getExecCount() {
		return execCount;
	}


	/**
	 * Returns the total execution time.
	 */
	public double getTotExecTime() {
		return totExecTime;
	}
	
	
	/**
	 * Calculates execution values.
	 */
	protected void calcExecValues() {
		int eCount = 0;
		double avgETime = 0;
		double totETime = 0;

		List<Process> pList = new ArrayList<Process>();
		for (int i=0; i<getChildCount(); ++i) {
			TreeNode child = getChildAt(i);
			if (child instanceof Process) {
				Process p = (Process) child;
				pList.add(p);
			}
		}
		
		Iterator<Process> pIter = pList.iterator();
		
		// calculate sums from all processes
		while (pIter.hasNext()) {
			Process p = pIter.next();
			eCount += p.getExecCount();
			totETime += p.getTotExecTime();
		}

		// calculate the thread average exec time
		if (eCount > 0) {
			avgETime = totETime / eCount;
		}
		
		execCount = eCount;
		avgExecTime = avgETime;
		totExecTime = totETime;
		
		fireModelNodeChanged(new ModelNodeEvent(this, DATA_CHANGED_EVENT));
	}
	
	
	/**
	 * Implements the Compararable-interface
	 */
	public int compareTo(Thread o) {
		if (priority < o.priority) {
			return -1;
		} else if (priority == o.priority) {
			return 0;
		} else {
			return 1;
		}
	}

	
	// Methods for tracking changes in this nodes children.

	/* (non-Javadoc)
	 * @see fi.cpu.data.ModelNode#childAdded(fi.cpu.data.ModelNode)
	 */
	public void childAdded(ModelNode child) {
		if (child instanceof Process) {
			Process p = (Process) child;
			p.addModelNodeListener(processListener);
		}
		calcExecValues();
	}

	/* (non-Javadoc)
	 * @see fi.cpu.data.ModelNode#childRemoved(fi.cpu.data.ModelNode)
	 */
	public void childRemoved(ModelNode child) {
		if (child instanceof Process) {
			Process p = (Process) child;
			p.removeModelNodeListener(processListener);
			calcExecValues();
		}
	}

	
	/**
	 * Listener for events from contained processes.
	 */
	private class MyProcessNodeListener implements ModelNodeListener {
		public void modelNodeChanged(ModelNodeEvent e) {
			int eventId = e.getEventId();
			if  (eventId == Process.DATA_CHANGED_EVENT) {
				calcExecValues();
			}
		}
	}
}
