/**
 *
 * @file trigger.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * $Id: trigger.cc 1420 2010-09-07 12:51:24Z lehton87 $
 *
 */

#include "trigger.hh"

#include "processing_element.hh"

#include <iostream>
#include <exception>
#include <algorithm>

namespace sctg
{
   Trigger::Trigger(const boost::property_tree::ptree& pt, 
		    Configuration& config,
		    Task* ownerTask)    
      :
      _triggered(0),
      _bytesSent(0),
      _cyclesExecuted(0),
      _receivedBytes(0),      
      _config(config),
      _bytesInBuffer(0),
      _currentReceive(sc_core::SC_ZERO_TIME)
   {
      using boost::property_tree::ptree;

      _ownerTask = ownerTask;
      _nextState = READY;

      std::string dep = pt.get<std::string>("<xmlattr>.dependence_type", "or");
      if(dep == "or")
      {	_dependence = OR; }
      else if(dep == "and")
      { _dependence = AND; }
      else
      {
	 std::string err = "Unknown dependence_type " + dep;
	 throw std::runtime_error(err.c_str());
      }

      for(ptree::const_iterator iter = pt.begin(); iter != pt.end(); ++iter)
      {
	 if((*iter).first == "exec_count")
	 {
	    _ExecCount* execCount = new _ExecCount;
	    execCount->period = (*iter).second.get_optional<unsigned long int>
	       ("<xmlattr>.mod_period");
	    execCount->min = (*iter).second.get_optional<unsigned long int>
	       ("<xmlattr>.min");
	    execCount->max = (*iter).second.get_optional<unsigned long int>
	       ("<xmlattr>.max");
	    boost::optional<unsigned long int> phase = 
	       (*iter).second.get_optional<unsigned long int>
	       ("<xmlattr>.mod_phase");
	    if(phase)
	    {
	       execCount->min = *phase;
	       execCount->max = *phase;
	    }
	    if(execCount->period && !execCount->min)
	    {	execCount->min = 0; }
	    if(execCount->period && !execCount->max)
	    {	execCount->max = execCount->period; }
	    parseExecCount((*iter).second, execCount, config);
	    _execCounts.push_back(execCount);
	 }
	 else if((*iter).first == "in_port")
	 {
	    unsigned long int id = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.id");
	    _inPorts.push_back(id);
	    // Add empty token queue for this port
	    _receivedPackets.push_back(std::queue<tgToken>());
	 }
      }
    
      _newOp = false;
    
   }

   Trigger::~Trigger()
   {
      for(unsigned int i = 0; i < _execCounts.size(); ++i)
      {	
	 for(unsigned int j = 0; j < _execCounts.at(i)->operations.size(); ++j)
	 {
	    for(unsigned int k = 0; 
		k < _execCounts.at(i)->operations.at(j)->amounts.size(); ++k)
	    {
	       delete _execCounts.at(i)->operations.at(j)->amounts.at(k);
	       _execCounts.at(i)->operations.at(j)->amounts.at(k) = 0;
	    }
	    delete _execCounts.at(i)->operations.at(j);
	    _execCounts.at(i)->operations.at(j) = 0;
	 }
	 delete _execCounts.at(i);
	 _execCounts.at(i) = 0;
      }
   }

  
   OperationType Trigger::getOperation()
   {
      // Don't ask if there is nothing
      if(_operationQueue.empty())
      {
	 throw std::runtime_error
	    ("Trigger::getOperation: Can't get operation from empty queue");
      }
      return _operationQueue.front().type;
   }

   bool Trigger::isNewOperation()
   {
      return _newOp;
   }

   unsigned long int Trigger::getAmount()
   {
      if(_operationQueue.empty())
      {	throw std::runtime_error("getAmount(): queue is empty"); }
      return _operationQueue.front().amount;
   }

   unsigned long int Trigger::getOutPort()
   {
      if(_operationQueue.empty())
      {	throw std::runtime_error("getOutPort(): queue is empty"); }
      if(_operationQueue.front().type == EXECUTE )
      {	throw std::runtime_error("getOutPort(): EXECUTE has no outPort"); }
      return _operationQueue.front().outPort;
   }

   unsigned long int Trigger::consume(unsigned long int amount)
   {
      if(_operationQueue.empty())
      {	throw std::runtime_error("consume(): queue is empty"); }
      if(_operationQueue.front().amount < amount )
      {	
	 amount = _operationQueue.front().amount;
	 //throw std::runtime_error
	 //("consume(): amount bigger than operation's");
      }
    
      switch(_operationQueue.front().type)
      {
	 case SEND:
	    _bytesSent += amount;
	    break;
	 case EXECUTE:
	    _cyclesExecuted += amount;
	    break;
	 default:
	    break;
      }

      _operationQueue.front().amount -= amount;
      // if nothing is left remove operation from queue
      if(_operationQueue.front().amount == 0)
      {
	 _operationQueue.pop();
	 _newOp = true;
	 // Update how many times this trigger has been triggered
	 if(_operationQueue.empty())
	 { 
	    // Remove used memory from buffer (bytes from event not counted)
	    _config.getBufferByPe
	       (_config.getPeByTask(_ownerTask->getId())
		->getId())->removeToken(_receivedBytes - _eventBytes);
	    _bytesInBuffer -= _receivedBytes;
	    ++_triggered; 	    
	    _ownerTask->incTriggered();
	    _ownerTask->changeState(_nextState); 	    
	 }
	 _ownerTask->setLastResponse(sc_core::sc_time_stamp()-_currentReceive);
	 return 0;
      }
      else
      {
	 _newOp = false;
	 return _operationQueue.front().amount;
      }
   }

   bool Trigger::hasInPort(unsigned long int id)
   {
      if(std::find(_inPorts.begin(), _inPorts.end(), id) != _inPorts.end())
      { return true; }
      else
      { return false; }
   }

   void Trigger::buildQueue()
   {
      if(_operationQueue.empty())
      {
	 for(unsigned int i = 0; i < _execCounts.size(); ++i)
	 {
	    // Add this exec_count if it has no period, min and max
	    //  or it's periodical and (min <= _triggered%period <= max)
	    //  or not periodical and _triggered >= min
	    //  or not periodical and _triggered <= max
	    if((!_execCounts.at(i)->period && !_execCounts.at(i)->min &&
		!_execCounts.at(i)->max) ||
	       (_execCounts.at(i)->period && 
		*_execCounts.at(i)->min <= 
		(_triggered % *_execCounts.at(i)->period) &&
		*_execCounts.at(i)->max >= 
		(_triggered % *_execCounts.at(i)->period)) ||
	       (!_execCounts.at(i)->period && _execCounts.at(i)->min &&
		*_execCounts.at(i)->min <= _triggered) ||
	       (!_execCounts.at(i)->period && _execCounts.at(i)->max &&
		*_execCounts.at(i)->max >= _triggered))
	    {		
	       for(unsigned int j = 0; j < _execCounts.at(i)->operations.size();
		   ++j)
	       {
		  _nextState = _execCounts.at(i)->nextState;
		  // Add current operation if random < probability
		  if(_execCounts.at(i)->operations.at(j)->prob == 1.0 ||
		     _config.random() < 
		     _execCounts.at(i)->operations.at(j)->prob)
		  {
		     _OperationList op;
		     op.type = _execCounts.at(i)->operations.at(j)->type;
		     op.outPort = _execCounts.at(i)->operations.at(j)->outPort;
		     op.amount = 0;			
		     for(unsigned int k = 0; 
			 k < _execCounts.at(i)->operations.at(j)->
			    amounts.size(); ++k)
		     {
			if(op.type == EXECUTE)
			{			  
			   op.amount += _execCounts.at(i)->
			      operations.at(j)->amounts.at(k)->
			      value(_receivedBytes) / 
			      getFactor(_execCounts.at(i)->
					operations.at(j)->factors.at(k));
			}			  
			else
			{
			   op.amount += _execCounts.at(i)->
			      operations.at(j)->amounts.at(k)->
			      value(_receivedBytes);
			}
		     }
		     _operationQueue.push(op);
		     /*std::cout << "Operation type " << op.type
		       << " outPort " << op.outPort
		       << " amount " << op.amount << std::endl;*/
		  }
	       }
	    }
	 }
      }
      if(_operationQueue.empty())
      {
	 throw std::runtime_error
	    ("Trigger::buildQueue: operationQueue is empty");
      }
   }

   void Trigger::parseExecCount(const boost::property_tree::ptree& pt,
				_ExecCount* execCount, 
				Configuration& config)
   {
      execCount->nextState = READY;

      using boost::property_tree::ptree;
      for(ptree::const_iterator iter = pt.begin(); iter != pt.end(); ++iter)
      {
	 if((*iter).first == "op_count")
	 {
	    _Operation* oper = new _Operation;
	    oper->type = EXECUTE;	    
	    oper->outPort = 0;
	    oper->prob = (*iter).second.get<double>
	       ("<xmlattr>.prob", 1.0);
	    for(ptree::const_iterator iter2 = (*iter).second.begin(); 
		iter2 != (*iter).second.end(); ++iter2)
	    {		
	       if((*iter2).first == "int_ops")
	       {
		  oper->factors.push_back(INT);
		  //(config.getPeByTask(_ownerTask->getId())->getIntOps());
		  oper->amounts.push_back(new Amount<double>
					  ((*iter2).second, config)); 
	       }
	       else if((*iter2).first == "float_ops")
	       {
		  oper->factors.push_back(FLOAT);
		  //(config.getPeByTask(_ownerTask->getId())->getFloatOps());
		  oper->amounts.push_back(new Amount<double>
					  ((*iter2).second, config)); 
	       }
	       else if((*iter2).first == "mem_ops")
	       {		    
		  oper->factors.push_back(MEM);
		  //(config.getPeByTask(_ownerTask->getId())->getMemOps());
		  oper->amounts.push_back(new Amount<double>
					  ((*iter2).second, config)); 
	       }
	    }
	    execCount->operations.push_back(oper);
	 }
	 else if((*iter).first == "send")
	 {
	    _Operation* oper = new _Operation;
	    oper->type = SEND;	    
	    oper->outPort = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.out_id");
	    oper->prob = (*iter).second.get<double>
	       ("<xmlattr>.prob", 1.0);
	    for(ptree::const_iterator iter2 = (*iter).second.begin(); 
		iter2 != (*iter).second.end(); ++iter2)
	    {
	       if((*iter2).first == "byte_amount")
	       {
		  oper->amounts.push_back(new Amount<double>
					  ((*iter2).second, config)); 
	       }
	    }
	    execCount->operations.push_back(oper);
	 }
	 else if((*iter).first == "read")
	 {
	 }
	 else if((*iter).first == "poll")
	 {
	 }
	 else if((*iter).first == "next_state")
	 {
	    if((*iter).second.get<std::string>("<xmlattr>.value") == "FREE")
	    {
	       execCount->nextState = FREE;
	       // No need to parse after this
	       break;
	    }
	 }
      }
   }

   void Trigger::receive(tgToken token)
   {
      // Check that packet was for this token
      if(!hasInPort(token.destination))
      {	throw std::runtime_error("Token to a wrong trigger"); }
      // Add token to its queue
      for(unsigned int i = 0; i < _inPorts.size(); ++i)
      {
	 if(_inPorts.at(i) == token.destination)
	 {
	    _receivedPackets.at(i).push(token);
	    _bytesInBuffer += token.size;
	 }
      }
      // Check if receiving this token makes this trigger active
      resolve();
   }
  
   void Trigger::resolve()
   {
      // Resolve new trigger state if not active already
      if(_operationQueue.empty())
      {
	 if(_dependence == AND)
	 {
	    // Build new queue only if all ports have received a token
	    for(unsigned int i = 0; i < _receivedPackets.size(); ++i)
	    {
	       if(_receivedPackets.at(i).empty()) {return;}
	    }
	    // Calculate new byte amount by adding all tokens together
	    //  and remove packets from queue
	    _receivedBytes = 0;
	    _eventBytes = 0;
	    sc_core::sc_time latestArrival = sc_core::SC_ZERO_TIME;
	    for(unsigned int i = 0; i < _receivedPackets.size(); ++i)
	    {
	       _receivedBytes += _receivedPackets.at(i).front().size;
	       if(_receivedPackets.at(i).front().isEvent)
	       {
		  _eventBytes += _receivedPackets.at(i).front().size;
	       }
	       if(_receivedPackets.at(i).front().timeReceived > latestArrival)
	       {
		  latestArrival = _receivedPackets.at(i).front().timeReceived;
	       }
	       _currentReceive = latestArrival;
	       /* / Remove token from PE's buffer
		  _config.getBufferByPe
		  (_config.getPeByTask(_ownerTask->getId())
		  ->getId())->removeToken(_receivedPackets.at(i).front().size);
	       */
	       _receivedPackets.at(i).pop();
	    }
	    // Build queue again
	    buildQueue();
	 }
	 else // i.e. _dependence is OR
	 {
	    // Take amount from the oldest token
	    bool found = false;
	    unsigned int index = 0;
	    sc_core::sc_time time;
	    for(unsigned int i = 0; i < _receivedPackets.size(); ++i)
	    {
	       if(!_receivedPackets.at(i).empty()) 
	       {
		  if(found == false)
		  {
		     found = true;
		     index = i;
		     time = _receivedPackets.at(i).front().timeReceived;
		  }
		  else
		  {
		     if(time > _receivedPackets.at(i).front().timeReceived)
		     {
			index = i;
			time = _receivedPackets.at(i).front().timeReceived;
		     }
		  }
	       }
	    }
	    
	    if(found == true)
	    {
	       // Store amount and remove token from queue
	       _eventBytes = 0;
	       _receivedBytes = _receivedPackets.at(index).front().size;
	       if(_receivedPackets.at(index).front().isEvent)
	       {
		  _eventBytes += _receivedPackets.at(index).front().size;
	       }
	       _currentReceive = 
		  _receivedPackets.at(index).front().timeReceived;
	       /* / Free tokens memory;
		  _config.getBufferByPe
		  (_config.getPeByTask(_ownerTask->getId())
		  ->getId())->removeToken(_receivedPackets.at(index).
		  front().size); */
	       _receivedPackets.at(index).pop();
	       // Build queue again
	       buildQueue();
	    }
	 }
      }
   }

   bool Trigger::isActive()
   {
      resolve();
      return !_operationQueue.empty();
   }

   unsigned long int Trigger::getTimesTriggered() const
   {
      return _triggered;
   }

   unsigned long int Trigger::getBytesSent() const
   {
      return _bytesSent;
   }

   unsigned long int Trigger::getCyclesExecuted() const
   {
      return _cyclesExecuted;
   }
  
   double Trigger::getFactor(_OpType optype) const
   {
      switch(optype)
      {
	 case INT:
	    return _config.getPeByTask(_ownerTask->getId())->getIntOps(); 
	    break;
	 case MEM: 
	    return _config.getPeByTask(_ownerTask->getId())->getMemOps(); 
	    break;
	 case FLOAT: 
	    return _config.getPeByTask(_ownerTask->getId())->getFloatOps(); 
	    break;
	 default: 
	    return 0.0;
      }
   }

   unsigned long int Trigger::getBufferUsage() const
   {
      return _bytesInBuffer;
   }

   const sc_core::sc_time& Trigger::getReceiveTime() const
   {
      return _currentReceive;
   }

}


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
