/**
 *
 * @file measure.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: measure.cc 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */


#include "measure.hh"

#ifdef SCTG_USE_EXECMON
#ifndef MTI_SYSTEMC
#include "tcp_server_if.hh"
#endif
#endif

#include <vector>
#include <iomanip>

namespace sctg
{

   Measure::Measure(sc_core::sc_module_name name, sctg::Configuration& config,
		    std::vector<sctg::ProcessingElement*> pes,
		    std::vector<sctg::Task*> tasks)
      : sc_core::sc_module(name),
	_config(config),
	_pes(pes),
	_tasks(tasks)
   {
      SC_THREAD(measureThread);
#ifdef SCTG_USE_EXECMON      
      if(_config.useExecMon() || _config.getExecMonStream())
      {
	 SC_THREAD(execMonThread);      
      }
#endif      
   }
  
  
   Measure::~Measure()
   {
    
   }
  
  
   void Measure::measureThread()
   { 
      sc_core::sc_time interval = _config.getMeasurementInterval();
      if(interval == sc_core::SC_ZERO_TIME)
      { return; }

      std::cout << "Measuring statistics with " << interval
		<< " interval" << std::endl;

      std::vector<PeMeasurements> oldPeData;
      std::vector<PeMeasurements> newPeData;
      std::vector<BufferMeasurements> oldBufferData;
      std::vector<BufferMeasurements> newBufferData;
      oldPeData.resize(_pes.size());
      newPeData.resize(_pes.size());
      oldBufferData.resize(_pes.size());
      newBufferData.resize(_pes.size());
    
      while(true)
      {
	 wait(interval);

	 std::cout << "Measuring. Current simulation time is "
		   << sc_core::sc_time_stamp() << std::endl;
	 if(_config.getPeStream())
	 {
	    **_config.getPeStream()
	       << "Measuring. Current simulation time is "
	       << sc_core::sc_time_stamp() << std::endl;
	 }
	 if(_config.getAppStream())
	 {
	    **_config.getAppStream()
	       << "Measuring. Current simulation time is "
	       << sc_core::sc_time_stamp() << std::endl;
	 }
	 
	
	 for(unsigned int i = 0; i < _pes.size(); ++i)
	 {
	    // Calculate tx/rx bytes
	    oldBufferData.at(i) = newBufferData.at(i);
	    newBufferData.at(i) = 
	       _config.getBufferByPe(_pes.at(i)->getId())->getMeasurements();

	    unsigned long int rxBytes = 
	       newBufferData.at(i).rxBytes - oldBufferData.at(i).rxBytes;
	    unsigned long int txBytes = 
	       newBufferData.at(i).txBytes - oldBufferData.at(i).txBytes;
	    unsigned long int intBytes = 
	       newBufferData.at(i).internalBytes - 
	       oldBufferData.at(i).internalBytes;

	    // Calculate utilization
	    oldPeData.at(i) = newPeData.at(i);
	    newPeData.at(i) = _pes.at(i)->getMeasurements();
	    sc_core::sc_time idle = 
	       newPeData.at(i).idleTime - oldPeData.at(i).idleTime;
	    sc_core::sc_time exec = 
	       newPeData.at(i).execTime - oldPeData.at(i).execTime;
	    /*
	    double idled = 
	       newPeData.at(i).idleCycles - oldPeData.at(i).idleCycles;
	    double execd = 
	       newPeData.at(i).execCycles - oldPeData.at(i).execCycles;
	    */
	    double util = (exec / (exec + idle));	    

	    if(_config.getPeStream())
	    {
	       **_config.getPeStream()
		  << "* PE " << _pes.at(i)->getName() << " utilization "
		  << std::setw(10) << std::left
		  << util << "; received " << std::setw(10) << std::right 
		  << rxBytes - intBytes << " bytes, sent " << std::setw(10) 
		  << std::right << txBytes 
		  << " bytes, currently rxBuffer has " 
		  << std::setw(10) << std::right 
		  << newBufferData.at(i).rxUsed 
		  << " bytes and txBuffer "
		  << std::setw(10) << std::right 
		  << newBufferData.at(i).txUsed << " bytes"
		  << std::endl;
	       (**_config.getPeStream()).flush();
	    }
	    /*std::cout << "  exec:" << std::setw(10) << std::right << execd
	      << " idled:" << std::setw(10) << std::right << idled
	      << " total:" << std::setw(10) << std::right 
	      << (idled + execd) << std::endl;
	      std::cout << "  exec: " << exec << " idle: " << idle
	      << std::endl;
	    */
	 }

	 if(_config.getAppStream())
	 {
	    for(unsigned int i = 0; i < _tasks.size(); ++i)
	    {
	       **_config.getAppStream()
		  << "* Task " << std::setw(16) << std::left
		  << _tasks.at(i)->getName() << " has "
		  << std::setw(10) << std::right 
		  << _tasks.at(i)->getBufferUsage()
		  << " unprocessed bytes in rx buffer, state is " 
		  << stateToString(_tasks.at(i)->getState())
		  << std::endl;
	       (**_config.getAppStream()).flush();
	    }
	 }	 
      }
      
   }

#ifdef SCTG_USE_EXECMON
   void Measure::execMonThread()
   {
      sc_core::sc_time interval = sc_core::sc_time(1, sc_core::SC_MS);
      unsigned long int currentTime = 0;

      std::ostringstream oss;

      std::vector<PeMeasurements> oldPeData;
      std::vector<PeMeasurements> newPeData;
      std::vector<BufferMeasurements> oldBufferData;
      std::vector<BufferMeasurements> newBufferData;
      oldPeData.resize(_pes.size());
      newPeData.resize(_pes.size());
      oldBufferData.resize(_pes.size());
      newBufferData.resize(_pes.size());
    
      while(true)
      {	 
	 oss.str("");
	 oss << "<measurement time=\"" << currentTime << "\">";
	 	
	 for(unsigned int i = 0; i < _pes.size(); ++i)
	 {
	    // Calculate tx/rx bytes
	    oldBufferData.at(i) = newBufferData.at(i);
	    newBufferData.at(i) = 
	       _config.getBufferByPe(_pes.at(i)->getId())->getMeasurements();

	    unsigned long int rxBytes = 
	       newBufferData.at(i).rxBytes - oldBufferData.at(i).rxBytes;
	    unsigned long int txBytes = 
	       newBufferData.at(i).txBytes - oldBufferData.at(i).txBytes;
	    unsigned long int intBytes = 
	       newBufferData.at(i).internalBytes - 
	       oldBufferData.at(i).internalBytes;

	    // Calculate utilization
	    oldPeData.at(i) = newPeData.at(i);
	    newPeData.at(i) = _pes.at(i)->getMeasurements();
	    sc_core::sc_time idle = 
	       newPeData.at(i).idleTime - oldPeData.at(i).idleTime;
	    sc_core::sc_time exec = 
	       newPeData.at(i).execTime - oldPeData.at(i).execTime;
	    /*
	    double idled = 
	       newPeData.at(i).idleCycles - oldPeData.at(i).idleCycles;
	    double execd = 
	       newPeData.at(i).execCycles - oldPeData.at(i).execCycles;
	    */
	    double util = (exec / (exec + idle));

	    if((exec + idle) == sc_core::SC_ZERO_TIME)
	    {
	       util = 0.0;
	    }

	    oss << "<graph cpu_id=\"" << _pes.at(i)->getId() << "\" value=\""
		<< (100.0 * util) << "\" id=\"0\"/>";

	 }

	 for(unsigned int i = 0; i < _pes.size(); ++i)
	 {
	    while(!(_pes.at(i)->getTokenQueue().empty()))
	    {
	       oss << "<signal " 
		   << "dest_pid=\"" << 
		  _config.getTaskByInPort(_pes.at(i)->getTokenQueue().front().destination)
		  ->getId() << "\" "
		   << "value=\"" << _pes.at(i)->getTokenQueue().front().size << "\" "
		   << "source_pid=\"";
	       unsigned long int src = _pes.at(i)->getTokenQueue().front().source;
	       for(unsigned int j = 0; j < _tasks.size(); ++j)
	       {
		  if(_tasks.at(j)->hasOutPort(src))
		  {
		     oss << _tasks.at(j)->getId() << "\"/>";
		     break;
		  }
	       }	       
	       _pes.at(i)->getTokenQueue().pop();
	    }
	 }

	 for(unsigned int i = 0; i < _tasks.size(); ++i)
	 {
	    double latency = _tasks.at(i)->getLastLatency() /
	       sc_core::sc_time(1.0, sc_core::SC_US);
	    double response = _tasks.at(i)->getLastResponse() /
	       sc_core::sc_time(1.0, sc_core::SC_US);
	    oss << "<process_exec " 
		<< "latency=\"" << latency << "\" "
		<< "local_comm_bytes=\"" << _tasks.at(i)->getLocalBytesSent() << "\" "
		<< "local_comm_time=\"" << 0 << "\" "
		<< "response_time=\"" << response << "\" "
		<< "remote_comm_bytes=\"" << _tasks.at(i)->getRemoteBytesSent() << "\" "
		<< "remote_comm_time=\"" << 0 << "\" "
		<< "count=\"" << _tasks.at(i)->getTimesTriggered() << "\" "
		<< "pid=\"" << _tasks.at(i)->getId() << "\" "
		<< "signal_queue=\"" << _tasks.at(i)->getBufferUsage() << "\" "
		<< "exec_time=\"" << _tasks.at(i)->getCyclesExecuted() << "\"/>";
	    	       
	 }

	 oss << "</measurement>" << std::endl;

	 std::string str = oss.str();
#ifndef MTI_SYSTEMC
	 if(_config.useExecMon())
	 {
	    _config.getTcpServer()->send(str);
	 }
#endif
	 if(_config.getExecMonStream())
	 {
	    **_config.getExecMonStream()
	       << oss.str();
	    (**_config.getExecMonStream()).flush();
	 }

	 wait(interval);
	 currentTime++;
      }
   }
#endif
   
}


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
