/**
 *
 * @file main.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: main.cc 1422 2010-09-08 11:08:40Z lehton87 $
 *
 */


#include "amount.hh"
#include "event.hh"
#include "configuration.hh"
#include "resource.hh"
#include "processing_element.hh"
#include "task.hh"
#include "trigger.hh"
#include "measure.hh"
#include "noc_factory.hh"

#include "cost_function.hh"

#ifdef SCTG_USE_EXECMON
#ifndef MTI_SYSTEMC
#include "non_sc_factory.hh"
#include "tcp_server.hh"
#include <boost/asio.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#endif
#endif


#include <cstdlib>
#include <iostream>
#include <iomanip>
#include <string>
#include <set>   
#include <exception>
#include <ctime> 
#include <vector>
#include <map> 

#include <boost/foreach.hpp>
#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <boost/random.hpp>
#include <boost/lexical_cast.hpp>

#ifndef MTI_SYSTEMC
#include <boost/program_options.hpp>
#endif

#include <systemc> 


int sc_main(int argc, char* argv[])
{    
   std::vector<sctg::ProcessingElement*> pes;
   std::vector<sctg::Task*>              tasks;
   std::vector<sctg::Event*>             events;      
   NocFactory*                           nocFactory = 0;
   sctg::Measure*                        measurements = 0;
   sctg::Configuration*                  conf = 0;
   
   sctg::CostFunction                    costFunctionParser;

   using boost::property_tree::ptree;
   ptree tgModel;
   ptree peLib;

   unsigned int playbackDelay = 0;

   enum ProgramMode {NORMAL, PLAYBACK, MIXEDSIM};
   ProgramMode programMode = NORMAL;

   bool useExecutionMonitor = false;

#ifdef SCTG_USE_EXECMON
#ifndef MTI_SYSTEMC
   sctg::NonScFactory* nonScFactory;
#endif
#endif      
  
   try
   {
      std::cout << std::endl
		<< "Transaction Generator 2" << std::endl;

      std::string inputFile;
      std::string playbackFile;
      std::string saveDir;

#ifndef MTI_SYSTEMC

      // Command line option parser using Boost.ProgramOptions
      
      namespace po = boost::program_options;
      po::options_description poDesc("Allowed options");
      poDesc.add_options()
	 ("help", "This message")
	 ("input-file,i", po::value<std::string>(), "Input file")
	 ("replay-file,r", po::value<std::string>(), 
	  "Recorded file to replay")
	 ("delay,d", po::value<unsigned int>(), 
	  "Delay (ms) to slow down replay")
	 ("save-directory,s", po::value<std::string>(), 
	  "Directory to save logs")
	 ("execution-monitor,e", "Use Execution Monitor");
      
      po::variables_map poVm;
      po::store(po::parse_command_line(argc, argv, poDesc), poVm);
      po::notify(poVm);
      
      if(poVm.count("help"))
      {
	 std::cout << poDesc << std::endl;
	 return EXIT_SUCCESS;
      }

      if(poVm.count("input-file"))
      {
	 std::cout << "input-file: "
		   << poVm["input-file"].as<std::string>()
		   << std::endl;
	 inputFile = poVm["input-file"].as<std::string>();
	 programMode = NORMAL;

	 if(poVm.count("execution-monitor"))
	 {
#ifndef SCTG_USE_EXECMON
	    std::cerr << "You need to compile Transaction Generator 2 "
		      << "with SCTG_USE_EXECMON flag to use Execution Monitor"
		      << std::endl;
	    return EXIT_FAILURE;
#endif	    
	    useExecutionMonitor = true;
	 }
      }
      else if(poVm.count("replay-file"))
      {
#ifndef SCTG_USE_EXECMON
	 std::cerr << "You need to compile Transaction Generator 2 "
		   << "with SCTG_USE_EXECMON flag to use Execution Monitor"
		   << std::endl;
	 return EXIT_FAILURE;
#endif
	 std::cout << "replay-file: "
		   << poVm["replay-file"].as<std::string>()
		   << std::endl;
	 playbackFile = poVm["replay-file"].as<std::string>();
	 programMode = PLAYBACK;
      }
      else
      {
	 std::cout << poDesc << std::endl;
	 return EXIT_FAILURE;
      }
      
      if(poVm.count("delay"))
      {
	 std::cout << "Delay for replay: "
		   << poVm["delay"].as<unsigned int>()
		   << " ms"
		   << std::endl;
	 playbackDelay = poVm["delay"].as<unsigned int>();
      }
      else
      {
	 std::cout << "Delay for playback: 1ms" << std::endl;
	 playbackDelay = 1;
      }

      if(poVm.count("save-directory"))
      {
	 std::cout << "Directory to save logs: "
		   << poVm["save-directory"].as<std::string>()
		   << std::endl;
	 saveDir = poVm["save-directory"].as<std::string>();
      }
      else
      {
	 std::cout << "Directory to save logs: ." << std::endl;
      }

#else // Commandline parser for Simulation with Modelsim

      // Parse input file name from parameters
      if(argc == 2)
      {
	 inputFile.assign(argv[1]);
      }
      else 
      {
	 std::cout << std::endl
		   << "Input file not set" << std::endl
		   << std::endl
		   << "Usage:  sctg <input-file>" << std::endl 
		   << std::endl;
	 return EXIT_FAILURE;
      }

      programMode = MIXEDSIM;

#endif      

#ifdef SCTG_USE_EXECMON
#ifndef MTI_SYSTEMC

      if(programMode == PLAYBACK)
      {
	 std::cout << "Replay mode" << std::endl;
	 sctg::TcpServer server(9990);

	 boost::asio::io_service io;
	 boost::asio::deadline_timer 
	    delay(io, boost::posix_time::milliseconds(playbackDelay));

	 std::ifstream inFile(playbackFile.c_str());
	 std::string line;

	 std::cout << "Waiting for Execution Monitor to connect" << std::endl;
	 server.waitConnection();
	 std::cout << "Execution Monitor connected" << std::endl;

	 while(std::getline(inFile, line))
	 {
	    server.send(line);
	    delay.wait();
	    delay.expires_at(delay.expires_at() + 
			     boost::posix_time::milliseconds(playbackDelay));
	 }
	 
	 std::cout << "Done" << std::endl;
	 return EXIT_SUCCESS;
      }

#endif
#endif

      
      // Read in the file defining application and platform model, mapping
      // constraints etc and store it to a property_tree      
      std::string tgFile(inputFile.c_str());      
      read_xml(tgFile, tgModel);
      
      // Parse constraints. This also sets sc_time_resolution
      ptree node = tgModel.get_child("system.constraints");
      conf = new sctg::Configuration(node, saveDir);

      conf->useExecMon(useExecutionMonitor);


#ifdef SCTG_USE_EXECMON
#ifndef MTI_SYSTEMC
      // Create NonScFactory
      nonScFactory = new sctg::NonScFactory(*conf);
#endif
#endif
      
      // Read in processing element library
      read_xml(tgModel.get<std::string>
	       ("system.constraints.pe_lib.<xmlattr>.file"), peLib);

      // Parse platform -> resource_list
      node = tgModel.get_child("system.platform.resource_list");
      for(ptree::const_iterator iter = node.begin(); iter != node.end(); 
	  ++iter)
      {
	 // Confirm this is a resource tag
	 if((*iter).first == "resource")
	 {
	    // Check that its class is "pe" or not defined
	    std::string resourceClass = (*iter).second.get<std::string>
	       ("<xmlattr>.class", "pe");
	    // Read in ID
	    unsigned long int id = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.id");
	    // Read name or give a default name of "<class> <id>"
	    std::string name = (*iter).second.get<std::string>
	       ("<xmlattr>.name", resourceClass + " " + 
		boost::lexical_cast<std::string>(id));	      
	      
	    if(resourceClass == "pe")
	    {
	       sctg::ProcessingElement* pe = new 
		  sctg::ProcessingElement(name.c_str(), 
					  (*iter).second, peLib, *conf);
	       conf->addPeMap(pe->getId(), pe);
	       pes.push_back(pe);
	    }
	    else if(resourceClass == "memory")
	    {
		  
	    }
	    else
	    {
	       std::string err = "Unknown resource class " + resourceClass;
	       throw std::runtime_error(err.c_str());
	    }
	 }
      }


      // Parse Tasks and task connections
      node = tgModel.get_child("system.application.task_graph");
      for(ptree::const_iterator iter = node.begin(); iter != node.end(); 
	  ++iter)
      {
	 if((*iter).first == "task")
	 {
	    sctg::Task* task = new sctg::Task((*iter).second, *conf);
	    conf->addTaskMap(task->getId(), task);
	    tasks.push_back(task);
	 }
	 else if((*iter).first == "task_connection")
	 {
	    unsigned long int src = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.src");
	    unsigned long int dst = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.dst");
	    conf->addTaskConnection(src, dst);
	 }
      }
      node = tgModel.get_child("system.application");
      for(ptree::const_iterator iter = node.begin(); iter != node.end(); 
	  ++iter)
      {
	 if((*iter).first == "task_connection")
	 {
	    unsigned long int src = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.src");
	    unsigned long int dst = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.dst");
	    conf->addTaskConnection(src, dst);
	 }
      }
      
      // Parse events
      node = tgModel.get_child("system.application.task_graph.event_list");
      for(ptree::const_iterator iter = node.begin(); iter != node.end(); 
	  ++iter)
      {
	 if((*iter).first == "event")
	 {
	    // Read in ID
	    unsigned long int id = (*iter).second.get<unsigned long int>
	       ("<xmlattr>.id");
	    // Read name or give a default name of "Event <id>"
	    std::string name = (*iter).second.get<std::string>
	       ("<xmlattr>.name", "Event " + 
		boost::lexical_cast<std::string>(id));	      
	    sctg::Event* event = new sctg::Event(name.c_str(),
						 *conf,
						 (*iter).second);	    
	    events.push_back(event);
	 }
      }


      // Map tasks to PEs
      node = tgModel.get_child("system.mapping");
      for(ptree::const_iterator iter = node.begin(); iter != node.end(); 
	  ++iter)
      {
	 if((*iter).first == "resource")
	 {
	    unsigned long int resourceId = 
	       (*iter).second.get<unsigned long int>("<xmlattr>.id");
	    // Check for sw_platform
	    ptree groupNode = 
	       (*iter).second.get_child("sw_platform", (*iter).second);
	    // If sw_platform is not found use resource instead

	    // Loop over all groups
	    for(ptree::const_iterator iter2 = groupNode.begin();
		iter2 != groupNode.end(); ++iter2)
	    {
	       if((*iter2).first == "group")
	       {
		  unsigned long int groupId = 
		     (*iter2).second.get<unsigned long int>("<xmlattr>.id");
		  for(ptree::const_iterator iter3 = (*iter2).second.begin();
		      iter3 != (*iter2).second.end(); ++iter3)
		  {		     
		     if((*iter3).first == "task")
		     {
			unsigned long int taskId = 
			   (*iter3).second.get<unsigned long int>
			   ("<xmlattr>.id");			
			conf->getPe(resourceId)->mapTask(conf->getTask(taskId));
			conf->addTaskToPeMap(taskId, conf->getPe(resourceId));
			conf->addTaskToGroup(taskId, groupId);
			conf->addGroupToPe(groupId, resourceId);
		     }
		  }
	       }
	    }

	 }
      }      

      // Map events to resources
      for(unsigned int i = 0; i < events.size(); ++i)
      {
	 for(unsigned int j = 0; j < pes.size(); ++j)
	 {	      
	    if(pes.at(j)->hasInPort
	       (conf->getDestination(events.at(i)->getOutPort())))
	    {	
	       events.at(i)->mapPe(pes.at(j));
	    }
	 }
      }
      
      // Connect network interfaces to resources
      for(unsigned int i = 0; i < pes.size(); ++i)
      {
	 unsigned long int terminalId = pes.at(i)->getTerminal();
	 unsigned long int portId = 0;
	 unsigned long int routerId = 0;
	 unsigned long int width = 0;
	 unsigned long int address = 0;
	 unsigned long int frequency = 0;
	 // Search router and its port id
	 node = tgModel.get_child("system.platform.noc.terminal_list");	 
	 for(ptree::const_iterator iter = node.begin(); iter != node.end(); 
	     ++iter)
	 {
	    if((*iter).first == "connection")
	    {
	       if((*iter).second.get<unsigned long int>("<xmlattr>.id") 
		  == terminalId)
	       {
		  portId = (*iter).second.get<unsigned long int>
		     ("<xmlattr>.port");
		  routerId = (*iter).second.get<unsigned long int>
		     ("<xmlattr>.router");
	       }
	    }	    
	 }
	 node = tgModel.get_child("system.platform.noc.router_list");
	 for(ptree::const_iterator iter = node.begin(); iter != node.end(); 
	     ++iter)
	 {
	    // Search for correct router
	    if((*iter).first == "router" && 
	       (*iter).second.get<unsigned long int>("<xmlattr>.id") ==
	       routerId)
	    {
	       width = (*iter).second.get<unsigned long int>
		  ("<xmlattr>.width");
	       // Search correct port
	       for(ptree::const_iterator iter2 = (*iter).second.begin();
		   iter2 != (*iter).second.end(); ++iter2)
	       {		  
		  if((*iter2).first == "port" &&
		     (*iter2).second.get<unsigned long int>("<xmlattr>.id") ==
		     portId)
		  {		     
		     std::string addrStr = 
			(*iter2).second.get<std::string>
			("<xmlattr>.address");		     
		     std::istringstream iss(addrStr);
		     iss >> std::hex >> address;
		     break;
		  }
	       }
	       break;
	    }
	 }
	 pes.at(i)->setTerminal(terminalId, address, width);

	 std::cout << "Found address 0x" << std::hex << address
		   << std::dec << " for PE " << pes.at(i)->getName()
		   << " (terminal:" << terminalId << " router:"
		   << routerId << " port:" << portId << ")" << std::endl;
      }

      
      // Create NoC
      std::string nocClass;
      std::string nocType;
      std::string nocSubType;
      node = tgModel.get_child("system.platform.noc");	 
      nocClass   = node.get<std::string>("<xmlattr>.class", "UNKNOWN");
      nocType    = node.get<std::string>("<xmlattr>.type", "UNKNOWN");
      nocSubType = node.get<std::string>("<xmlattr>.subtype", "UNKNOWN");

      conf->setNocClass(nocClass);
      conf->setNocType(nocType);
      conf->setNocSubType(nocSubType); 
      
      nocFactory = new NocFactory(conf);

      // Create measurement thread
      measurements = new sctg::Measure("measurer", *conf, pes, tasks);

      
#ifdef SCTG_USE_EXECMON
      
      if(conf->useExecMon())
      {

#ifndef MTI_SYSTEMC
	 std::cout << "Waiting for Execution Monitor to connect" << std::endl;
	 conf->getTcpServer()->waitConnection();
	 std::cout << "Execution Monitor connected" << std::endl;
#endif
	 // Configure Execution Monitor
      
	 std::ostringstream oss;      
	 std::string str;
	 oss.str("");
	 oss << "<configuration>";
	 for(std::map<unsigned long int, 
		      sctg::ProcessingElement*>::iterator cpuIter = 
		conf->getPeBegin(); cpuIter != conf->getPeEnd(); ++cpuIter)
	 {
	    std::string cpuName = (*cpuIter).second->getName();
	    unsigned long int cpuId = (*cpuIter).first;
	    oss << "<cpu name=\"" << cpuName
		<< "\" id=\"" << cpuId << "\">";
	    for(std::map<unsigned long int, 
			 unsigned long int>::iterator groupIter = 
		   conf->getGroupPeBegin(); groupIter != conf->getGroupPeEnd();
		++groupIter)
	    {
	       unsigned long int group = (*groupIter).first;
	       unsigned long int pe    = (*groupIter).second;
	       if(cpuId == pe)
	       {
		  oss << "<thread name=\"Thread_" << group << "\" id=\"" 
		      << group << "\" " << "priority=\"0\">";
		  for(std::map<unsigned long int, unsigned long int>::iterator
			 taskIter = conf->getTaskGroupBegin(); 
		      taskIter != conf->getTaskGroupEnd(); ++taskIter)
		  {
		     unsigned long int task = (*taskIter).first;
		     unsigned long int gr   = (*taskIter).second;
		     if(group == gr)
		     {
			oss << "<process name=\"" 
			    << conf->getTask(task)->getName()
			    << "\" id=\"" << task << "\"/>";
		     }
		  }

		  oss << "</thread>";
	       }
	    }
	    oss << "</cpu>";
	 }      

	 oss << "</configuration>" << std::endl;
     
	 str = oss.str();
#ifndef MTI_SYSTEMC
	 conf->getTcpServer()->send(str);
#endif
	 if(conf->getExecMonStream())
	 {
	    **conf->getExecMonStream()
	       << oss.str();
	 }

      }

#endif // USE_EXECMON     
      


      // RUN THE SIMULATION
      std::cout << "Starting simulation with seed " 
		<< conf->getSeed() << std::endl;

#ifdef MTI_SYSTEMC
      sc_core::sc_start();
#else
      sc_core::sc_start(conf->getSimLength());
      sc_core::sc_start(sc_core::SC_ZERO_TIME);
#endif

      std::cout << "Simulation ends at " 
		<< sc_core::sc_time_stamp() << std::endl;
      

      // Add variables for cost function parser
      std::ostringstream oss;
      std::string str;
      double avgPeUtil = 0.0;
      for(unsigned int i = 0; i < pes.size(); ++i)
      {
	 oss.str("");
	 oss << "pu_" << pes.at(i)->getId();
	 double util = pes.at(i)->getAvgUtilization();
	 avgPeUtil += util;
	 str = oss.str();
	 costFunctionParser.addVariable(str, util);
	 oss.str("");
	 oss << "pu_" << pes.at(i)->getName();
	 str = oss.str();
	 costFunctionParser.addVariable(str, util);
	 oss.str("");
	 oss << "pf_" << pes.at(i)->getId();
	 str = oss.str();
	 costFunctionParser.addVariable(str, pes.at(i)->getFrequency());
      }
      str = "pu_avg";
      avgPeUtil /= pes.size();
      costFunctionParser.addVariable(str, avgPeUtil);

      double totalCount = 0.0;
      for(unsigned int i = 0; i < tasks.size(); ++i)
      {
	 oss.str("");
	 oss << "tc_" << tasks.at(i)->getId();
	 double count = tasks.at(i)->getTimesTriggered();
	 totalCount += count;
	 str = oss.str();
	 costFunctionParser.addVariable(str, count);
	 oss.str("");
	 oss << "tc_" << tasks.at(i)->getName();
	 str = oss.str();
	 costFunctionParser.addVariable(str, count);
      }
      str = "tc_tot";
      costFunctionParser.addVariable(str, totalCount);

      totalCount = 0.0;
      for(unsigned int i = 0; i < events.size(); ++i)
      {
	 oss.str("");
	 oss << "ec_" << events.at(i)->getId();
	 double count = events.at(i)->getTimesHappened();
	 totalCount += count;
	 str = oss.str();
	 costFunctionParser.addVariable(str, count);
	 oss.str("");
	 oss << "ec_" << events.at(i)->getName();
	 str = oss.str();
	 costFunctionParser.addVariable(str, count);
      }
      str = "ec_tot";
      costFunctionParser.addVariable(str, totalCount);

      std::map<unsigned long int, 
	       std::map<unsigned long int, sc_core::sc_time> >::iterator outer;
      std::map<unsigned long int, sc_core::sc_time>::iterator inner;

      unsigned long int latcount = 0;
      double latavg = 0.0;

      for(outer = conf->getTokenLatency().begin(); 
	  outer != conf->getTokenLatency().end(); ++outer)
      {
	 for(inner = (*outer).second.begin(); 
	     inner != (*outer).second.end(); ++inner)
	 {
	    oss.str("");
	    oss << "latf_" << (*outer).first << "_" << (*inner).first << "_avg";
	    double val = 
	       ((*inner).second / sc_core::sc_time(1000.0, sc_core::SC_MS)) 
	       / conf->getTokenCount()[(*outer).first][(*inner).first];
	    latcount++;
	    latavg += val;
	    str = oss.str();
	    costFunctionParser.addVariable(str, val);
	    oss.str("");
	    oss << "latf_" << (*outer).first << "_" << (*inner).first << "_max";
	    val = conf->getTokenLatencyMax()[(*outer).first][(*inner).first] / 
	       sc_core::sc_time(1000.0, sc_core::SC_MS);
	    str = oss.str();
	    costFunctionParser.addVariable(str, val);
	    oss.str("");
	    oss << "latf_" << (*outer).first << "_" << (*inner).first << "_min";
	    val = conf->getTokenLatencyMin()[(*outer).first][(*inner).first] / 
	       sc_core::sc_time(1000.0, sc_core::SC_MS);
	    str = oss.str();
	    costFunctionParser.addVariable(str, val);
	 }
      }
      
      oss.str("");
      oss << "latf_tot_avg";
      str = oss.str();
      latavg /= latcount;
      costFunctionParser.addVariable(str, latavg);
     
      // Calculate latencies with unsent tokens

      latcount = 0;
      latavg = 0.0;

      for(unsigned int i = 0; i < pes.size(); ++i)
      {
	 pes.at(i)->updateUnfinishedTokensLatency();
      }

      for(outer = conf->getTokenLatency().begin(); 
	  outer != conf->getTokenLatency().end(); ++outer)
      {
	 for(inner = (*outer).second.begin(); 
	     inner != (*outer).second.end(); ++inner)
	 {
	    oss.str("");
	    oss << "lat_" << (*outer).first << "_" << (*inner).first << "_avg";
	    double val = 
	       ((*inner).second / sc_core::sc_time(1000.0, sc_core::SC_MS)) 
	       / conf->getTokenCount()[(*outer).first][(*inner).first];
	    latcount++;
	    latavg += val;
	    str = oss.str();
	    costFunctionParser.addVariable(str, val);
	    oss.str("");
	    oss << "lat_" << (*outer).first << "_" << (*inner).first << "_max";
	    val = conf->getTokenLatencyMax()[(*outer).first][(*inner).first] / 
	       sc_core::sc_time(1000.0, sc_core::SC_MS);
	    str = oss.str();
	    costFunctionParser.addVariable(str, val);
	    oss.str("");
	    oss << "lat_" << (*outer).first << "_" << (*inner).first << "_min";
	    val = conf->getTokenLatencyMin()[(*outer).first][(*inner).first] / 
	       sc_core::sc_time(1000.0, sc_core::SC_MS);
	    str = oss.str();
	    costFunctionParser.addVariable(str, val);
	 }
      }
      
      oss.str("");
      oss << "lat_tot_avg";
      str = oss.str();
      latavg /= latcount;
      costFunctionParser.addVariable(str, latavg);

      // Add path measurements
      for(std::map<std::string, sc_core::sc_time>::iterator iter =
	     conf->getTotPathLat().begin(); iter != conf->getTotPathLat().end();
	  ++ iter )
      {
	 oss.str("");
	 oss << "path_" << (*iter).first << "_avg";
	 str = oss.str();
	 double val = ((*iter).second / sc_core::sc_time(1000.0, sc_core::SC_MS)) /
	    conf->getPathCount()[(*iter).first];
	 costFunctionParser.addVariable(str, val);
	 oss.str("");
	 oss << "path_" << (*iter).first << "_count";
	 str = oss.str();
	 val = conf->getPathCount()[(*iter).first];
	 costFunctionParser.addVariable(str, val);
      }
      for(std::map<std::string, sc_core::sc_time>::iterator iter =
	     conf->getMaxPathLat().begin(); iter != conf->getMaxPathLat().end();
	  ++ iter )
      {
	 oss.str("");
	 oss << "path_" << (*iter).first << "_max";
	 str = oss.str();
	 double val = ((*iter).second / sc_core::sc_time(1000.0, sc_core::SC_MS));
	 costFunctionParser.addVariable(str, val);
      }
      for(std::map<std::string, sc_core::sc_time>::iterator iter =
	     conf->getMinPathLat().begin(); iter != conf->getMinPathLat().end();
	  ++ iter )
      {
	 oss.str("");
	 oss << "path_" << (*iter).first << "_min";
	 str = oss.str();
	 double val = ((*iter).second / sc_core::sc_time(1000.0, sc_core::SC_MS));
	 costFunctionParser.addVariable(str, val);
      }

      // Add misc cost function variables
      for(std::map<std::string, double>::iterator iter = 
	     conf->getCostVariables().begin(); 
	  iter != conf->getCostVariables().end(); ++iter)
      {
	 std::string ss((*iter).first);
	 double dd = (*iter).second;
	 costFunctionParser.addVariable(ss, dd);
      }

      // Evaluate cost functions
      unsigned int n = 1;
      for(std::vector<std::string>::iterator iter = 
	     conf->getCostFunctions().begin(); 
	  iter != conf->getCostFunctions().end(); ++iter)
      {
	 double answer = costFunctionParser.parse((*iter));
	 std::cout << std::setiosflags(std::ios::left)
		   << "Cost " << std::setw(3) 
		   << n << " ; " << std::setprecision(16) 
		   << std::setw(22) << answer
		   << std::resetiosflags(std::ios::left)
	    //<< std::resetiosflags(std::ios::fixed)
	    //	   << std::setiosflags(std::ios::scientific)
	    //	   << " ; " << std::setprecision(16) 
	    //	   << std::setw(26) << answer
		   << "   ; \"" << (*iter)
		   << "\"" << std::endl;
	 if(n % 5 == 0) {std::cout << std::endl;}
	 if(conf->getSummaryStream())
	 {
	    **(conf->getSummaryStream())
	       << std::setiosflags(std::ios::fixed)
	       << "Cost " << std::setw(3) 
	       << n << " ; " << std::setprecision(16) 
	       << std::setw(26) << answer
	       << std::resetiosflags(std::ios::fixed)
	       << std::setiosflags(std::ios::scientific)
	       << " ; " << std::setprecision(16) 
	       << std::setw(26) << answer
	       << "   ; \"" << (*iter)
	       << "\"" << std::endl;    
	 }
	 ++n;
      }

      
   }
   catch(std::exception& e)
   {
      std::cerr << "EXCEPTION: "
		<< e.what() << std::endl;
   }
   catch(...)
   {
      std::cerr << "EXCEPTION: unknown source (not std::exception)."
		<< std::endl;
   }
  
   // CLEAN UP
  
   // Destroy all processing elements
   for(unsigned int i = 0; i < pes.size(); ++i)
   {
      delete pes.at(i);
      pes.at(i) = 0;
   }
  
   // Destroy all tasks
   for(unsigned int i = 0; i < tasks.size(); ++i)
   {
      delete tasks.at(i);
      tasks.at(i) = 0;
   }
  
   // Destroy all events
   for(unsigned int i = 0; i < events.size(); ++i)
   {
      delete events.at(i);
      events.at(i) = 0;
   }  

   delete nocFactory; nocFactory = 0;
   delete measurements; measurements = 0;

#ifdef SCTG_USE_EXECMON
#ifndef MTI_SYSTEMC
   // Destroy NonScFactory
   delete nonScFactory; nonScFactory = 0;
#endif
#endif
   
   // Destroy configuration
   delete conf; conf = 0;

   return EXIT_SUCCESS;
}


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
