/**
 *
 * @file trigger.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: trigger.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */


#ifndef SCTG_TRIGGER_HH
#define SCTG_TRIGGER_HH

#include "amount.hh"
#include "configuration.hh"
#include "tg_token.hh"
#include "task.hh"
#include "common.hh"

#include <boost/property_tree/ptree.hpp>
#include <vector>
#include <queue>

#include <systemc>

namespace sctg
{
  class Task;

  class Trigger
  {
  public:
        
    //* Constructor
    Trigger(const boost::property_tree::ptree& pt,
	    Configuration& config,
	    Task* ownerTask);
    
    //* Destructor
    virtual ~Trigger();

    //* Returns current operation
    sctg::OperationType getOperation();

    //* True if its new operation (not yet partially consumed)
    bool isNewOperation();

    //* Returns current amount what is left from current operation
    unsigned long int getAmount();

    //* Return current operation's out_port_id
    unsigned long int getOutPort();

    //* Consumes amount of operations (or bytes for SENDs)
    //* @return How many operations are left
    unsigned long int consume(unsigned long int amount);

    //* True if trigger has in_port with that id
    bool hasInPort(unsigned long int id);
       
    //* Trigger receives a token
    void receive(tgToken token);

    //* True if trigger is triggered
    bool isActive();

    //* How many times this trigger has been triggered (executed)
    unsigned long int getTimesTriggered() const;

    //* How many bytes trigger has sent
    unsigned long int getBytesSent() const;

    //* How many clock cycles trigger has executed on PE
    unsigned long int getCyclesExecuted() const;

    //* Returns buffered bytes
    unsigned long int getBufferUsage() const;

     //* Returns Receive time of the current token
     //* (or last for AND dependency trigger)
     const sc_core::sc_time& getReceiveTime() const;

  private:
    
    enum _DependenceType {AND, OR};
    enum _OpType {INT, MEM, FLOAT};
    
    struct _OperationList
    {
      sctg::OperationType type;
      unsigned long int outPort;
      unsigned long int amount;
    };
        
    struct _Operation
    {
      sctg::OperationType     type;
      unsigned long int outPort;
      double            prob;
      std::vector<_OpType> factors;
       std::vector<Amount<double>* > amounts;      //uli
    };

    struct _ExecCount
    {
      std::vector<_Operation*> operations;
      boost::optional<unsigned long int> period;
      boost::optional<unsigned long int> min;
      boost::optional<unsigned long int> max;      
      sctg::State nextState;
    };
   
    // Measured information
    unsigned long int _triggered; // How many times this has been triggered 
    unsigned long int _bytesSent;
    unsigned long int _cyclesExecuted;


    unsigned long int _receivedBytes; // Current received bytes
    unsigned long int _eventBytes; // How many came from event
    Configuration& _config;
    _DependenceType _dependence;
    std::vector<_ExecCount*> _execCounts;
    std::vector<unsigned long int> _inPorts;
    std::queue<_OperationList> _operationQueue;
    std::vector<std::queue<tgToken> > _receivedPackets;

    Task*  _ownerTask;
    sctg::State  _nextState;
    
    bool _newOp;

    unsigned long int _bytesInBuffer;

    //* Builds execution queue 
    void buildQueue();
    
    //* Parses exec_count tag
    void parseExecCount(const boost::property_tree::ptree& pt,
			_ExecCount* execCount,
			Configuration& config);

    //* Resolves whether or not trigger should be active and builds queue
    void resolve();

    double getFactor(_OpType optype) const;

    /*double _intFactor;
    double _memFactor;
    double _floatFactor;*/

     sc_core::sc_time _currentReceive;

  };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
