/**
 *
 * @file packet_encoder_decoder.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: packet_encoder_decoder.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */


#ifndef ASEBT_SC_RTL_1_PACKET_ENCODER_DECODER_HH
#define ASEBT_SC_RTL_1_PACKET_ENCODER_DECODER_HH

#include "packet_encoder_ctrl.hh"
#include "packet_decoder_ctrl.hh"
#include "sc_rtl_1/fifo.hh"

#include <systemc>

namespace asebt
{
namespace sc_rtl_1
{


   template<int wait_empty_fifo_g = 0,
	    int data_width_g      = 36,
	    int addr_width_g      = 32,
	    int tx_len_width_g    = 8,
	    int packet_length_g   = 3,
	    int timeout_g         = 0,
	    int fill_packet_g     = 0,
	    int lut_en_g          = 1,
	    int net_type_g        = 1,
	    int len_flit_en_g     = 1,
	    int oaddr_flit_en_g   = 1,
	    int status_en_g       = 0>
   class packet_encoder_decoder : public sc_core::sc_module
   {
   public:
      
      sc_core::sc_in_clk                clk;
      sc_core::sc_in<sc_dt::sc_logic> rst_n;

      // Signals between IP and encoder
      sc_core::sc_in<sc_dt::sc_logic>               av_ip_enc_in;
      sc_core::sc_in<sc_dt::sc_lv<data_width_g> >   data_ip_enc_in;
      sc_core::sc_in<sc_dt::sc_logic>               we_ip_enc_in;
      sc_core::sc_in<sc_dt::sc_lv<tx_len_width_g> > len_ip_enc_in;
      sc_core::sc_out<sc_dt::sc_logic>              full_enc_ip_out;
      sc_core::sc_out<sc_dt::sc_logic>              empty_enc_ip_out;

      // Signals between NoC and encoder
      sc_core::sc_out<sc_dt::sc_logic>             av_enc_net_out;
      sc_core::sc_out<sc_dt::sc_lv<data_width_g> > data_enc_net_out;
      sc_core::sc_out<sc_dt::sc_logic>             we_enc_net_out;
      sc_core::sc_in<sc_dt::sc_logic>              full_net_enc_in;
      sc_core::sc_in<sc_dt::sc_logic>              empty_net_enc_in;

      // Signals between NoC and decoder
      sc_core::sc_in<sc_dt::sc_lv<data_width_g> > data_net_dec_in;
      sc_core::sc_in<sc_dt::sc_logic>             empty_net_dec_in;
      sc_core::sc_out<sc_dt::sc_logic>            re_dec_net_out;

      // Signals between IP and decoder
      sc_core::sc_out<sc_dt::sc_logic>             av_dec_ip_out;
      sc_core::sc_out<sc_dt::sc_lv<data_width_g> > data_dec_ip_out;
      sc_core::sc_in<sc_dt::sc_logic>              re_ip_dec_in;
      sc_core::sc_out<sc_dt::sc_logic>             empty_dec_ip_out;

      SC_HAS_PROCESS(packet_encoder_decoder);

      packet_encoder_decoder(sc_core::sc_module_name name)
	 : sc_core::sc_module(name),
	   clk("clk"),
	   rst_n("rst_n"),
	   av_ip_enc_in("av_ip_enc_in"),
	   data_ip_enc_in("data_ip_enc_in"),
	   we_ip_enc_in("we_ip_enc_in"),
	   len_ip_enc_in("len_ip_enc_in"),
	   full_enc_ip_out("full_enc_ip_out"),
	   empty_enc_ip_out("empty_enc_ip_out"),
	   av_enc_net_out("av_enc_net_out"),
	   data_enc_net_out("data_enc_net_out"),
	   we_enc_net_out("we_enc_net_out"),
	   full_net_enc_in("full_enc_net_in"),
	   empty_net_enc_in("empty_net_enc_in"),
	   data_net_dec_in("data_net_dec_in"),
	   empty_net_dec_in("empty_net_dec_in"),
	   re_dec_net_out("re_dec_net_out"),
	   av_dec_ip_out("av_dec_ip_out"),
	   data_dec_ip_out("data_dec_ip_out"),
	   re_ip_dec_in("re_ip_dec_in"),
	   empty_dec_ip_out("empty_dec_ip_out"),
	   d_av_to_encfifo("d_av_to_encfifo"),
	   d_av_from_encfifo("d_av_from_encfifo"),
	   full_from_encfifo("full_from_encfifo"),
	   empty_from_encfifo("empty_from_encfifo"),
	   we_to_encfifo("we_to_encfifo"),
	   av_fifo_enc("av_fifo_enc"),
	   data_fifo_enc("data_fifo_enc"),
	   re_enc_fifo("re_enc_fifo"),
	   stall_from_enc("stall_from_enc"),
	   d_av_to_decfifo("de_av_to_decfifo"),
	   d_av_from_decfifo("de_av_from_decfifo"),
	   av_dec_fifo("av_dec_fifo"),
	   data_dec_fifo("data_dec_fifo"),
	   we_dec_fifo("we_dec_fifo"),
	   full_fifo_dec("full_fifo_dec"),
	   encode_control("encode_control"),
	   decode_control("decode_control"),
	   encode_fifo("encode_fifo"),
	   decode_fifo("decode_fifo"),
	   enc_one_d("enc_one_d"),
	   dec_one_d("dec_one_d"),
	   enc_one_p("enc_one_p"),
	   dec_one_p("dec_one_p")
      {

	 SC_METHOD(davtodecfifo);
	 sensitive << data_dec_fifo << av_dec_fifo;

	 SC_METHOD(davfifoenc);
	 sensitive << d_av_from_encfifo;

	 SC_METHOD(davtoenc);
	 sensitive << data_ip_enc_in << av_ip_enc_in;

	 SC_METHOD(weenc);
	 sensitive << we_ip_enc_in << stall_from_enc;

	 SC_METHOD(davdecipout);
	 sensitive << d_av_from_decfifo;

	 SC_METHOD(fullencipout);
	 sensitive << stall_from_enc << full_from_encfifo;

	 SC_METHOD(emptyencipout);
	 sensitive << empty_from_encfifo;	 	 

	 encode_control.clk(clk);
	 encode_control.rst_n(rst_n);

	 encode_control.ip_av_in(av_ip_enc_in);
	 encode_control.ip_data_in(data_ip_enc_in);
	 encode_control.ip_we_in(we_ip_enc_in);
	 encode_control.ip_tx_len_in(len_ip_enc_in);
	 encode_control.ip_stall_out(stall_from_enc);

	 encode_control.fifo_av_in(av_fifo_enc);
	 encode_control.fifo_data_in(data_fifo_enc);
	 encode_control.fifo_full_in(full_from_encfifo);
	 encode_control.fifo_empty_in(empty_from_encfifo);
	 encode_control.fifo_re_out(re_enc_fifo);

	 encode_control.net_av_out(av_enc_net_out);
	 encode_control.net_data_out(data_enc_net_out);
	 encode_control.net_we_out(we_enc_net_out);
	 encode_control.net_empty_in(empty_net_enc_in);
	 encode_control.net_full_in(full_net_enc_in);

	 decode_control.clk(clk);
	 decode_control.rst_n(rst_n);
	 
	 decode_control.net_data_in(data_net_dec_in);
	 decode_control.net_empty_in(empty_net_dec_in);
	 decode_control.net_re_out(re_dec_net_out);
	 
	 decode_control.fifo_av_out(av_dec_fifo);
	 decode_control.fifo_data_out(data_dec_fifo);
	 decode_control.fifo_we_out(we_dec_fifo);
	 decode_control.fifo_full_in(full_fifo_dec);

	 encode_fifo.clk(clk);
	 encode_fifo.rst_n(rst_n);
	 encode_fifo.data_in(d_av_to_encfifo);
	 encode_fifo.we_in(we_to_encfifo);
	 encode_fifo.full_out(full_from_encfifo);
	 encode_fifo.data_out(d_av_from_encfifo);
	 encode_fifo.re_in(re_enc_fifo);
	 encode_fifo.empty_out(empty_from_encfifo);

	 encode_fifo.one_p_out(enc_one_p);
	 encode_fifo.one_d_out(enc_one_d);

	 decode_fifo.clk(clk);
	 decode_fifo.rst_n(rst_n);
	 decode_fifo.data_in(d_av_to_decfifo);
	 decode_fifo.we_in(we_dec_fifo);
	 decode_fifo.full_out(full_fifo_dec);
	 decode_fifo.data_out(d_av_from_decfifo);
	 decode_fifo.re_in(re_ip_dec_in);
	 decode_fifo.empty_out(empty_dec_ip_out);

	 decode_fifo.one_p_out(dec_one_p);
	 decode_fifo.one_d_out(dec_one_d);

      }

      ~packet_encoder_decoder()
      {
      }

   private:

      void davtodecfifo()
      {
	 d_av_to_decfifo.write((data_dec_fifo.read() , av_dec_fifo.read()));
      }

      void davfifoenc()
      {
	 data_fifo_enc.write(d_av_from_encfifo.read().range(data_width_g, 1));
	 av_fifo_enc.write(d_av_from_encfifo.read()[0]);
      }

      void davtoenc()
      {
	 d_av_to_encfifo.write((data_ip_enc_in.read() , av_ip_enc_in.read()));
      }

      void weenc()
      {
	 we_to_encfifo.write(we_ip_enc_in.read() & (~stall_from_enc.read()));
      }

      void davdecipout()
      {
	 av_dec_ip_out.write(d_av_from_decfifo.read()[0]);
	 data_dec_ip_out.write(d_av_from_decfifo.read().range(data_width_g, 1));
      }

      void fullencipout()
      {
	 full_enc_ip_out
	    .write(stall_from_enc.read() | full_from_encfifo.read());
      }

      void emptyencipout()
      {
	 empty_enc_ip_out.write(empty_from_encfifo.read());
      }
      
      sc_core::sc_signal<sc_dt::sc_lv<data_width_g+1> > d_av_to_encfifo;
      sc_core::sc_signal<sc_dt::sc_lv<data_width_g+1> > d_av_from_encfifo;
      
      sc_core::sc_signal<sc_dt::sc_logic> full_from_encfifo;
      sc_core::sc_signal<sc_dt::sc_logic> empty_from_encfifo;
      sc_core::sc_signal<sc_dt::sc_logic> we_to_encfifo;

      sc_core::sc_signal<sc_dt::sc_logic>             av_fifo_enc;
      sc_core::sc_signal<sc_dt::sc_lv<data_width_g> > data_fifo_enc;
      sc_core::sc_signal<sc_dt::sc_logic>             re_enc_fifo;
      sc_core::sc_signal<sc_dt::sc_logic>             stall_from_enc;

      sc_core::sc_signal<sc_dt::sc_lv<data_width_g+1> > d_av_to_decfifo;
      sc_core::sc_signal<sc_dt::sc_lv<data_width_g+1> > d_av_from_decfifo;

      sc_core::sc_signal<sc_dt::sc_logic>             av_dec_fifo;
      sc_core::sc_signal<sc_dt::sc_lv<data_width_g> > data_dec_fifo;
      sc_core::sc_signal<sc_dt::sc_logic>             we_dec_fifo;
      sc_core::sc_signal<sc_dt::sc_logic>             full_fifo_dec;

      packet_encoder_ctrl<wait_empty_fifo_g,
			  data_width_g,
			  addr_width_g,
			  tx_len_width_g,
			  packet_length_g,
			  timeout_g,
			  fill_packet_g,
			  lut_en_g,
			  net_type_g,
			  len_flit_en_g,
			  oaddr_flit_en_g,
			  status_en_g> encode_control;

      packet_decoder_ctrl<data_width_g,
			  addr_width_g,
			  packet_length_g,
			  fill_packet_g,
			  len_flit_en_g,
			  oaddr_flit_en_g> decode_control;

      fifo<data_width_g+1, packet_length_g> encode_fifo;
      
      fifo<data_width_g+1, packet_length_g> decode_fifo;

      sc_core::sc_signal<sc_dt::sc_logic> enc_one_d;
      sc_core::sc_signal<sc_dt::sc_logic> dec_one_d;
      sc_core::sc_signal<sc_dt::sc_logic> enc_one_p;
      sc_core::sc_signal<sc_dt::sc_logic> dec_one_p;

   };
}
}

#endif

// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
