/**
 *
 * @file ocp_tl3_mesh_router.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: ocp_tl3_mesh_router.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef SCTG_OCPTL3_MESH_ROUTER_HH
#define SCTG_OCPTL3_MESH_ROUTER_HH


#include <tlm.h>
#include <ocpip.h>

#ifdef MTI_SYSTEMC
#include "peq_with_get.h"
#else
#include "tlm_utils/peq_with_get.h"
#endif

#include <systemc>
#include <stdexcept>
#include <iostream>
#include <queue>

namespace sctg
{
   namespace mesh_2d_sc_ocp_tl3_1
   {
      
      template<unsigned int data_width_g>
      class OcpTl3MeshRouter : public sc_core::sc_module
      {
      public:
	 
	 SC_HAS_PROCESS(OcpTl3MeshRouter);

	 // Interface sockets: N, W, S, E, IP
	 ocpip::ocp_master_socket_tl3<data_width_g, 1>* masterSocket[5];
	 ocpip::ocp_slave_socket_tl3<data_width_g, 1>*  slaveSocket[5];

	 
	 //* Constructor
	 OcpTl3MeshRouter(sc_core::sc_module_name name,
			  unsigned int row,
			  unsigned int col,
			  unsigned int n_rows,
			  unsigned int n_cols_g,
			  sc_core::sc_time cycleLength)
	    : sc_module(name),
	      _row(row),
	      _col(col),
	      _cycleLength(cycleLength)
	 {

	    // OCP-IP parameters
	    _ocpParameters.byteen = false;

	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       for(unsigned int j = 0; j < 5; ++j)
	       {
		  std::ostringstream oss;
		  oss << "tlm_router_peq_" << i << "_" << j;
		  _peq[i][j] = 
		     new tlm_utils::peq_with_get<tlm::tlm_generic_payload>
		     (oss.str().c_str());
	       }
	    }

	    // Create IP side sockets

	    std::ostringstream oss;
	    oss << "router_" << _row << "_" << _col << "_ip_master_socket";
	    masterSocket[4] = new ocpip::ocp_master_socket_tl3<data_width_g, 1>
	       (oss.str().c_str(), 
		ocpip::ocp_master_socket_tl3<data_width_g, 1>::mm_txn_with_data());
	    
	    oss.str("");
	    oss << "router_" << _row << "_" << _col << "_ip_slave_socket";
	    slaveSocket[4] = new ocpip::ocp_slave_socket_tl3<data_width_g, 1>
	       (oss.str().c_str());

	    masterSocket[4]->set_ocp_config(_ocpParameters);
	    slaveSocket[4]->set_ocp_config(_ocpParameters);

	    // Create North sockets
	    if(_row != 0)
	    {
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_n_master_socket";
	       masterSocket[0] = 
		  new ocpip::ocp_master_socket_tl3<data_width_g, 1>
		  (oss.str().c_str(), 
		   ocpip::ocp_master_socket_tl3<data_width_g, 1>::mm_txn_with_data());
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_n_slave_socket";
	       slaveSocket[0]  = 
		  new ocpip::ocp_slave_socket_tl3<data_width_g, 1>
		  (oss.str().c_str());
	       masterSocket[0]->set_ocp_config(_ocpParameters);
	       slaveSocket[0]->set_ocp_config(_ocpParameters);
	    }
	    else
	    {
	       masterSocket[0] = 0;
	       slaveSocket[0]  = 0;
	    }

	    // Create South sockets
	    if(_row != (n_rows - 1))
	    {
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_w_master_socket";
	       masterSocket[2] = 
		  new ocpip::ocp_master_socket_tl3<data_width_g, 1>
		  (oss.str().c_str(), 
		   ocpip::ocp_master_socket_tl3<data_width_g, 1>::mm_txn_with_data());
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_w_slave_socket";
	       slaveSocket[2]  = 
		  new ocpip::ocp_slave_socket_tl3<data_width_g, 1>
		  (oss.str().c_str());
	       masterSocket[2]->set_ocp_config(_ocpParameters);
	       slaveSocket[2]->set_ocp_config(_ocpParameters);
	    }
	    else
	    {
	       masterSocket[2] = 0;
	       slaveSocket[2]  = 0;
	    }

	    // Create West sockets
	    if(_col != 0)
	    {
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_s_master_socket";
	       masterSocket[1] = 
		  new ocpip::ocp_master_socket_tl3<data_width_g, 1>
		  (oss.str().c_str(), 
		   ocpip::ocp_master_socket_tl3<data_width_g, 1>::mm_txn_with_data());
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_s_slave_socket";
	       slaveSocket[1]  = 
		  new ocpip::ocp_slave_socket_tl3<data_width_g, 1>
		  (oss.str().c_str());
	       masterSocket[1]->set_ocp_config(_ocpParameters);
	       slaveSocket[1]->set_ocp_config(_ocpParameters);
	    }
	    else
	    {
	       masterSocket[1] = 0;
	       slaveSocket[1]  = 0;
	    }

	    // Create East sockets
	    if(_col != (n_cols_g - 1))
	    {
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_e_master_socket";
	       masterSocket[3] = 
		  new ocpip::ocp_master_socket_tl3<data_width_g, 1>
		  (oss.str().c_str(), 
		   ocpip::ocp_master_socket_tl3<data_width_g, 1>::mm_txn_with_data());
	       oss.str("");
	       oss << "router_" << _row << "_" << _col << "_e_slave_socket";
	       slaveSocket[3]  = 
		  new ocpip::ocp_slave_socket_tl3<data_width_g, 1>
		  (oss.str().c_str());
	       masterSocket[3]->set_ocp_config(_ocpParameters);
	       slaveSocket[3]->set_ocp_config(_ocpParameters);
	    }
	    else
	    {
	       masterSocket[3] = 0;
	       slaveSocket[3]  = 0;
	    }
	    
	    // Bind callbacks and spawn a thread for outgoing links (masters)
	    
	    
	    if(masterSocket[0])
	    {
	       masterSocket[0]->register_nb_transport_bw
		  (this, &OcpTl3MeshRouter::nb_transport_bw_0, false);
	       slaveSocket[0]->register_nb_transport_fw
		  (this, &OcpTl3MeshRouter::nb_transport_fw_0, false);
	       sc_spawn(sc_bind(&OcpTl3MeshRouter::thread, this, 0));
	    }
	    if(masterSocket[1])
	    {
	       masterSocket[1]->register_nb_transport_bw
		  (this, &OcpTl3MeshRouter::nb_transport_bw_1, false);
	       slaveSocket[1]->register_nb_transport_fw
		  (this, &OcpTl3MeshRouter::nb_transport_fw_1, false);
	       sc_spawn(sc_bind(&OcpTl3MeshRouter::thread, this, 1));
	    }
	    if(masterSocket[2])
	    {
	       masterSocket[2]->register_nb_transport_bw
		  (this, &OcpTl3MeshRouter::nb_transport_bw_2, false);
	       slaveSocket[2]->register_nb_transport_fw
		  (this, &OcpTl3MeshRouter::nb_transport_fw_2, false);
	       sc_spawn(sc_bind(&OcpTl3MeshRouter::thread, this, 2));
	    }
	    if(masterSocket[3])
	    {
	       masterSocket[3]->register_nb_transport_bw
		  (this, &OcpTl3MeshRouter::nb_transport_bw_3, false);
	       slaveSocket[3]->register_nb_transport_fw
		  (this, &OcpTl3MeshRouter::nb_transport_fw_3, false);
	       sc_spawn(sc_bind(&OcpTl3MeshRouter::thread, this, 3));
	    }
	    if(masterSocket[4])
	    {
	       masterSocket[4]->register_nb_transport_bw
		  (this, &OcpTl3MeshRouter::nb_transport_bw_4, false);
	       slaveSocket[4]->register_nb_transport_fw
		  (this, &OcpTl3MeshRouter::nb_transport_fw_4, false);
	       sc_spawn(sc_bind(&OcpTl3MeshRouter::thread, this, 4));
	    }
	    

	    // Configure Sockets	    

	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       if(slaveSocket[i]) 
	       {slaveSocket[i]->set_ocp_config(_ocpParameters);}
	       if(masterSocket[i]) 
	       {masterSocket[i]->set_ocp_config(_ocpParameters);}
	    }
	 }


	 //* Destructor
	 ~OcpTl3MeshRouter()
	 {
	    // Free sockets
	    for(unsigned int i = 0; i < 5; ++i)
	    {
	       if(slaveSocket[i]) 
	       {delete slaveSocket[i]; slaveSocket[i] = 0;}
	       if(masterSocket[i]) 
	       {delete masterSocket[i]; masterSocket[i] = 0;}
	    }
	 }

      private:

	 void thread(unsigned int dir)
	 {
	    tlm::tlm_generic_payload* trans = 0;
	    tlm::tlm_phase            phase;
	    sc_core::sc_time          delay;
	    tlm::tlm_sync_enum        retval;
	    unsigned int              source = 0;

	    while(true)
	    {
	       // Check for pending transactions
	       for(unsigned int i = 0; i < 5; ++i)
	       {
		  if((trans = _peq[dir][i]->get_next_transaction()) != 0)
		  {
		     source = i;
		     break;
		  }
	       }

	       if(trans == 0)
	       {
		  // Wait for transaction
		  wait(_peq[dir][0]->get_event() |
		       _peq[dir][1]->get_event() |
		       _peq[dir][2]->get_event() |
		       _peq[dir][3]->get_event() |
		       _peq[dir][4]->get_event());
		  continue; // Jump back to the beginning of while(true)
	       }

	       // Calculate delay

	       phase = tlm::BEGIN_RESP;
	       if(source == 4)
	       {
		  delay = sc_core::SC_ZERO_TIME;
	       }
	       else
	       {
		  delay = _cycleLength * 
		     ((trans->get_data_length() / trans->get_streaming_width())
		      + 3);
	       }
	       
	       // Guess the completion time

	       retval = (*slaveSocket[source])->nb_transport_bw(*trans, 
								phase, 
								delay);

	       if(retval != tlm::TLM_COMPLETED)
	       {
		  std::ostringstream oss;
		  oss << "OcpTl3MeshRouter::thread : Not supporting responses";
		  throw std::runtime_error(oss.str().c_str());
	       }

	       // Forward transaction

	       phase = tlm::BEGIN_REQ;
	       delay = _cycleLength * 3;

	       retval = (*masterSocket[dir])->
		  nb_transport_fw(*trans, phase, delay);

	       if(retval == tlm::TLM_ACCEPTED || retval == tlm::TLM_UPDATED)
	       {
		  if(phase == tlm::BEGIN_REQ)
		  {		  
		     wait(_txCompleteEvent[dir]);		
		  }
		  else if(phase == tlm::END_REQ)
		  {
		     std::ostringstream oss;
		     oss << "OcpTlm3MeshRouter::thread : END_REQ not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else if(phase == tlm::BEGIN_RESP)
		  {
		     std::ostringstream oss;
		     oss << "OcpTlm3MeshRouter::thread : BEGIN_RESP"
			 << " not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else
		  {
		     std::ostringstream oss;
		     oss << "OcpTlm3MeshRouter::thread : invalid PHASE";
		     throw std::runtime_error(oss.str().c_str());
		  }	       
	       }
	       else if(retval == tlm::TLM_COMPLETED)
	       {
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
	       }
	       else
	       {
		  std::ostringstream oss;
		  oss << "OcpTl3MeshRouter::thread : invalid SYNC_ENUM";
		  throw std::runtime_error(oss.str().c_str());
	       }	       
	    	    
	       trans->release();


	    } // End of while(true)

	 }

	 

	 tlm::tlm_sync_enum nb_transport_fw(int id,
					    tlm::tlm_generic_payload &trans,
					    tlm::tlm_phase           &phase,
					    sc_core::sc_time         &delay)
	 {


	    // Only write command is supported
	    if(trans.get_command() != tlm::TLM_WRITE_COMMAND)
	    {
	       std::ostringstream oss;
	       oss << "OcpTl3MeshRouter::nb_tranport_fw " << id 
		   << ": only write command is supported";
	       throw std::runtime_error(oss.str().c_str());
	    }
	 
	    if(phase == tlm::BEGIN_REQ)
	    {
	       trans.acquire();
	       unsigned int destination = parseDest(trans.get_address());
	       _peq[destination][id]->notify(trans, delay);	       
	    }
	    else if(phase == tlm::END_RESP)
	    {
	       trans.set_response_status(tlm::TLM_OK_RESPONSE);
	       return tlm::TLM_COMPLETED;
	    }
	    else
	    {
	       std::ostringstream oss;
	       oss << "OcpTl3MeshRouter::nb_tranport_fw " << id 
		   << ": got invalid PHASE";
	       throw std::runtime_error(oss.str().c_str());
	    }
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_ACCEPTED;
	 }

	 
	 tlm::tlm_sync_enum nb_transport_bw(int id,
					    tlm::tlm_generic_payload &trans,
					    tlm::tlm_phase           &phase,
					    sc_core::sc_time         &delay)
	 {


	    if(phase == tlm::BEGIN_REQ || phase == tlm::END_RESP)
	    {
	       std::ostringstream oss;
	       oss << "OcpTl3MeshRouter::nb_tranport_bw " << id 
		   << " got wrong phase";
	       throw std::runtime_error(oss.str().c_str());
	    }

	    _txCompleteEvent[id].notify(delay);
	 
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_COMPLETED;
	 }

	 tlm::tlm_sync_enum nb_transport_bw_0(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_bw(0, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_fw_0(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_fw(0, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_bw_1(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_bw(1, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_fw_1(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_fw(1, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_bw_2(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_bw(2, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_fw_2(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_fw(2, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_bw_3(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_bw(3, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_fw_3(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_fw(3, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_bw_4(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_bw(4, trans, phase, delay);
	 }

	 tlm::tlm_sync_enum nb_transport_fw_4(tlm::tlm_generic_payload &trans,
					      tlm::tlm_phase           &phase,
					      sc_core::sc_time         &delay)
	 {
	    nb_transport_fw(4, trans, phase, delay);
	 }


	 unsigned int parseDest(unsigned long int addr)
	 {
	    unsigned long int rowPart = (addr & 0xFFFF0000) >> 16;
	    unsigned long int colPart = addr & 0x0000FFFF;

	    if(_row == rowPart && _col == colPart)
	    {
	       return 4; // IP
	    }
	    else if(_row < rowPart)
	    {
	       return 2; // S
	    }
	    else if(_row > rowPart)
	    {
	       return 0; // N
	    }
	    else if(_col < colPart)
	    {
	       return 3; // E
	    }
	    else
	    {
	       return 1; // W
	    }
	 }
	 
	 unsigned int _row;
	 unsigned int _col;
	 sc_core::sc_time _cycleLength;
	 ocpip::ocp_parameters _ocpParameters;

	 tlm_utils::peq_with_get<tlm::tlm_generic_payload>* _peq[5][5];
	 sc_core::sc_event                     _txCompleteEvent[5];
      };
	 
   }
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:

