/**
 * File:    UIConfigIO.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 12.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.xml;

import java.awt.Point;
import java.awt.geom.Point2D;
import java.util.Iterator;
import java.util.List;

import javax.swing.JInternalFrame;
import javax.swing.SwingUtilities;

import fi.cpu.data.Chart;
import fi.cpu.data.Configuration;
import fi.cpu.data.Model;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Process;
import fi.cpu.data.ProcessingElement;
import fi.cpu.data.ProcessingElementType;
import fi.cpu.data.Thread;
import fi.cpu.ui.MainWindow;
import fi.cpu.ui.graph.GraphPanel;
import fi.cpu.xml.bind.configuration.ServicesType;
import fi.cpu.xml.bind.uiconfig.AcceleratorType;
import fi.cpu.xml.bind.uiconfig.ChartType;
import fi.cpu.xml.bind.uiconfig.ConfigurationType;
import fi.cpu.xml.bind.uiconfig.ControlViewType;
import fi.cpu.xml.bind.uiconfig.CpuType;
import fi.cpu.xml.bind.uiconfig.MainWindowType;
import fi.cpu.xml.bind.uiconfig.MonitorViewType;
import fi.cpu.xml.bind.uiconfig.ObjectFactory;
import fi.cpu.xml.bind.uiconfig.ServiceViewType;
import fi.cpu.xml.bind.uiconfig.ThreadType;
import fi.cpu.xml.bind.uiconfig.UIConfigType;
import fi.cpu.xml.bind.uiconfig.WindowLocationType;
import fi.cpu.xml.bind.uiconfig.WindowSizeType;
import fi.cpu.xml.bind.uiconfig.ServiceViewType.Connection;
import fi.cpu.xml.bind.uiconfig.ServiceViewType.Service;
import fi.cpu.xml.bind.uiconfig.ServiceViewType.ConnectionType.PointType;


/**
 * UIConfigIO handles reading and writing UI configurations.
 */
public class UIConfigIO {
	
	private UIConfigIO() {
		// only static methods, no use for constructor
	}
	
	
    /**
     * Writes UI configuration to file.
     * @param configuration
     * @param mw
     * @param filename
     */
    public static void writeConfiguration(
    		Configuration configuration, MainWindow mw, String filename ) {
    	
    	Model peModel = configuration.getPeModel();
    	Model serviceModel = configuration.getServiceModel();
    	
        try {
            UIConfigMarshaller m = new UIConfigMarshaller();
            ObjectFactory of = m.getObjectFactory();

            UIConfigType uiconfig = m.getObjectFactory().createUIConfig();
            uiconfig.setMainWindow( of.createMainWindowType() );
            uiconfig.setControlView( of.createControlViewType() );
            uiconfig.setMonitorView( of.createMonitorViewType() );
            
            if (serviceModel.getChildCount(serviceModel.getModelRoot()) > 0) {
            	uiconfig.setServiceView( of.createServiceViewType() );
            }
            
            MainWindowType mainWindow = uiconfig.getMainWindow();
            mainWindow.setWindowSize( of.createWindowSizeType() );

            WindowSizeType windowSize = mainWindow.getWindowSize();
            windowSize.setX(mw.getWidth());
            windowSize.setY(mw.getHeight());

            ControlViewType controlView = uiconfig.getControlView();
            controlView.setConfiguration( of.createConfigurationType() );
            controlView.setAutoActivateState(mw.isAutoActive());

            // Save processing element information
            ConfigurationType confType = controlView.getConfiguration();
            List cpus = confType.getCpu();
            List acs = confType.getAccelerator();

            Iterator<ModelNode> peIter = peModel.getChildren(peModel.getModelRoot()).iterator();

            // for all pes
            while (peIter.hasNext()) {
            	ModelNode peNode = peIter.next();
            	if (!(peNode instanceof ProcessingElement)) {
            		continue;
            	}
                ProcessingElement pe = (ProcessingElement)peNode;

                // pe is cpu?
                if (pe.getType() == ProcessingElementType.CPU) {
                	// create cputype
                    ConfigurationType.Cpu cpuType = of
                            .createConfigurationTypeCpu();
                    cpuType.setId(Integer.parseInt(pe.getId()));
                    cpuType.setName(pe.getName());
                    
                    List xml_threads = cpuType.getThread();
                    Iterator<ModelNode> threadIter = peModel.getChildren(pe).iterator();
                    
                    // for all threads
                    while (threadIter.hasNext()) {
                    	ModelNode tNode = threadIter.next();
                    	if (!(tNode instanceof Thread)) {
                    		continue;
                    	}
                        Thread thread = (Thread)tNode;
                        
                        // create threadtype
                        CpuType.Thread threadType = of.createCpuTypeThread();
                        threadType.setName(thread.getName());
                        threadType.setId(Integer.parseInt(thread.getId()));
                        threadType.setPriority(thread.getPriority());

                        List processList = threadType.getProcess();
                        Iterator<ModelNode> processIter = peModel.getChildren(thread).iterator();

                        // for all processes
                        while (processIter.hasNext()) {
                        	ModelNode pNode = processIter.next();
                        	if (!(pNode instanceof Process)) {
                        		continue;
                        	}
                            Process p = (Process)pNode;
                            
                            // create xml process
                            ThreadType.Process process = of
                                    .createThreadTypeProcess();
                            process.setId(p.getId());
                            process.setName(p.getName());

                            processList.add(process);
                        }

                        xml_threads.add(threadType);
                    }

                    List xml_charts = cpuType.getChart();
                    Iterator<Chart> chartIter = pe.getCharts().iterator();
                    // for all charts
                    while (chartIter.hasNext()) {
                        Chart chart = chartIter.next();

                        String graphID = chart.getId();
                        int typeID = chart.getTypeId();
                        double maxValue = chart.getMaxValue();
                        double minValue = chart.getMinValue();
                        double length = chart.getLength();
                        String title = chart.getTitle();
                        String xAxis = chart.getXAxis();
                        String yAxis = chart.getYAxis();

                        CpuType.Chart chartType = of.createCpuTypeChart();
                        chartType.setAutomaticScaling(chart.isAutoScale());
                        chartType.setChartTitle( title );
                        chartType.setGraphID( "" + graphID );
                        chartType.setLength( length );
                        chartType.setMaxValue( maxValue );
                        chartType.setMinValue( minValue );
                        chartType.setTypeID( "" + typeID );
                        chartType.setXaxisTitle( xAxis );
                        chartType.setYaxisTitle( yAxis );
                        chartType.setLogged(chart.isLogged());

                        xml_charts.add( chartType );

                    }
                    String visibleChart = pe.getVisibleChart();
                    if (visibleChart != null) {
                    	cpuType.setVisibleChart(visibleChart);
                    }
                    cpus.add( cpuType );

                // pe is accelerator
                } else {
                	// create acceleratortype
                    ConfigurationType.Accelerator acceleratorType = of
                            .createConfigurationTypeAccelerator();
                    acceleratorType.setId(Integer.parseInt(pe.getId()));
                    acceleratorType.setName(pe.getName());

                    List xml_threads = acceleratorType.getThread();
                    Iterator<ModelNode> threadIter = peModel.getChildren(pe).iterator();
                    
                    // for all threads
                    while (threadIter.hasNext()) {
                    	ModelNode tNode = threadIter.next();
                    	if (!(tNode instanceof Thread)) {
                    		continue;
                    	}
                        Thread thread = (Thread)tNode;
                        
                        // create threadtype
                        CpuType.Thread threadType = of.createCpuTypeThread();
                        threadType.setName(thread.getName());
                        threadType.setId(Integer.parseInt(thread.getId()));
                        threadType.setPriority(thread.getPriority());

                        List processList = threadType.getProcess();
                        Iterator<ModelNode> processIter = peModel.getChildren(thread).iterator();

                        // for all processes
                        while (processIter.hasNext()) {
                        	ModelNode pNode = processIter.next();
                        	if (!(pNode instanceof Process)) {
                        		continue;
                        	}
                            Process p = (Process)pNode;
                            
                            // create xml process
                            ThreadType.Process process = of
                                    .createThreadTypeProcess();
                            process.setId(p.getId());
                            process.setName(p.getName());

                            processList.add(process);
                        }

                        xml_threads.add(threadType);
                    }

                    List xml_charts = acceleratorType.getChart();
                    Iterator<Chart> chartIter = pe.getCharts().iterator();
                    // for all charts
                    while (chartIter.hasNext()) {
                        Chart chart = chartIter.next();

                        String graphID = chart.getId();
                        int typeID = chart.getTypeId();
                        double maxValue = chart.getMaxValue();
                        double minValue = chart.getMinValue();
                        double length = chart.getLength();
                        String title = chart.getTitle();
                        String xAxis = chart.getXAxis();
                        String yAxis = chart.getYAxis();

                        AcceleratorType.Chart chartType = of
                                .createAcceleratorTypeChart();
                        chartType.setAutomaticScaling(chart.isAutoScale());
                        chartType.setChartTitle( title );
                        chartType.setGraphID( "" + graphID );
                        chartType.setLength( length );
                        chartType.setMaxValue( maxValue );
                        chartType.setMinValue( minValue );
                        chartType.setTypeID( "" + typeID );
                        chartType.setXaxisTitle( xAxis );
                        chartType.setYaxisTitle( yAxis );
                        chartType.setLogged(chart.isLogged());

                        xml_charts.add( chartType );

                    }
                    String visibleChart = pe.getVisibleChart();
                    if (visibleChart != null) {
                    	acceleratorType.setVisibleChart(visibleChart);
                    }
                    acs.add( acceleratorType );
                }

            }

            //Tallennetaan Monitorview-ikkunat
            MonitorViewType monitorView = uiconfig.getMonitorView();
            List windowList = monitorView.getWindow();

            // Save debug window
            JInternalFrame debugWindow = mw.getDebugWindow();
            if (debugWindow != null) {
                MonitorViewType.Window window = of
                        .createMonitorViewTypeWindow();

                //Tallennetaan paikka
                WindowLocationType windowLocation = of
                        .createWindowLocationType();
                Point currentLocation = new Point();
                //Konvertoi ikonin paikan suhteessa Monitor view-ikkunaan
                currentLocation = SwingUtilities.convertPoint(debugWindow,
                        currentLocation, mw.getChartContainerPanel());

                if ( currentLocation.x < 0 )
                {
                    currentLocation.x = 20;
                }
                if ( currentLocation.y < 0 )
                {
                    currentLocation.y = 20;
                }

                windowLocation.setX( currentLocation.x );
                windowLocation.setY( currentLocation.y );
                window.setWindowLocation( windowLocation );

                //Tieto, ett kyseess on debug-window
                window.setType( "debug" );

                //Tallennetaan ikkunan koko
                WindowSizeType debugWindowSize = of.createWindowSizeType();
                debugWindowSize.setX( debugWindow.getWidth() );
                debugWindowSize.setY( debugWindow.getHeight() );
                window.setWindowSize( debugWindowSize );

                windowList.add( window );
            }

            //Sitten loput mahdolliset kuvaajat
            Iterator<GraphPanel> chartIter = mw.getChartPanels().iterator();
            while (chartIter.hasNext()) {
            	GraphPanel gPanel = chartIter.next();
            	MonitorViewType.Window window = of
            	.createMonitorViewTypeWindow();
            	
            	//Tallennetaan paikka
            	WindowLocationType windowLocation = of
            	.createWindowLocationType();
            	Point currentLocation = new Point();
            	//Konvertoi ikonin paikan suhteessa Monitor view-ikkunaan
            	currentLocation = SwingUtilities.convertPoint( gPanel,
            			currentLocation, mw.getChartContainerPanel() );
            	
            	if ( currentLocation.x < 0 )
            	{
            		currentLocation.x = 20;
            	}
            	if ( currentLocation.y < 0 )
            	{
            		currentLocation.y = 20;
            	}
            	
            	windowLocation.setX( currentLocation.x );
            	windowLocation.setY( currentLocation.y );
            	window.setWindowLocation( windowLocation );
            	
            	//Tieto, ett kyseess on chart-window
            	window.setType( "chart" );
            	
            	//Tallennetaan ikkunan koko
            	WindowSizeType chartWindowSize = of.createWindowSizeType();
            	chartWindowSize.setX( gPanel.getWidth() );
            	chartWindowSize.setY( gPanel.getHeight() );
            	window.setWindowSize( chartWindowSize );
            	
            	//Luodaan ChartType, joka kertoo chartin yksityiskohdat
            	String graphID = gPanel.getChartModel().getId();
            	int typeID = gPanel.getChartModel().getTypeId();
            	double maxValue = gPanel.getChartModel().getMaxValue();
            	double minValue = gPanel.getChartModel().getMinValue();
            	double length = gPanel.getChartModel().getLength();
            	String title = gPanel.getChartModel().getTitle();
            	String xAxis = gPanel.getChartModel().getXAxis();
            	String yAxis = gPanel.getChartModel().getYAxis();
            	
            	ChartType chartType = of.createChartType();
            	chartType.setAutomaticScaling( gPanel.getChartModel().isAutoScale() );
            	chartType.setChartTitle( title );
            	chartType.setGraphID( "" + graphID );
            	chartType.setLength( length );
            	chartType.setMaxValue( maxValue );
            	chartType.setMinValue( minValue );
            	chartType.setTypeID( "" + typeID );
            	chartType.setXaxisTitle( xAxis );
            	chartType.setYaxisTitle( yAxis );
            	chartType.setLogged(gPanel.getChartModel().isLogged());
            	
            	window.setChart( chartType );
            	
            	windowList.add( window );
            }

            // handle services
            ServiceViewType servicesType = uiconfig.getServiceView();
            if (servicesType != null) {
            	
            	Iterator<ModelNode> nodeIter = serviceModel.getChildren(
            			serviceModel.getModelRoot()).iterator();
            	List serviceList = servicesType.getService();
            	List connectionList = servicesType.getConnection();

            	// Transform ui services and connections to
            	// configuration services and connections
            	while (nodeIter.hasNext()) {
            		ModelNode node = nodeIter.next();
            		if (node instanceof fi.cpu.data.Service) {
            			fi.cpu.data.Service uiService = (fi.cpu.data.Service) node;
						
    					Service service = of.createServiceViewTypeService();
    					service.setId(uiService.getId());
    					service.setName(uiService.getName());
    					service.setX((int)uiService.getLocation().getX());
    					service.setY((int)uiService.getLocation().getY());

    					List processes = service.getProcess();

    	                Iterator<ModelNode> pIter = serviceModel.getChildren(uiService).iterator();
    	                while (pIter.hasNext()) {
    	                	ModelNode pNode = pIter.next();
    	                	if (!(pNode instanceof Process)) {
    	                		continue;
    	                	}
    	                    Process uiProcess = (Process)pNode;
    						ServiceViewType.ServiceType.Process p = of.createServiceViewTypeServiceTypeProcess();
    						p.setId(uiProcess.getId());
    						p.setName(uiProcess.getName());
    						processes.add(p);
    					}
    					
    					serviceList.add(service);
    					
					} else if (node instanceof fi.cpu.data.Connection) {
	            		fi.cpu.data.Connection uiConnection = (fi.cpu.data.Connection) node;
						
	            		Connection connection = of.createServiceViewTypeConnection();
	            		connection.setId(uiConnection.getId());
	            		connection.setSrc(uiConnection.getSrcService());
	            		connection.setDest(uiConnection.getDestService());
	            		
	            		List points = connection.getPoint();
	            		List<Point2D> uiPoints = uiConnection.getPoints();
	            		for (int j=0; j<uiPoints.size(); ++j) {
	            			Point2D uiPoint = uiPoints.get(j);
	            			
	            			PointType p = of.createServiceViewTypeConnectionTypePointType();
	            			p.setX((int)uiPoint.getX());
	            			p.setY((int)uiPoint.getY());
	            			
	            			points.add(p);
	            		}
	            		
						connectionList.add(connection);
					}
            	}
            }

            m.saveUIConfig( filename, uiconfig );
        }
        catch ( Exception e )
        {
            e.printStackTrace();
        }
    }

    
    /**
     * Reads configuration from a file.
     * @param configuration
     * @param mw
     * @param filename
     */
    public static void readConfiguration(
    		Configuration configuration, MainWindow mw, String filename) {
    	Model peModel = new Model();
    	Model serviceModel = new Model();
    	
        try {
            UIConfigMarshaller m = new UIConfigMarshaller();
            UIConfigType loadedUIConfig = m.loadUIConfig( filename );

            MainWindowType window = loadedUIConfig.getMainWindow();
            mw.setSize( window.getWindowSize().getX(), window.getWindowSize()
                    .getY() );

            // Get the auto active state
            ControlViewType controlWindow = loadedUIConfig.getControlView();
            mw.setAutoActive(controlWindow.isAutoActivateState());

            //Haetaan CPU:iden tiedot
            ConfigurationType confType = controlWindow.getConfiguration();
            List cpus = confType.getCpu();
            Iterator iter = cpus.iterator();
            
            while ( iter.hasNext() )
            {
                ConfigurationType.Cpu cpu = (ConfigurationType.Cpu) iter.next();

                ProcessingElement pe = new ProcessingElement(""+cpu.getId(),
                			cpu.getName(), ProcessingElementType.CPU);

                Iterator threads = cpu.getThread().iterator();

                while ( threads.hasNext() )
                {
                    CpuType.Thread threadType = (CpuType.Thread) threads.next();

                    Thread thread = new Thread(""+threadType.getId(),
                    		threadType.getName(), threadType.getPriority());

                    Iterator processes = threadType.getProcess().iterator();

                    while ( processes.hasNext() )
                    {
                        ThreadType.Process process = (ThreadType.Process) processes
                                .next();

                        Process p = new Process(process.getId(), process.getName());
                        peModel.insertNode(p, thread);
                    }

                    peModel.insertNode(thread, pe);
                }

                Iterator charts = cpu.getChart().iterator();

                while ( charts.hasNext() )
                {
                    CpuType.Chart chartType = (CpuType.Chart) charts.next();

                    String graphID = chartType.getGraphID();
                    int typeID = Integer.parseInt( chartType.getTypeID() );
                    double maxValue = chartType.getMaxValue();
                    double minValue = chartType.getMinValue();
                    double length = chartType.getLength();
                    String title = chartType.getChartTitle();
                    String xAxis = chartType.getXaxisTitle();
                    String yAxis = chartType.getYaxisTitle();
                    boolean automaticScaling = chartType.isAutomaticScaling();
                    boolean logged = chartType.isLogged();
                    
                    Chart chart = new Chart(graphID, typeID, title,
                    		xAxis, yAxis, minValue, maxValue, length,
                    		automaticScaling, logged);
                    pe.setChart(graphID, chart);
                }
                pe.setDefinesCharts(true);
                pe.setVisibleChart(cpu.getVisibleChart());
                peModel.insertNode(pe, peModel.getModelRoot());
            }

            //Haetaan AC:iden tiedot
            List accelerators = confType.getAccelerator();
            iter = accelerators.iterator();

            while ( iter.hasNext() )
            {
                ConfigurationType.Accelerator ac = (ConfigurationType.Accelerator) iter.next();

                ProcessingElement pe = new ProcessingElement(""+ac.getId(),
                		ac.getName(), ProcessingElementType.ACC);
                
                Iterator threads = ac.getThread().iterator();
                
                while ( threads.hasNext() )
                {
                	AcceleratorType.Thread threadType = (AcceleratorType.Thread) threads.next();
                	
                	Thread thread = new Thread(""+threadType.getId(),
                			threadType.getName(), threadType.getPriority());
                	
                	Iterator processes = threadType.getProcess().iterator();
                	
                	while ( processes.hasNext() )
                	{
                		ThreadType.Process process = (ThreadType.Process) processes
                		.next();
                		
                		Process p = new Process(process.getId(), process.getName());
                        peModel.insertNode(p, thread);
                	}
                	
                    peModel.insertNode(thread, pe);
                }
                
                Iterator charts = ac.getChart().iterator();
                
                while ( charts.hasNext() )
                {
                	AcceleratorType.Chart chartType = (AcceleratorType.Chart) charts.next();
                	
                	String graphID = chartType.getGraphID();
                	int typeID = Integer.parseInt( chartType.getTypeID() );
                	double maxValue = chartType.getMaxValue();
                	double minValue = chartType.getMinValue();
                	double length = chartType.getLength();
                	String title = chartType.getChartTitle();
                	String xAxis = chartType.getXaxisTitle();
                	String yAxis = chartType.getYaxisTitle();
                	boolean automaticScaling = chartType.isAutomaticScaling();
                    boolean logged = chartType.isLogged();
                	
                    Chart chart = new Chart(graphID, typeID, title,
                    		xAxis, yAxis, minValue, maxValue, length,
                    		automaticScaling, logged);
                    pe.setChart(graphID, chart);
                }
                pe.setDefinesCharts(true);
                pe.setVisibleChart(ac.getVisibleChart());
                peModel.insertNode(pe, peModel.getModelRoot());
            }

            configuration.initializePeModel(peModel);
            
            //Haetaan Monitorview-ikkunat
            MonitorViewType monitorView = loadedUIConfig.getMonitorView();
            Iterator windowList = monitorView.getWindow().iterator();
            
            // Remove old charts
            mw.removeCharts();

            while ( windowList.hasNext() )
            {
                MonitorViewType.Window monitorWindow = (MonitorViewType.Window) windowList
                        .next();

                //Haetaan paikka
                WindowLocationType windowLocation = monitorWindow
                        .getWindowLocation();

                //Haetaan ikkunan koko
                WindowSizeType windowSize = monitorWindow.getWindowSize();

                if ( monitorWindow.getType().equalsIgnoreCase( "debug" ) )
                {
                    //Debug-window
                    mw.showDebugWindow( windowLocation.getX(), windowLocation
                            .getY(), windowSize.getX(), windowSize.getY() );
                }
                else
                {
                    //Normal chart
                    ChartType chartType = monitorWindow.getChart();

                    GraphPanel gPanel = new GraphPanel();

                    String graphID = chartType.getGraphID();
                    int typeID = Integer.parseInt( chartType.getTypeID() );
                    double maxValue = chartType.getMaxValue();
                    double minValue = chartType.getMinValue();
                    double length = chartType.getLength();
                    String title = chartType.getChartTitle();
                    String xAxis = chartType.getXaxisTitle();
                    String yAxis = chartType.getYaxisTitle();
                    boolean automaticScaling = chartType.isAutomaticScaling();
                    boolean logged = chartType.isLogged();

                    Chart chart = new Chart(graphID, typeID, title,
                    		xAxis, yAxis, minValue, maxValue, length,
                    		automaticScaling, logged);

                    gPanel.setChart(chart);

                    //Joudutaan tekemn pieni hackki. Tiedostoon tallennetut
                    // tiedot
                    //koskevat niden graaffien osalta JInternalFramen kokoa,
                    // mutta
                    //addGraph muuttaa ja sijoittaa JInternalFramea. Siksi
                    // pit paikka- ja kokotietoon
                    //list muutama arvo, ett ikkunat menevt samoihin
                    // kohtiin kuin oli talletustilanteessa
                    mw.addChart( gPanel, windowLocation.getX() - 5, windowLocation
                            .getY() - 27, windowSize.getX() + 10, windowSize
                            .getY() + 32, false);
                }
            }

            
            // handle services
            ServiceViewType servicesType = loadedUIConfig.getServiceView();
            if (servicesType != null) {
            	
            	// Transform configuration services to ui services
            	List serviceList = servicesType.getService();
            	for (int i=0; i<serviceList.size(); ++i) {
            		Object s = serviceList.get(i);
            		if (s instanceof Service) {
						Service service = (Service) s;
						fi.cpu.data.Service uiService = new fi.cpu.data.Service(
								service.getId(), service.getName());
						uiService.setLocation(new Point2D.Double(service.getX(), service.getY()));
						
						// Handle processes
	                	Iterator processes = service.getProcess().iterator();
	                	
	                	while (processes.hasNext()) {
	                		ServiceViewType.ServiceType.Process process =
	                			(ServiceViewType.ServiceType.Process)processes.next();
	                		Process p = new Process(process.getId(), process.getName());

	                		serviceModel.insertNode(p, uiService);
	                	}
						
	                    serviceModel.insertNode(uiService, serviceModel.getModelRoot());
					}
            	}

            	// Transform configuration connections to ui connections
            	List connectionList = servicesType.getConnection();
            	for (int i=0; i<connectionList.size(); ++i) {
            		Object c = connectionList.get(i);
            		if (c instanceof Connection) {
						Connection conn = (Connection) c;
						fi.cpu.data.Connection uiConnection = new fi.cpu.data.Connection(
									conn.getId(), conn.getSrc(), conn.getDest());
						
						List points = conn.getPoint();
						for (int j=0; j<points.size(); ++j) {
							PointType p = (PointType)points.get(j);
							Point2D uiPoint = new Point2D.Double(p.getX(), p.getY());
							uiConnection.addPoint(uiPoint);
						}
						
	                    serviceModel.insertNode(uiConnection, serviceModel.getModelRoot());
					}
            	}
            }
        	configuration.initializeServiceModel(serviceModel);
        }
        catch ( Exception e )
        {
            e.printStackTrace();
        }
    }
    
    
    /**
     * Reads configuration from a string.
     * @param configuration
     * @param data
     */
    public static void readConfigurationString(
    		Configuration configuration, MainWindow mw, String data) {
    	Model peModel = new Model();
    	Model serviceModel = new Model();

        try
        {            	
            XML_Source_FPGA fpga = new XML_Source_FPGA();
            
            fi.cpu.xml.bind.configuration.ConfigurationType configType = fpga.getConfiguration( data );
            
        	// Handle cpus
            List cpuList = configType.getCpu();
            
            for ( int i = 0; i < cpuList.size(); i++ )
            {
            	fi.cpu.xml.bind.configuration.CpuType cpu = 
            		(fi.cpu.xml.bind.configuration.CpuType) cpuList.get( i );
                
                ProcessingElement pe = new ProcessingElement(
                		""+cpu.getId(), cpu.getName(), ProcessingElementType.CPU);

                List threadList = cpu.getThread();

                for ( int j = 0; j < threadList.size(); j++ )
                {
                	fi.cpu.xml.bind.configuration.ThreadType threadType = 
                		(fi.cpu.xml.bind.configuration.ThreadType) threadList.get( j );

                    Thread thread = new Thread(
                    		""+threadType.getId(), threadType.getName(), threadType.getPriority());

                    List processList = threadType.getProcess();

                    for ( int k = 0; k < processList.size(); k++ )
                    {
                    	fi.cpu.xml.bind.configuration.ProcessType processType = 
                    		(fi.cpu.xml.bind.configuration.ProcessType) processList.get( k );
                        
                        Process process = new Process(
                        		processType.getId(), processType.getName());
                        peModel.insertNode(process, thread);
                    }

                    peModel.insertNode(thread, pe);
                }
                peModel.insertNode(pe, peModel.getModelRoot());
            }
            
        	// Handle accelerators
            List acceleratorList = configType.getAccelerator();
                            
            for ( int i = 0; i < acceleratorList.size(); i++ )
            {
            	fi.cpu.xml.bind.configuration.AcceleratorType acc = 
            		(fi.cpu.xml.bind.configuration.AcceleratorType) acceleratorList.get( i );
                
                ProcessingElement pe = new ProcessingElement(
                		""+acc.getId(), acc.getName(), ProcessingElementType.ACC);
                                    
                List threadList = acc.getThread();

                for ( int j = 0; j < threadList.size(); j++ )
                {
                	fi.cpu.xml.bind.configuration.ThreadType threadType = 
                		(fi.cpu.xml.bind.configuration.ThreadType) threadList.get( j );

                    Thread thread = new Thread(
                    		""+threadType.getId(), threadType.getName(), threadType.getPriority());

                    List processList = threadType.getProcess();

                    for ( int k = 0; k < processList.size(); k++ )
                    {
                    	fi.cpu.xml.bind.configuration.ThreadType.Process processType = 
                    		(fi.cpu.xml.bind.configuration.ThreadType.Process)processList.get( k );
                        Process process = new Process(
                        		processType.getId(), processType.getName());
                        peModel.insertNode(process, thread);
                    }

                    peModel.insertNode(thread, pe);
                }
                                    
                peModel.insertNode(pe, peModel.getModelRoot());
            }
            
            configuration.initializePeModel(peModel);
            
            // Handle services
            ServicesType servicesType = configType.getServices();
            if (servicesType != null) {
            	
            	// Transform configuration services to ui services
            	List serviceList = servicesType.getService();
            	for (int i=0; i<serviceList.size(); ++i) {
            		Object s = serviceList.get(i);
            		if (s instanceof ServicesType.Service) {
            			ServicesType.Service service = (ServicesType.Service) s;
            			fi.cpu.data.Service uiService = new fi.cpu.data.Service(
    							service.getId(), service.getName());
            			
						// Handle processes
	                	Iterator processes = service.getProcess().iterator();
	                	
	                	while (processes.hasNext()) {
	                		ServicesType.ServiceType.Process process =
	                			(ServicesType.ServiceType.Process)processes.next();
	                		Process p = new Process(process.getId(), process.getName());

	                		serviceModel.insertNode(p, uiService);
	                	}
						
	                    serviceModel.insertNode(uiService, serviceModel.getModelRoot());
    				}
            	}

            	// Transform configuration connections to ui connections
            	List connectionList = servicesType.getConnection();
            	for (int i=0; i<connectionList.size(); ++i) {
            		Object c = connectionList.get(i);
            		if (c instanceof ServicesType.Connection) {
            			ServicesType.Connection conn = (ServicesType.Connection) c;
            			fi.cpu.data.Connection uiConnection = new fi.cpu.data.Connection(
    							conn.getId(), conn.getSrc(), conn.getDest());
    					
	                    serviceModel.insertNode(uiConnection, serviceModel.getModelRoot());
    				}
            	}
            }
            configuration.initializeServiceModel(serviceModel);
        }
        catch ( Exception e )
        {
            e.printStackTrace();
        }
    }
    
    
    /**
     * Creates an XML-string describing the current mapping.
     * @param configuration
     * @return
     */
    public static String createMappingXMLString(Configuration configuration) {
        StringBuffer xmlString = new StringBuffer();
        Model peModel = configuration.getPeModel();
        
        // start tag
        xmlString.append("<mapping>");

        // for all pes
        Iterator<ModelNode> peIter = peModel.getChildren(peModel.getModelRoot()).iterator();
        while (peIter.hasNext()) {
        	ModelNode peNode = peIter.next();
        	if (!(peNode instanceof ProcessingElement)) {
        		continue;
        	}
            ProcessingElement pe = (ProcessingElement)peNode;
            
            // for all threads
            Iterator<ModelNode> threadIter = peModel.getChildren(pe).iterator();
            while (threadIter.hasNext()) {
            	ModelNode tNode = threadIter.next();
            	if (!(tNode instanceof Thread)) {
            		continue;
            	}
                Thread t = (Thread)tNode;

                xmlString.append("<thread id=\"" + t.getId() + "\" priority=\"" + t.getPriority() + "\">");

                // for all processes
                Iterator<ModelNode> pIter = peModel.getChildren(t).iterator();
                while (pIter.hasNext()) {
                	ModelNode pNode = pIter.next();
                	if (!(pNode instanceof Process)) {
                		continue;
                	}
                    Process p = (Process)pNode;
                    xmlString.append("<process id=\"" + p.getId() + "\" />");
                }
                xmlString.append( "</thread>" );
            }
        }

        // end tag
        xmlString.append( "</mapping>" );

        return xmlString.toString();
    }
}
