/**
 * File:    ServiceReportPanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 26.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.service;

import java.awt.GridLayout;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.swing.JPanel;

import fi.cpu.data.Configuration;
import fi.cpu.data.Model;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Service;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;


/**
 * A panel for showing service report information.
 */
public class ServiceReportPanel extends JPanel {
	protected Configuration config;
	protected Map<String, ServicePanel> servicePanels;
	
	
	/**
	 * Creates a new ServiceReportPanel.
	 */
	public ServiceReportPanel(Configuration config) {
		super();
		this.config = config;
		config.getServiceModel().addModelListener(new MyModelListener());
		
		servicePanels = new HashMap<String, ServicePanel>();

		setLayout(new GridLayout(0,3));
		
		initPanel();
	}

	
	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		servicePanels.clear();
		
		Model serviceModel = config.getServiceModel();
        Iterator<ModelNode> sIter = serviceModel.getChildren(serviceModel.getModelRoot()).iterator();

        while (sIter.hasNext()) {
        	ModelNode node = sIter.next();
        	if (!(node instanceof Service)) {
        		continue;
        	}
        	Service s = (Service) node;
        	servicePanels.put(""+s.getId(), new ServicePanel(s));
        }
        drawServicePanels();
	}

	
    /**
     * Draws processing element panels.
     */
    protected void drawServicePanels() {
    	// Clear old panels
		removeAll();

		// Sort panels
		ArrayList<ServicePanel> sList = new ArrayList<ServicePanel>(servicePanels.values());
		Collections.sort(sList);
		
		// Add new panels
		Iterator<ServicePanel> sIter = sList.iterator();
		while (sIter.hasNext()) {
			add(sIter.next());
		}
		
		validate();
		repaint();
    }

    
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		if (parent != null && parent.equals(config.getServiceModel().getModelRoot())) {
    			if (e.getNode() instanceof Service) {
    				// Services changed.
    				initPanel();
    			}
    		}
    	}
    	public void nodeRemoved(ModelEvent e) {
    		ModelNode oldParent = e.getOldParent();
    		
    		if (oldParent != null && oldParent.equals(config.getServiceModel().getModelRoot())) {
    			if (e.getNode() instanceof Service) {
    				// Services changed.
    				initPanel();
    			}
    		}
    	}
    	public void nodeMoved(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		ModelNode oldParent = e.getOldParent();
    		
    		if ((parent != null && parent.equals(config.getServiceModel().getModelRoot()))
    				|| (oldParent != null && oldParent.equals(config.getServiceModel().getModelRoot()))) {
    			if (e.getNode() instanceof Service) {
    				// Services changed.
    				initPanel();
    			}
    		}
    	}
    	public void structureChanged(ModelEvent e) {
    		ModelNode node = e.getNode();
    		
    		if (node != null && node.equals(config.getServiceModel().getModelRoot())) {
    			// Structure changed starting from the root.
    			initPanel();
    		}
    	}
    }
}
