/*
 * CVS-tietoja:
 * $Author: perttul5 $ 
 * $Date: 2008-05-20 17:23:56 +0300 (ti, 20 touko 2008) $ 
 * $Revision: 3514 $
 *
 * Created on 25.7.2005
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *
 */
package fi.cpu.ui.graph;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;

import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.AxisLocation;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.data.time.Millisecond;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import fi.cpu.Settings;
import fi.cpu.data.Chart;
import fi.cpu.data.ChartDataLogger;
import fi.cpu.ui.MainWindow;
import fi.cpu.ui.ctrl.PePanel;


/**
 * @author jsikio
 *
 * Panel that contain some JFreeChart graph
 *
 */
public class GraphPanel extends JPanel
{
	protected static final Color COLOR_CURVE;
	protected static final Color COLOR_MIN;
	protected static final Color COLOR_AVG;
	protected static final Color COLOR_MAX;

    protected JFreeChart chart_ = null;
    protected ChartPanel chartPanel_ = null;
    protected Chart chartModel;
    
	protected static final String MIN_CMD = "min";
	protected static final String AVG_CMD = "avg";
	protected static final String MAX_CMD = "max";
	
	protected JCheckBoxMenuItem minCB_;
	protected JCheckBoxMenuItem avgCB_;
	protected JCheckBoxMenuItem maxCB_;

	protected JCheckBox loggingSelection_;
	protected int statCurvesSelected_ = 0;
	protected TimeSeries minSeries_ = null;
	protected TimeSeries avgSeries_ = null;
	protected TimeSeries maxSeries_ = null;
	
	protected double minimum_ = 0;
	protected double maximum_ = 0;
	protected double sum_ = 0;
	protected int valueCount_ = 0;
	
	protected Map<JCheckBoxMenuItem, Integer> seriesMap_;
	
	protected static long simulationTime = 0;
	protected static long previousSimulationTime = 0;
	
    static {
    	// Initialize curve colors
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("CURVE_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("CURVE_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("CURVE_COLOR_BLUE") );
        
        COLOR_CURVE = new Color(red, green, blue);
        
        red = Integer.parseInt( Settings.getAttributeValue("CURVE_MIN_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("CURVE_MIN_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("CURVE_MIN_COLOR_BLUE") );
        
        COLOR_MIN = new Color(red, green, blue);
        
        red = Integer.parseInt( Settings.getAttributeValue("CURVE_AVG_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("CURVE_AVG_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("CURVE_AVG_COLOR_BLUE") );
        
        COLOR_AVG = new Color(red, green, blue);

        red = Integer.parseInt( Settings.getAttributeValue("CURVE_MAX_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("CURVE_MAX_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("CURVE_MAX_COLOR_BLUE") );
        
        COLOR_MAX  = new Color(red, green, blue);
    }
    
	
    public GraphPanel()
    {
        this.setLayout( new BoxLayout( this, BoxLayout.X_AXIS) );
        seriesMap_ = new HashMap<JCheckBoxMenuItem, Integer>();
    }
    
    /**
     * 
     */
    public void setChart(Chart chart)
    {
    	chartModel = chart;
    	
        this.removeAll();
        chart_ = SimpleChartFactory.createChart(chartModel.getTypeId(), chartModel.getTitle(),
        		chartModel.getYAxis(), chartModel.getXAxis());
        chartPanel_ = new ChartPanel( chart_ );
//      chartPanel_.setPreferredSize( new Dimension(100, 50) );
        
        loggingSelection_ = new JCheckBox(MainWindow.bundle.getString("CHART_LOG_THIS_CHART"));
        loggingSelection_.addItemListener(new MyLoggingListener());
        
        this.setLayout(new GridBagLayout());

        this.add(chartPanel_, new GridBagConstraints(
        		0, 0, GridBagConstraints.REMAINDER, GridBagConstraints.RELATIVE,
        		1.0, 1.0, GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));
        
        this.add(loggingSelection_, new GridBagConstraints(
        		0, 1, GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER,
        		0.0, 0.0, GridBagConstraints.NORTHEAST, GridBagConstraints.NONE,
        		new Insets(0, 0, 0, 0), 0, 0));
        
        // put min, avg, max selections to popup
        ResourceBundle bundle = MainWindow.bundle;
        minCB_ = new JCheckBoxMenuItem(bundle.getString("CHART_MENU_MIN"));
        avgCB_ = new JCheckBoxMenuItem(bundle.getString("CHART_MENU_AVG"));
        maxCB_ = new JCheckBoxMenuItem(bundle.getString("CHART_MENU_MAX"));
        
        minCB_.setActionCommand(MIN_CMD);
        avgCB_.setActionCommand(AVG_CMD);
        maxCB_.setActionCommand(MAX_CMD);
        
        MyItemListener listener = new MyItemListener();
        minCB_.addItemListener(listener);
        avgCB_.addItemListener(listener);
        maxCB_.addItemListener(listener);

        JMenuItem changeRangeItem = new JMenuItem(bundle.getString("CHART_MENU_RANGE"));
        changeRangeItem.addActionListener(new MyActionListener());

        JPopupMenu popup = chartPanel_.getPopupMenu();
        popup.addSeparator();
        popup.add(maxCB_);
        popup.add(avgCB_);
        popup.add(minCB_);
        popup.addSeparator();
        popup.add(changeRangeItem);
        
        ValueAxis axis = chart_.getXYPlot().getRangeAxis();
        if( chartModel.isAutoScale() )
        {
            axis.setAutoRange( true );
        }
        else
        {
            axis.setAutoRange( false );
            axis.setRange( chartModel.getMinValue(), chartModel.getMaxValue() );
        }

        //chart_.getXYPlot().getDomainAxis().setFixedAutoRange( chartModel.getLength()*1000.0 );
        chart_.getXYPlot().getDomainAxis().setFixedAutoRange( chartModel.getLength() );
        
        float width = Float.parseFloat( Settings.getAttributeValue( "LINE_WIDTH" ) );
                
//        chart_.getXYPlot().getRenderer().setSeriesStroke(
//                0, 
//                new BasicStroke( width )
//            );
        chart_.getXYPlot().getRenderer().setSeriesPaint(0, COLOR_CURVE);
        chart_.getXYPlot().getRenderer().setStroke(new BasicStroke( width ));
        
        loggingSelection_.setSelected(chartModel.isLogged());
    }
    
    public JFreeChart getChart() {
        return chart_;
    }
    
    
    public Chart getChartModel() {
    	return chartModel;
    }
    
        
    public String getLogName() {
    	String name = "";
    	
    	// is a pe chart?
    	Container cpu = SwingUtilities.getAncestorOfClass(PePanel.class, this);
    	if (cpu != null && cpu instanceof PePanel) {
    		PePanel p = (PePanel)cpu;
    		name = p.getProcessingElement().getName() + ": ";
    	}
    	name = name + chart_.getTitle().getText();
    	name = name.replace(" ", "_");
    	name = name.replace(":", "_");
    	name = name.replace(".", "_");
    	name = name.replace("__", "_");
    	name = name +".txt";

    	return name;
    }
    
    
    public void addData( double value )
    {  
//        ((TimeSeriesCollection)chart_.getXYPlot().getDataset()).getSeries(0).addOrUpdate( new Millisecond() , value);
    	//TimeSeriesCollection collection = (TimeSeriesCollection)chart_.getXYPlot().getDataset();
    	
    	//Millisecond ms = new Millisecond();
        //collection.getSeries(0).addOrUpdate( ms , value);
        
        XYSeriesCollection collection = (XYSeriesCollection)chart_.getXYPlot().getDataset();
    	
        long currentTime = simulationTime + previousSimulationTime;
        collection.getSeries(0).addOrUpdate( currentTime , value);
        
        if (statCurvesSelected_ > 0) {
        	double avg = 0.0;
        	if (valueCount_ == 0) {
        		minimum_ = value;
        		maximum_ = value;
        		avg = value;
        		sum_ = value;
        		
        	} else {
        		if (value < minimum_) {
        			minimum_ = value;
        		}
        		if (value > maximum_) {
        			maximum_ = value;
        		}
        		sum_ += value;
        		avg = sum_ / (valueCount_+1);
        	}
        	
        	Integer series = seriesMap_.get(minCB_);
        	if (series != null) {
        		collection.getSeries(series.intValue()).addOrUpdate(currentTime, minimum_);
        	}
        	series = seriesMap_.get(avgCB_);
        	if (series != null) {
        		collection.getSeries(series.intValue()).addOrUpdate(currentTime, avg);
        	}
        	series = seriesMap_.get(maxCB_);
        	if (series != null) {
        		collection.getSeries(series.intValue()).addOrUpdate(currentTime, maximum_);
        	}
        	++valueCount_;
        }
        
        if (chartModel.isLogged()) {
        	try {
            	long time = System.currentTimeMillis();
            	Calendar calendar = Calendar.getInstance();
            	calendar.setTimeInMillis(time);

            	int hour = calendar.get(Calendar.HOUR_OF_DAY);
            	int min = calendar.get(Calendar.MINUTE);
            	int sec = calendar.get(Calendar.SECOND);

            	DecimalFormat format = new DecimalFormat("00"); // two digits

            	StringBuilder xvalue = new StringBuilder();
            	xvalue.append(format.format(hour));
            	xvalue.append(":");
            	xvalue.append(format.format(min));
            	xvalue.append(":");
            	xvalue.append(format.format(sec));

        		ChartDataLogger.getInstance().logData(this, xvalue.toString(), ""+value);
        		
        	} catch (IOException ioe) {
        		String eMsg = ChartDataLogger.getInstance().getErrorMessage();
        		if (eMsg != null) {
        			MainWindow.getInstance().stopLogging(eMsg);
        		} else {
        			MainWindow.getInstance().stopLogging(ioe.getMessage());
        		}
        	}
        }
    }
    
    public void addMarker() {
    	final XYPlot plot = chart_.getXYPlot();
   	 	final Marker configChanged = new ValueMarker(simulationTime + previousSimulationTime);
   	 	configChanged.setPaint(Color.RED);
   	 	plot.addDomainMarker(configChanged);
    }
    
    public static void setSimulationTime(long time) {
    	if (time < previousSimulationTime) {
        	simulationTime += previousSimulationTime;
        }
        previousSimulationTime = time;
    }
    
    private class MyActionListener implements ActionListener {
    	public void actionPerformed(ActionEvent e) {
    		String msg = MainWindow.bundle.getString("CHART_RANGE_TEXT");
    		String title = MainWindow.bundle.getString("CHART_RANGE_TITLE");
    		
    		int length= 0;
			String answer = null;
    		
    		while (true) {
    			answer = JOptionPane.showInputDialog(chartPanel_, msg, title, JOptionPane.QUESTION_MESSAGE);
    			if (answer == null) {
    				return;  // cancel pressed
    			}
    			try {
    				length = Integer.parseInt(answer);
    			} catch (NumberFormatException nfe) {
    				continue;  // value was not an integer
    			}
    			break; // a legal value received
    		}
            chart_.getXYPlot().getDomainAxis().setFixedAutoRange( length*1000.0 );
            chartModel.setLength(length);
    	}
    }
    
    private class MyItemListener implements ItemListener {
    	public void itemStateChanged(ItemEvent e) {
//    		TimeSeriesCollection seriesCollection = (TimeSeriesCollection)chart_.getXYPlot().getDataset();
            XYSeriesCollection seriesCollection = (XYSeriesCollection)chart_.getXYPlot().getDataset();
			XYItemRenderer renderer = chart_.getXYPlot().getRenderer();
			
    		if (e.getStateChange() == ItemEvent.SELECTED) {
    			if (statCurvesSelected_ == 0) {
    				// first item that was selected
    				
    				// Create series for containing data of stat. curves
//    		        TimeSeries series = new TimeSeries( "", Millisecond.class );
    		        XYSeries series = new XYSeries( "1" );

    		        if (SimpleChartFactory.historySize > 0) { 
//    		            series.setHistoryCount(SimpleChartFactory.historySize);
    		            series.setMaximumItemCount(SimpleChartFactory.historySize);
    		        }
    		        seriesCollection.addSeries(series);
    		        seriesMap_.put(minCB_, new Integer(1));
    		        
//    		        series = new TimeSeries( "", Millisecond.class );
    		        series = new XYSeries( "2" );
    		        
    		        if (SimpleChartFactory.historySize > 0) { 
//    		            series.setHistoryCount(SimpleChartFactory.historySize);
    		            series.setMaximumItemCount(SimpleChartFactory.historySize);
    		        }
    		        seriesCollection.addSeries(series);
    		        seriesMap_.put(avgCB_, new Integer(2));

//    		        series = new TimeSeries( "", Millisecond.class );
    		        series = new XYSeries( "3" );
    		        
    		        if (SimpleChartFactory.historySize > 0) { 
//    		            series.setHistoryCount(SimpleChartFactory.historySize);
    		            series.setMaximumItemCount(SimpleChartFactory.historySize);
    		        }
    		        seriesCollection.addSeries(series);
    		        seriesMap_.put(maxCB_, new Integer(3));
    		        
    		        // set curves initially invisible
    		        renderer.setSeriesVisible(1, false);
    		        renderer.setSeriesVisible(2, false);
    		        renderer.setSeriesVisible(3, false);
    		        renderer.setSeriesPaint(1, COLOR_MIN);
    		        renderer.setSeriesPaint(2, COLOR_AVG);
    		        renderer.setSeriesPaint(3, COLOR_MAX);
    			}

    			Integer series = seriesMap_.get(e.getSource());
    			if (series != null) {
    				renderer.setSeriesVisible(series.intValue(), true);
    			}
				++statCurvesSelected_;
    			
    		} else if (e.getStateChange() == ItemEvent.DESELECTED){
				--statCurvesSelected_;

				Integer series = seriesMap_.get(e.getSource());
    			if (series != null) {
    				renderer.setSeriesVisible(series.intValue(), false);
    			}
    			
    			// are all deselected now?
    			if (statCurvesSelected_ == 0) {
    				seriesMap_.clear();
    				seriesCollection.removeSeries(3);
    				seriesCollection.removeSeries(2);
    				seriesCollection.removeSeries(1);
    				
    				valueCount_ = 0;
    			}
    		}
    	}
    }
    
    private class MyLoggingListener implements ItemListener {
    	public void itemStateChanged(ItemEvent e) {
    		chartModel.setLogged(loggingSelection_.isSelected());
    	}
    }
}
