/**
 * CVS:
 * $Id: SignalsPanel.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    SignalsPanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 20.11.2006
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui;

import java.awt.BorderLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingConstants;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;

import fi.cpu.data.Chart;
import fi.cpu.data.Configuration;
import fi.cpu.table.SignalsTableHeaderRenderer;
import fi.cpu.table.SignalsTableModel;
import fi.cpu.ui.graph.GraphPanel;
import fi.cpu.ui.graph.SimpleChartFactory;


/**
 * SignalsPanel is a panel for examining the amount
 * of signals sent from a process to another.
 */
public class SignalsPanel extends JPanel {
	protected SignalsPanel self;
	protected JPopupMenu popupMenu;
	protected SignalsTableModel tableModel;
	protected JTable tableView;
	protected SignalsTableHeaderRenderer headerRenderer;
	protected int popupRow = 0;
	protected int popupColumn = 0;
	
	
	/**
	 * Creates a new SignalsPanel.
	 */
	public SignalsPanel(Configuration config) {
		super();
		this.self = this;
		
		tableModel = new SignalsTableModel(config);
		tableModel.addTableModelListener(new MyTableModelListener());

		headerRenderer = new SignalsTableHeaderRenderer(tableModel);
	    headerRenderer.setHorizontalAlignment(SwingConstants.CENTER);
	    
	    DefaultTableCellRenderer tableCellRenderer = new DefaultTableCellRenderer();
	    tableCellRenderer.setHorizontalAlignment(SwingConstants.RIGHT);
		
		tableView = new JTable(tableModel) {
			public String getToolTipText(MouseEvent e) {
				Point p = e.getPoint();
				int rowIndex = rowAtPoint(p);
				int columnIndex = columnAtPoint(p);
				
				if (columnIndex < 1) {
					return null;
				}
				
				StringBuilder text = new StringBuilder("From \"");
				text.append(getValueAt(rowIndex, 0));
				text.append("\" to \"");
				text.append(getValueAt(columnIndex-1, 0));
				text.append("\": ");
				text.append(getValueAt(rowIndex, columnIndex));
				return text.toString();
			}
		};
		tableView.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		tableView.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		tableView.getTableHeader().setReorderingAllowed(false);
		tableView.getTableHeader().setDefaultRenderer(headerRenderer);
		tableView.setDefaultRenderer(Object.class, tableCellRenderer);
		tableView.addMouseListener(new MyMouseListener());

	    JScrollPane scrollPane = new JScrollPane(tableView,
	    		ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
	    		ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
	    
	    this.setLayout(new BorderLayout());
	    this.add(scrollPane, BorderLayout.CENTER);
	    
		popupMenu = new JPopupMenu();
		JMenuItem showChartItem = new JMenuItem(MainWindow.bundle.getString("SHOW_GRAPH"));
		showChartItem.addActionListener(new MyActionListener());
		popupMenu.add(showChartItem);
	}
	

	/**
	 * Returns the tableModel.
	 */
	public SignalsTableModel getTableModel() {
		return tableModel;
	}

	
	/**
	 * Listener for TableModelEvents.
	 */
	private class MyTableModelListener implements TableModelListener {
		public void tableChanged(TableModelEvent e) {
			// This implementation assumes that the table changes only
			// in two ways: eiher the whole structure is reorganized or
			// a single value cell is updated.
			int firstRow = e.getFirstRow();
			if (firstRow == TableModelEvent.HEADER_ROW) {
				// The table header has changed. Renderer for the
				// header column (first column) needs to be set again.
				TableColumn column = tableView.getColumnModel().getColumn(0);
				column.setCellRenderer(headerRenderer);
				
			} else if (e.getColumn() != 0 && e.getColumn() != TableModelEvent.ALL_COLUMNS) {
				// The change conserned a single value cell
				String source = tableModel.getValueAt(firstRow, 0).toString();
				String dest = tableModel.getColumnName(e.getColumn());
				Object value = tableModel.getValueAt(firstRow, e.getColumn());
				
				if (value instanceof Number) {
					Number n = (Number) value;
					
					String graphID = source +"_to_" + dest;
					MainWindow.getInstance().addValueToChart(
							null, graphID, n.doubleValue());
				}
			}

			self.repaint();
		}
	}

	
	/**
	 * Listener for MouseEvents to post the popup menu
	 */
	private class MyMouseListener extends MouseAdapter {
		public void mousePressed(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		public void mouseReleased(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		private void showPopupIfTriggered(MouseEvent e) {
			if (e.isPopupTrigger()) {
				Point p = e.getPoint();
				popupRow = tableView.rowAtPoint(p);
				popupColumn = tableView.columnAtPoint(p);
				if (popupColumn > 0 && popupRow >= 0) {
					popupMenu.show(e.getComponent(), e.getX(), e.getY());
				}
			}			
		}
	}

	
	/**
	 * Listener for actions from the popup menu.
	 */
	private class MyActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String fromProcess = (String)tableView.getValueAt(popupRow, 0);
			String toProcess = tableView.getColumnName(popupColumn);
			String graphID = fromProcess +"_to_" + toProcess;
			String title = "Signals from " + fromProcess + " to " + toProcess;
			int historySize = 0;
			
			try {
				String history = System.getProperty("history");
				if (history != null) {
					historySize = Integer.parseInt(history);
				}
			} catch (NumberFormatException nfe) {
				// do nothing
			}
			
			GraphPanel gPanel = new GraphPanel();
			Chart chart = new Chart(graphID.toString(), SimpleChartFactory.LINECHART_TYPE,
					title.toString(), MainWindow.bundle.getString("XAXIS"), "", 0.0, 0.0,
					historySize, true, false);
			gPanel.setChart(chart);
			MainWindow.getInstance().addChart( gPanel, 10, 10, 200, 300 );
		}
	}
}
