/**
 * File:    ChartCreationDialog.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 9.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SpringLayout;

import fi.cpu.data.Chart;
import fi.cpu.ui.graph.SimpleChartFactory;
import fi.cpu.ui.xml.SystemConfigMarshaller;
import fi.cpu.xml.bind.systemconfig.GraphType;
import fi.cpu.xml.bind.systemconfig.SupportedGraphsType;
import fi.cpu.xml.bind.systemconfig.SystemConfigType;


/**
 * ChartCreationDialog implements a dialog window for
 * creating charts.
 */
public class ChartCreationDialog extends JDialog {
    private static final String CPU_GRAPHS = "/cpugraphconfiguration.xml";
    private static final String NORMAL_GRAPHS = "/graphconfiguration.xml";
    protected static ChartCreationDialog self;

    protected String[] graphNames;
    protected String[] graphIDs;
    protected String[] cpuGraphNames;
    protected String[] cpuGraphIDs;
    
    private JLabel nameLabel;
    private JLabel typeLabel;
    private JLabel lengthLabel;
    private JLabel valueLabel;
    private JLabel scalingLabel;
    protected JComboBox nameCoB;
    protected JComboBox typeCoB;
    protected JTextField valueTF;
    protected JTextField maxValueTF;
    protected JTextField minValueTF;
    protected JTextField lengthTF;
    protected JRadioButton automaticRB;
    protected JRadioButton selectedRangeRB;
    protected JButton cancelButton;
    protected JButton okButton;
    
    protected boolean cpuSpecific = false;
    protected Chart createdChart;
    private MyActionListener actionListener = new MyActionListener();
    
    
    /**
     * @return An instance of ChartCreationDialog.
     */
    public static ChartCreationDialog getInstance() {
        if ( self == null ) {
            self = new ChartCreationDialog();
        }

        return self;
    }

    
    /**
     * Creates a new ChartCreationDialog.
     */
    private ChartCreationDialog() {
        super(MainWindow.getInstance(), true);
        self = this;
        initGraphNames();
        init();
    }

    
    /**
     * Reads graph names and ids from a file.
     */
    private void initGraphNames() {

        try {
        	SystemConfigMarshaller m = new SystemConfigMarshaller();
        	
        	// Other than pe spefic graphs
        	SystemConfigType systemConfig = m.loadSystemConfig(
        			this.getClass().getResourceAsStream(NORMAL_GRAPHS));
        	
        	graphNames = new String[ systemConfig.getSupportedGraphs().size() ];
        	graphIDs = new String[ graphNames.length ];
        	        	
        	int index = 0;
        	
        	Iterator graphs = systemConfig.getSupportedGraphs().iterator();
        	while( graphs.hasNext() ) {
        		SupportedGraphsType graphType = (SupportedGraphsType)graphs.next();
        		
        		GraphType graph = graphType.getGraph();
        		graphNames[index] = graph.getName();
        		graphIDs[index] = graph.getId();
        		
        		index++;                   
        	}
        	
        	// Pe specific graphs
        	systemConfig = m.loadSystemConfig(
        			this.getClass().getResourceAsStream(CPU_GRAPHS));
        	
        	cpuGraphNames = new String[ systemConfig.getSupportedGraphs().size() ];
        	cpuGraphIDs = new String[ cpuGraphNames.length ];
        	
        	index = 0;
        	
        	graphs = systemConfig.getSupportedGraphs().iterator();
        	while( graphs.hasNext() ) {
        		SupportedGraphsType graphType = (SupportedGraphsType)graphs.next();
        		
        		GraphType graph = graphType.getGraph();
        		cpuGraphNames[index] = graph.getName();
        		cpuGraphIDs[index] = graph.getId();              
        		index++;                   
        	}
        	
        } catch ( Exception e ) {
        	e.printStackTrace();
        }
    }
    
    
    /**
     *  Initializes the dialog view.
     */
    private void init() {
        this.getContentPane().setLayout( new SpringLayout() );
        this.setTitle(MainWindow.bundle.getString("CHART_DIALOG_TITLE"));
        this.setResizable(false);

        nameLabel = new JLabel( MainWindow.bundle.getString( "NAME_LABEL" ) );
        typeLabel = new JLabel( MainWindow.bundle.getString( "TYPE_LABEL" ) );
        lengthLabel = new JLabel( MainWindow.bundle.getString( "LENGTH_LABEL" ) );
        valueLabel = new JLabel( MainWindow.bundle.getString( "VALUE_LABEL" ) );
        scalingLabel = new JLabel( MainWindow.bundle.getString( "SCALING_LABEL" ) );

        this.getContentPane().add( nameLabel );
        this.getContentPane().add( typeLabel );
        this.getContentPane().add( lengthLabel );
        this.getContentPane().add( valueLabel );
        this.getContentPane().add( scalingLabel );

        SpringLayout sLayout = (SpringLayout) this.getContentPane().getLayout();

        //Eka 10 pixeli alaspin
        sLayout.putConstraint( SpringLayout.NORTH, nameLabel, 10,
                SpringLayout.NORTH, this.getContentPane() );

        //Kaikki 10 pixeli vasemmalta
        sLayout.putConstraint( SpringLayout.WEST, nameLabel, 10, SpringLayout.WEST,
                this.getContentPane() );
        sLayout.putConstraint( SpringLayout.WEST, typeLabel, 10, SpringLayout.WEST,
                this.getContentPane() );
        sLayout.putConstraint( SpringLayout.WEST, lengthLabel, 10,
                SpringLayout.WEST, this.getContentPane() );
        sLayout.putConstraint( SpringLayout.WEST, valueLabel, 10,
                SpringLayout.WEST, this.getContentPane() );
        sLayout.putConstraint( SpringLayout.WEST, scalingLabel, 10,
                SpringLayout.WEST, this.getContentPane() );

        //Ja alekkain 15 pixelin vlein
        sLayout.putConstraint( SpringLayout.NORTH, typeLabel, 25,
                SpringLayout.SOUTH, nameLabel );
        sLayout.putConstraint( SpringLayout.NORTH, lengthLabel, 15,
                SpringLayout.SOUTH, typeLabel );
        sLayout.putConstraint( SpringLayout.NORTH, valueLabel, 15,
                SpringLayout.SOUTH, lengthLabel );
        sLayout.putConstraint( SpringLayout.NORTH, scalingLabel, 10,
                SpringLayout.SOUTH, valueLabel );

        nameCoB = new JComboBox( graphNames );
        typeCoB = new JComboBox( SimpleChartFactory.getChartTypes() );
        lengthTF = new JTextField( 8 );
        valueTF = new JTextField( 8 );

        this.getContentPane().add( nameCoB );
        this.getContentPane().add( typeCoB );
        this.getContentPane().add( lengthTF );
        this.getContentPane().add( valueTF );

        sLayout.putConstraint( SpringLayout.NORTH, nameCoB, 10,
                SpringLayout.NORTH, this.getContentPane() );

        //value is widest
        sLayout.putConstraint( SpringLayout.WEST, nameCoB, 10,
                SpringLayout.EAST, valueLabel );

        sLayout.putConstraint( SpringLayout.WEST, typeCoB, 10,
                SpringLayout.EAST, valueLabel );
        sLayout.putConstraint( SpringLayout.NORTH, typeCoB, 10,
                SpringLayout.SOUTH, nameCoB );

        sLayout.putConstraint( SpringLayout.WEST, lengthTF, 10,
                SpringLayout.EAST, valueLabel );
        sLayout.putConstraint( SpringLayout.NORTH, lengthTF, 10,
                SpringLayout.SOUTH, typeCoB );

        sLayout.putConstraint( SpringLayout.WEST, valueTF, 10,
                SpringLayout.EAST, valueLabel );
        sLayout.putConstraint( SpringLayout.NORTH, valueTF, 10,
                SpringLayout.SOUTH, lengthTF );

        ButtonGroup group = new ButtonGroup();

        automaticRB = new JRadioButton( MainWindow.bundle
                .getString( "AUTOMATIC_RADIO" ) );
        automaticRB.setSelected( true );
        automaticRB.addActionListener( actionListener );

        selectedRangeRB = new JRadioButton();
        selectedRangeRB.addActionListener( actionListener );

        group.add( automaticRB );
        group.add( selectedRangeRB );

        this.getContentPane().add( automaticRB );
        this.getContentPane().add( selectedRangeRB );

        sLayout.putConstraint( SpringLayout.WEST, automaticRB, 10,
                SpringLayout.EAST, valueLabel );
        sLayout.putConstraint( SpringLayout.NORTH, automaticRB, 10,
                SpringLayout.SOUTH, valueTF );

        sLayout.putConstraint( SpringLayout.WEST, selectedRangeRB, 10,
                SpringLayout.EAST, valueLabel );
        sLayout.putConstraint( SpringLayout.NORTH, selectedRangeRB, 20,
                SpringLayout.SOUTH, automaticRB );

        JLabel max = new JLabel( MainWindow.bundle.getString( "MAX" ) );
        JLabel min = new JLabel( MainWindow.bundle.getString( "MIN" ) );

        maxValueTF = new JTextField( 5 );
        maxValueTF.setEditable( false );
        minValueTF = new JTextField( 5 );
        minValueTF.setEditable( false );

        this.getContentPane().add( max );
        this.getContentPane().add( min );

        sLayout.putConstraint( SpringLayout.WEST, max, 10, SpringLayout.EAST,
                selectedRangeRB );
        sLayout.putConstraint( SpringLayout.NORTH, max, 10, SpringLayout.SOUTH,
                automaticRB );

        sLayout.putConstraint( SpringLayout.WEST, min, 10, SpringLayout.EAST,
                selectedRangeRB );
        sLayout.putConstraint( SpringLayout.NORTH, min, 10, SpringLayout.SOUTH,
                max );

        this.getContentPane().add( maxValueTF );
        this.getContentPane().add( minValueTF );

        sLayout.putConstraint( SpringLayout.WEST, maxValueTF, 10,
                SpringLayout.EAST, max );
        sLayout.putConstraint( SpringLayout.NORTH, maxValueTF, 10,
                SpringLayout.SOUTH, automaticRB );

        sLayout.putConstraint( SpringLayout.WEST, minValueTF, 10,
                SpringLayout.EAST, max );
        sLayout.putConstraint( SpringLayout.NORTH, minValueTF, 10,
                SpringLayout.SOUTH, maxValueTF );

        cancelButton = new JButton( MainWindow.bundle.getString( "CANCEL" ) );
        cancelButton.addActionListener( actionListener );
        okButton = new JButton( MainWindow.bundle.getString( "OK" ) );
        okButton.addActionListener( actionListener );

        this.getContentPane().add( okButton );
        this.getContentPane().add( cancelButton );

        sLayout.putConstraint( SpringLayout.SOUTH, okButton, -10,
                SpringLayout.SOUTH, this.getContentPane() );

        sLayout.putConstraint( SpringLayout.SOUTH, cancelButton, -10,
                SpringLayout.SOUTH, this.getContentPane() );

        sLayout.putConstraint( SpringLayout.EAST, cancelButton, -10, SpringLayout.EAST,
                this.getContentPane() );

        sLayout.putConstraint( SpringLayout.EAST, okButton, -10,
                SpringLayout.WEST, cancelButton );

        this.setSize( 250, 300 );
    }


    /**
     * @param yes Whether to show a list of pe specific graphs of other graphs.
     */
    public void setCPUspecifig(boolean yes) {
        cpuSpecific = yes;
        
        if( cpuSpecific ) {
            nameCoB.removeAllItems();    
            for( int i=0; i < cpuGraphNames.length; i++ ) {
                nameCoB.addItem( cpuGraphNames[i] );
            }
            
        } else {
            nameCoB.removeAllItems();    
            for( int i=0; i < graphNames.length; i++ )
            {
                nameCoB.addItem( graphNames[i] );
            }
        }
    }
    
    
    /**
     * Shows the dialog.
     * @return The created chart, or null if canceled.
     */
    public Chart showDialog() {
    	setVisible(true);
    	return createdChart;
    }
    
    
    /**
     * Listener for ActionEvents.
     */
    private class MyActionListener implements ActionListener {
        public void actionPerformed( ActionEvent e ) {
            Object source = e.getSource();

            if ( source.equals( cancelButton ) ) {
            	createdChart = null;
                self.dispose();
                
            } else if ( source.equals( okButton ) ) {
            	// Create graph panel
                String name = (String) nameCoB.getSelectedItem();
                String graphID = "";
                
                if( cpuSpecific ) {
                	graphID =  cpuGraphIDs[ nameCoB.getSelectedIndex() ];
                } else {
                	graphID = graphIDs[ nameCoB.getSelectedIndex() ];
                }

                int typeID = typeCoB.getSelectedIndex();
                double length = 0;
                double maxValue = 0.0;
                double minValue = 0.0;

                try {
                    length = Double.parseDouble( lengthTF.getText() );

                    if ( selectedRangeRB.isSelected() ) {
                        maxValue = Double.parseDouble( maxValueTF.getText() );
                        minValue = Double.parseDouble( minValueTF.getText() );
                    }

                } catch ( NumberFormatException exp ) {
                    JOptionPane.showMessageDialog( self, MainWindow.bundle
                            .getString( "ERROR_NUMBER_TEXT" ), MainWindow.bundle
                            .getString( "ERROR_NUMBER_TITLE" ),
                            JOptionPane.ERROR_MESSAGE );
                    return;
                }
                
                String label = valueTF.getText();

                createdChart = new Chart(graphID, typeID, name, MainWindow.bundle.getString("XAXIS"),
                		label, minValue, maxValue, length, automaticRB.isSelected(), false);
                self.dispose();
                
            } else if ( source.equals( automaticRB ) ) {
                maxValueTF.setEditable( false );
                minValueTF.setEditable( false );

            } else if ( source.equals( selectedRangeRB ) ) {
                maxValueTF.setEditable( true );
                minValueTF.setEditable( true );
            }
        }
    }
}
