/**
 * File:    ReportTableCellRenderer.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 27.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.table;

import java.awt.Component;
import java.awt.Font;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.DefaultTableCellRenderer;

import fi.cpu.Settings;


/**
 * Renderer for other than header cells in a report table.
 */
public class ReportTableCellRenderer extends DefaultTableCellRenderer {
    private static final int FRACTION_DIGITS =
    	Integer.parseInt(Settings.getAttributeValue("REPORT_FRACTION_DIGITS"));

    private DecimalFormat doubleFormatter;
	private DecimalFormat intFormatter;
    private Font sumRowFont;
    private Font defaultFont;
	
    
	/**
	 * Creates a new ReportTableCellRenderer
	 */
	public ReportTableCellRenderer(Font defaultFont, Font sumRowFont) {
		super();
		this.defaultFont = defaultFont;
		this.sumRowFont = sumRowFont;
		
		// Create formatters to show numbers in the view nicely
		DecimalFormatSymbols symbols = new DecimalFormatSymbols();
		symbols.setDecimalSeparator('.');

		doubleFormatter = new DecimalFormat();
		doubleFormatter.setDecimalFormatSymbols(symbols);
		doubleFormatter.setMaximumFractionDigits(FRACTION_DIGITS);
		doubleFormatter.setMinimumFractionDigits(FRACTION_DIGITS);
		
		intFormatter = new DecimalFormat();
		intFormatter.setDecimalFormatSymbols(symbols);
		intFormatter.setMaximumFractionDigits(0);
		intFormatter.setMinimumFractionDigits(0);
	}
	
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
	 */
	public Component getTableCellRendererComponent(
			JTable table, Object value, boolean isSelected,
			boolean hasFocus, int row, int column) {
		
		// Format the cell value
		if (value instanceof Integer) {
			Integer intVal = (Integer) value;
			value = intFormatter.format(intVal.longValue());
			
		} else if (value instanceof Double) {
			Double doubleVal = (Double) value;
			value = doubleFormatter.format(doubleVal.doubleValue());
		}
		
		Component comp = super.getTableCellRendererComponent(table, value,
				false, hasFocus, row, column);

		// Do alignment
		if (comp instanceof JLabel) {
			JLabel label = (JLabel)comp;
			if (column == 0) {
				label.setHorizontalAlignment(SwingConstants.LEFT);
			} else {
				label.setHorizontalAlignment(SwingConstants.RIGHT);
			}				
		}
		
		// Set cell font
		if (row == table.getRowCount()-1) {
			comp.setFont(sumRowFont);
		} else {
			comp.setFont(defaultFont);
		}

		return comp;
	}
}
