/*
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package fi.cpu.data;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.text.DecimalFormat;
import java.util.Calendar;

import fi.cpu.handler.Handler;
import fi.cpu.ui.MainWindow;

public class TraceDataLogger {
	private static TraceDataLogger self;
	private File parentDirectory;
	private File logDirectory;
	private PrintStream stream;
	private Handler connectionHandler;
	private String errorMessage;
	public enum TracingStatus {STOPPED, TRACING, WAITING_FOR_STOP};
	private TracingStatus status; 

	
	public static TraceDataLogger getInstance() {
		if (self == null) {
			self = new TraceDataLogger();
		}
		return self;
	}
	
	
	/**
	 * Creates a new TraceDataLogger.
	 */
	public TraceDataLogger() {
		status = TracingStatus.STOPPED;
	}
	
	
	/**
	 * Sets the directory where trace files are created.
	 */
	public void setTraceDirectory(File directory) throws IOException {
		directory.mkdir();
		parentDirectory = directory;
	}
	
	
	/**
	 * Returns the directory where trace files are created.
	 */
	public File getTraceDirectory() {
		return parentDirectory;
	}
	
	
	/**
	 * Returns the message of an error during tracing. If no error happened,
	 * null returned.
	 */
	public String getErrorMessage() {
		return errorMessage;
	}
	
	
	/**
	 * Begins tracing.
	 */
	public File startTracing() throws IOException {
		// Get current date and time
    	long time = System.currentTimeMillis();
    	Calendar calendar = Calendar.getInstance();
    	calendar.setTimeInMillis(time);

    	int year = calendar.get(Calendar.YEAR);
    	int month = calendar.get(Calendar.MONTH);
    	int day = calendar.get(Calendar.DATE);
    	int hour = calendar.get(Calendar.HOUR_OF_DAY);
    	int min = calendar.get(Calendar.MINUTE);
    	int sec = calendar.get(Calendar.SECOND);

    	DecimalFormat format = new DecimalFormat("00"); // two digits
    	
    	StringBuilder name = new StringBuilder();
    	name.append(year);
    	name.append(format.format(month));
    	name.append(format.format(day));
    	name.append("_");
    	name.append(format.format(hour));
    	name.append(format.format(min));
    	name.append(format.format(sec));

    	logDirectory = new File(parentDirectory, name.toString());
    	logDirectory.mkdir();
		errorMessage = null;
		
		// Get connection handler and ask device to start tracing
		if (connectionHandler == null) {
			connectionHandler = MainWindow.getInstance().getConnectionHandler();
		}
		try {
			connectionHandler.writeToDestination("<start_trace>");
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		status = TracingStatus.TRACING;
		
		return logDirectory;
	}
	
	
	/**
	 * Stops tracing.
	 */
	public void stopTracing() {	
		// Ask device to stop tracing and send rest log messages
		if (connectionHandler != null) {
			try {
				connectionHandler.writeToDestination("<stop_trace>");
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		
		status = TracingStatus.WAITING_FOR_STOP;
	}
	
	
	/**
	 * All trace log messages has been received.
	 */
	public void traceReady() {
		if (status == TracingStatus.WAITING_FOR_STOP) {
			
			// print file footer
			stream.println("</log>");
			
			// Close opened stream
			if (stream != null) {
				stream.close();
				stream = null;
			}
	
			// Release connection handler
			connectionHandler = null;
			logDirectory = null;
			
			status = TracingStatus.STOPPED;
		}
	}
	
	
	/**
	 * Returns status of tracing.
	 */
	public TracingStatus getStatus() {
		return status;
	}
	
	
	/**
	 * Adds trace data to log.
	 */
	public void logTraceData(String data) throws IOException {
		if (status == TracingStatus.STOPPED || logDirectory == null) {
			return;
		}
		
		try {	
			if (stream == null) {
				// create new stream
				File traceFile = new File(logDirectory, "Execution_trace.xml");
				if (traceFile.exists()) {
					traceFile.delete();
				}
				
				traceFile.createNewFile();
				stream = new PrintStream(traceFile);
				
				// print file header
				stream.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
				stream.println("<log>");
			}
			stream.println(data);

		} catch (IOException e) {
			errorMessage = "Couldn't log trace data to: "+logDirectory.getAbsolutePath()+": "+e.getMessage();
			throw e;
		}
	}
}
