/**
 * File:    Process.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 23.2.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import fi.cpu.event.ModelNodeEvent;


/**
 * Class for process information.
 */
public class Process extends ModelNode implements Comparable<Process> {
	public static final int DATA_CHANGED_EVENT = 1;
	private Integer id;
	private String name;
	private int execCount;
	private int avgExecTime;
	private int totExecTime;
	private int avgCommTime;
	private int totCommTime;
	private double relCommTime;
	private int localCommTime;
	private int localCommBytes;
	private int localCommPenalty;
	private int remoteCommTime;
	private int remoteCommBytes;
	private int remoteCommPenalty;
	private int sigInCount;
	private int sigOutCount;
	private int sigQueue;
	private double latency;
	private double responseTime;
	
	
	/**
	 * Creates a new process.
	 */
	public Process(Integer id, String name) {
		super(name);
		this.id = id;
		this.name = name;
		initData();
	}


	/**
	 * Returns the name.
	 */
	public String getName() {
		return name;
	}


	/**
	 * Sets the name.
	 */
	public void setName(String name) {
		this.name = name;
	}


	/**
	 * Returns the id.
	 */
	public Integer getId() {
		return id;
	}


	/**
	 * Sets the object to initial state.
	 */
	public void initData() {
		execCount = 0;
		avgExecTime = 0;
		totExecTime = 0;
		avgCommTime = 0;
		totCommTime = 0;
		relCommTime = 0;
		localCommTime = 0;
		localCommBytes = 0;
		localCommPenalty = 0;
		remoteCommTime = 0;
		remoteCommBytes = 0;
		remoteCommPenalty = 0;
		sigInCount = 0;
		sigOutCount = 0;
	}
	
	
	/**
	 * Adds new execution data values.
	 */
	public void addExecDataPoint(int eCount,
								 int eTime,
								 int lcTime,
								 int lcBytes,
								 int rcTime,
								 int rcBytes,
								 int sQueue,
								 double respTime,
								 double lat) {
		execCount = eCount;
		totExecTime = eTime;
		totCommTime = lcTime + rcTime;
		localCommTime = lcTime;
		localCommBytes = lcBytes;
		remoteCommTime = rcTime;
		remoteCommBytes = rcBytes;
		sigQueue = sQueue;
		if (respTime != 0) {
			responseTime = respTime;
		}
		if (latency != 0) {
			latency = lat;
		}
			
		if (execCount != 0) {
			avgExecTime = totExecTime / execCount;
			avgCommTime = totCommTime / execCount;
			relCommTime = (double) totCommTime / (double) totExecTime * 100;
		} else {
			avgExecTime = 0;
			avgCommTime = 0;
			relCommTime = 0;
		}
		
		if (localCommBytes != 0) {
			localCommPenalty = localCommTime / localCommBytes;
		} else {
			localCommPenalty = 0;
		}
		
		if (remoteCommBytes != 0) {
			remoteCommPenalty = remoteCommTime / remoteCommBytes;
		} else {
			remoteCommPenalty = 0;
		}
		
		fireModelNodeChanged(new ModelNodeEvent(this, DATA_CHANGED_EVENT));
	}
	
	/**
	 * Clears signal count values.
	 */
	public void clearSignalCount() {
		sigInCount = 0;
		sigOutCount = 0;
	}

	/**
	 * Returns the execution count.
	 */
	public int getExecCount() {
		return execCount;
	}

	/**
	 * Returns the average execution time.
	 */
	public int getAvgExecTime() {
		return avgExecTime;
	}
	
	/**
	 * Returns the total execution time.
	 */
	public int getTotExecTime() {
		return totExecTime;
	}
	
	/**
	 * Returns the average communication time.
	 */
	public int getAvgCommTime() {
		return avgCommTime;
	}
	
	/**
	 * Returns the total communication time.
	 */
	public int getTotCommTime() {
		return totCommTime;
	}
	
	/**
	 * Returns the relative communication time.
	 */
	public double getRelCommTime() {
		return relCommTime;
	}
	
	/**
	 * Returns local communication time.
	 */
	public int getLocalCommTime() {
		return localCommTime;
	}
	
	/**
	 * Returns local communication bytes.
	 */
	public int getLocalCommBytes() {
		return localCommBytes;
	}
	
	/**
	 * Returns local communication penalty.
	 */
	public int getLocalCommPenalty() {
		return localCommPenalty;
	}
	
	/**
	 * Returns remote communication time.
	 */
	public int getRemoteCommTime() {
		return remoteCommTime;
	}
	
	/**
	 * Returns remote communication bytes.
	 */
	public int getRemoteCommBytes() {
		return remoteCommBytes;
	}
	
	/**
	 * Returns remote communication penalty.
	 */
	public int getRemoteCommPenalty() {
		return remoteCommPenalty;
	}
	
	/**
	 * Returns count of input signals.
	 */
	public int getSignalsInCount() {
		return sigInCount;
	}
	
	/**
	 * Sets count of input signals.
	 */
	public void setSignalsInCount(int value) {
		sigInCount = value;
	}
	
	/**
	 * Returns count of output signals.
	 */
	public int getSignalsOutCount() {
		return sigOutCount;
	}
	
	/**
	 * Sets count of output signals.
	 */
	public void setSignalsOutCount(int value) {
		sigOutCount = value;
	}
	
	/**
	 * Returns the sigQueue.
	 * @return the sigQueue
	 */
	public int getSigQueue() {
		return sigQueue;
	}

	/**
	 * Returns the latency.
	 * @return the latency
	 */
	public double getLatency() {
		return latency;
	}

	/**
	 * Returns the responseTime.
	 * @return the responseTime
	 */
	public double getResponseTime() {
		return responseTime;
	}


	/**
	 * Implements the Compararable-interface
	 */
	public int compareTo(Process o) {
		return name.compareTo(o.name);
	}
}
