/**
 * File:    ModelNode.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 29.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import javax.swing.event.EventListenerList;
import javax.swing.tree.DefaultMutableTreeNode;

import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;


/**
 * ModelNode represent one node in the model hierarchy.
 */
public class ModelNode extends DefaultMutableTreeNode {
	private EventListenerList listenerList;

	/**
	 * Creates a new ModelNode.
	 * @param name The name of the node.
	 */
	public ModelNode(String name) {
		super(name);
		listenerList = new EventListenerList();
	}
	
	
	/**
	 * @return Returns the name.
	 */
	public String getName() {
		return getUserObject().toString();
	}

	
	/**
	 * Sets the name.
	 * @param name New name.
	 */
	public void setName(String name) {
		setUserObject(name);
	}
	
	
	/**
	 * @return Returns true if node is model root.
	 */
	public boolean isModelRoot() {
		return isRoot();
	}
	
	
	/**
	 * @return Returns the parentNode.
	 */
	public ModelNode getParentNode() {
		return (ModelNode)getParent();
	}
	
	
	/**
	 * @return Returns the path from model root to this node, including end points.
	 */
	public ModelNode[] getModelPath() {
		return (ModelNode[])getPath();
	}
	
	
	/**
	 * Called when a child has been added to this node.
	 */
	public void childAdded(ModelNode child) {
		// Do nothing. Child classes override this method if they want.
	}
	
	
	/**
	 * Called when a child has been removed from this node.
	 */
	public void childRemoved(ModelNode child) {
		// Do nothing. Child classes override this method if they want.
	}

	
	/**
	 * Registers a listener to receive ModelNodeEvents.
	 * @param l
	 */
	public void addModelNodeListener(ModelNodeListener l) {
		listenerList.add(ModelNodeListener.class, l);
	}
	
	
	/**
	 * Unregisters a listener from receiving ModelObjectEvents.
	 * @param l
	 */
	public void removeModelNodeListener(ModelNodeListener l) {
		listenerList.remove(ModelNodeListener.class, l);
	}
	
	
	/**
	 * Notifies listener about model node change.
	 */
	protected void fireModelNodeChanged(ModelNodeEvent e) {
		Object[] listeners = listenerList.getListenerList();
		for (int i = listeners.length-2; i>=0; i-=2) {
			if (listeners[i] == ModelNodeListener.class) {
				ModelNodeListener listener = (ModelNodeListener)listeners[i+1];
				listener.modelNodeChanged(e);
			}
		}
	}
}
