// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Stephane Guntz, PROSILOG, guntz@prosilog.com
//				  Yann Bajot, PROSILOG, bajot@prosilog.com                
//
//  Description : Transaction Level - Layer-1 to Layer-2 Slave Adapter (OCP 2.0 supported) 
//  $Id: ocp_tl1_tl2_slave_adapter.cpp,v 1.1.1.1 2004/09/24 09:26:27 sguntz Exp $
//
//  Modifications (09/01/2004): 
//		- correction of array indexing,
//		- correct bug for beginning index of TL1 requests, and computing of nb_responses (occurs when slave adapter receives more than adapter_depth TL2 chunks)
//		- correct bug when a request chunk is being processed, the buffer is full and a new one is received,
//		- add slave adapter name in debug printing messages,
//		- change size of m_request_buffer array
//		- change some debug messages
//	
//  Modifications (09/09/2004): 
//		- support for mixed RD/WR transfers with a read response blocked	
//  
//  Modifications (09/22/2004): new version which is compatible with both new TL2 performance channel and old layered TL2 channel
//	    - remove TL2 dataclass as a template (only one template now: the TL1 dataclass)
//	    - remove references to direct interfaces 
//  
//  
//  Parameters    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	                   
//	Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_chunk_length: maximum size of a response chunk
//		- max_burst_length: Maximum size for a burst
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response.
//						 It sends them when the current request/response is finished.
//	 
//	 
//  The TL1 slave adapter has a TL2 slave interface on one side, and a TL1 master interface on the other side.
//  It must be connected to:
//      - a TL2 master through a TL2 channel
//      - a TL1 slave through a TL1 channel
//
//	The adapter retrieves TL2 requests and stores them in its internal buffer. It sends corresponding TL1 requests to the TL1 slave.
//	  
//  On the response side, the TL1 slave adapter retrieves the TL1 responses from the slave, packs them in burst and sends corresponding
//  TL2 responses to the master. 
//
//	The following is unsupported:
//	- Single request/multiple data
//	- Data handshake
//	- Multiple threads
//	- SThreadBusy/MThreadBusy compliances (m_threadbusy_exact=0, s_threadbusy_exact=0)
//	- address sequence which is not STRM or INCR
// 
// 
// ============================================================================

#include "ocp_tl1_tl2_slave_adapter.h"


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter constructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::OCP_TL1_TL2_Slave_Adapter(sc_module_name name_, int max_chunk_length_, int max_burst_length_, int adapter_depth_): 
	sc_module(name_),
	MasterP("MasterP"),
	SlaveP("SlaveP"),
    max_chunk_length(max_chunk_length_),
	max_burst_length(max_burst_length_),
	adapter_depth(adapter_depth_)
{

	buffer_size=max_burst_length*adapter_depth;

	m_request_buffer= new OCPRequestGrp<Td, Ta>[adapter_depth];
	m_response_buffer=new OCPResponseGrp<Td>[buffer_size];
	m_data=new Td[buffer_size];
	m_response_data= new Td[buffer_size];
	
	//arrays for storing characteristics of TL2 responses
	burst_response_index=new int[adapter_depth];
	burst_response_length=new unsigned int[adapter_depth];
	last_of_a_burst=new bool[adapter_depth];
	nb_response=new int[adapter_depth];
    chunk_is_last_of_burst=new bool[adapter_depth];

	//arrays for storing characteristics of TL1 requests
	request_index=new int[adapter_depth];
	request_length=new unsigned int[adapter_depth];

	//initialize internal request, data and response buffer
	for(unsigned int i=0; i<buffer_size; i++)  {
		m_response_buffer[i].reset();
		m_data[i]=0;
		m_response_data[i]=0;
	}

	beginning_burst_index= current_index=0;

	for(int i=0; i<adapter_depth; i++)   {
		burst_response_index[i]=0;
		burst_response_length[i]=0;
		request_index[i]=0;
		request_length[i]=0;
		last_of_a_burst[i]=false;
		nb_response[i]=0;
        chunk_is_last_of_burst[i]=false;
	}
	new_chunk_transaction=true;
	release_request_thread=false;
	release_response_thread=false;
    read_response_not_finished=false;
	m_pending_request=0;
	m_request_number=0;
	m_pending_response=0;
	m_response_number=0;
	current_processed_request=current_processed_response=0;
	count_response=0;
	response_index=0;
	current_response_burst=0;


	m_NumBytesPerWord = sizeof(Td);

	//SystemC processes 
	SC_THREAD(SlaveResponse);
	sensitive<<MasterP.ResponseStartEvent();

	SC_THREAD(MasterResponse);
	sensitive<<send_TL2_response_event;

	SC_THREAD(SlaveRequest);
	sensitive<<SlaveP.RequestStartEvent();
    
	SC_THREAD(MasterRequest);
	sensitive<<send_TL1_request_event;

}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter destructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::~OCP_TL1_TL2_Slave_Adapter()    {

	if(m_response_data)   {
		delete [] m_response_data;
		m_response_data=0;
	}
	if(m_request_buffer)   {
		delete [] m_request_buffer;
		m_request_buffer=0;
	}
	if(m_response_buffer)   {
		delete [] m_response_buffer;
		m_response_buffer=0;
	}
	if(m_data)   {
		delete [] m_data;
		m_data=0;
	}
	if(burst_response_index)   {
		delete [] burst_response_index;
		burst_response_index=0;
	}
	if(burst_response_length)   {
		delete [] burst_response_length;
		burst_response_length=0;
	}
	if(request_index)   {
		delete [] request_index;
		request_index=0;
	}
	if(request_length)   {
		delete [] request_length;
		request_length=0;
	}
	if(last_of_a_burst)   {
		delete [] last_of_a_burst;
		last_of_a_burst=0;
	}
	if(nb_response)   {
		delete [] nb_response;
		nb_response=0;
	}
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::SlaveResponse 
//get the TL1 responses, store them in the internal buffer, prepare TL2 response to be sent to the TL2 master
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SlaveResponse()   {

	OCPResponseGrp<Td> resp;

	while(true)   {
		
		if(MasterP->getOCPResponseBlocking(resp))   {  //get a pending response

			//store the response fields in the internal buffer
			m_response_buffer[current_index].copy(resp);
			m_response_data[current_index]=m_response_buffer[current_index].SData;
			
			#ifdef DEBUG_SLAVE_ADAPTER_TL1   
				cout<<"---\nTL1-TL2 Slave Adapter \""<<name()<< "\" :";
				if(new_chunk_transaction)
					cout<<"\nSlaveResponse(): slave adapter gets the first TL1 response of a chunk at time "<<sc_time_stamp()<<endl;
				else cout<<"\nSlaveResponse(): slave adapter gets a TL1 response at time "<<sc_time_stamp()<<endl;
				cout<<"SThreadID: "<<m_response_buffer[current_index].SThreadID<<endl;
				cout<<"SResp: "<<m_response_buffer[current_index].SResp<<endl;
				cout<<"SRespInfo: "<<m_response_buffer[current_index].SRespInfo<<endl;
                cout<<"SData: "<<m_response_buffer[current_index].SData<<"\n---"<<endl;
			#endif

			switch(define_adapter_task())   {

			case SEND_CHUNK_WITH_DATA:
				//specify burst length and beginning
				burst_response_index[m_response_number]=beginning_burst_index;
				
				if(current_index<beginning_burst_index)   
					burst_response_length[m_response_number]=current_index+buffer_size-beginning_burst_index+1; 
				else burst_response_length[m_response_number]=current_index-beginning_burst_index+1; 
				

				//change the internal index
				if(current_index!=buffer_size-1)
					beginning_burst_index=current_index+1;
				else beginning_burst_index=0;

				//the following transaction is the beginning of a new chunk
				new_chunk_transaction=true;

				//increment number of pending responses that the adapter has to send
				m_pending_response++;
				//increment number of the next response that is going to be stored
				//check if the response buffer is full: next response is currently being processed
				if((m_response_number+1)%adapter_depth==current_processed_response)
					release_response_thread=true;
				else {
					release_response_thread=false;
					m_response_number=(m_response_number+1)%adapter_depth;
					MasterP->putMRespAccept();
				}
				
				//notify response event
				send_TL2_response_event.notify();	
				break;

			case SEND_CHUNK_WITHOUT_DATA:
				//specify burst length and beginning
				burst_response_index[m_response_number]=beginning_burst_index;
			
				if(current_index<beginning_burst_index)
					burst_response_length[m_response_number]=current_index+buffer_size-beginning_burst_index; 
				else burst_response_length[m_response_number]=current_index-beginning_burst_index; 
			
				//change the internal index
				beginning_burst_index=current_index;

				//the following transaction is the beginning of a new chunk
				new_chunk_transaction=true;

				//increment number of pending responses that the adapter has to send
				m_pending_response++;
				//increment number of the next response that is going to be stored
				//check if the response buffer is full: next response is not currently being processed
				if((m_response_number+1)%adapter_depth==current_processed_response)
					release_response_thread=true;
				else {
					release_response_thread=false;
					m_response_number=(m_response_number+1)%adapter_depth;
					MasterP->putMRespAccept();
				}
				
				//notify response event
				send_TL2_response_event.notify();
				break;

			case STORE_NEW_DATA:
				new_chunk_transaction=false;
				MasterP->putMRespAccept();
				release_response_thread=false;
				break;

			case SEND_CHUNK_AND_SINGLE_RESPONSE:
				//specify burst length and beginning
				burst_response_index[m_response_number]=beginning_burst_index;
				
				if(current_index<beginning_burst_index)
					burst_response_length[m_response_number]=current_index+buffer_size-beginning_burst_index;
				else burst_response_length[m_response_number]=current_index-beginning_burst_index; 
			
				//change the internal index
				beginning_burst_index=current_index;

				//change the internal index
				if(current_index!=buffer_size-1)
					current_index++;
				else current_index=0;

				//increment number of pending responses that the adapter has to send
				m_pending_response++;

				//increment number of the next response that is going to be stored
				//check if the response buffer is full: next response is not currently being processed
				if((m_response_number+1)%adapter_depth==current_processed_response)   {
					release_response_thread=true;
					send_TL2_response_event.notify();
					wait(thread_released_event);  //wait for the current request to be finished to store the characteristics of the next request
				}
				else {
					release_response_thread=false;
					m_response_number=(m_response_number+1)%adapter_depth;
				}

				//continue with single response
				burst_response_index[m_response_number]=beginning_burst_index;
				burst_response_length[m_response_number]=1; 
				

				//change the internal index
				if(current_index!=buffer_size-1)
					beginning_burst_index=current_index+1;
				else beginning_burst_index=0;

				//increment number of pending requests that the adapter has to send
				m_pending_response++;

				//increment number of the next response that is going to be stored
				//check if the response buffer is full: next request is not currently being processed
				if((m_response_number+1)%adapter_depth==current_processed_response)
					release_response_thread=true;
				else {
					release_response_thread=false;
					m_response_number=(m_response_number+1)%adapter_depth;
					MasterP->putMRespAccept();
                }

                //notify request event
                send_TL2_response_event.notify();
                break;

			default:
				break;
			}		

			//change the internal index
			if(current_index!=buffer_size-1)
				current_index++;
			else current_index=0;

		}  //get a pending response
	} //end of infinite loop
}  //end of SlaveResponse


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::MasterResponse
//send TL2 response made of atomic TL1 response
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MasterResponse()   {
	
	OCPResponseGrp<Td> resp;
	Td* tl2_response_data=new Td[max_chunk_length];
	for(int i=0; i<max_chunk_length; i++)
		tl2_response_data[i]=0;

	while(true)   {
		
		if(m_pending_response==0)
			wait();  //wait for notification of send_TL2_response_event

		//set the different fields of the response
		resp.copy(m_response_buffer[burst_response_index[current_processed_response]]);
		//set the response data
		for(int i=0; i<(int)burst_response_length[current_processed_response]; i++)
			tl2_response_data[i]=m_response_data[(i+burst_response_index[current_processed_response])%buffer_size];

		resp.SDataPtr=&tl2_response_data[0];

		
		#ifdef DEBUG_SLAVE_ADAPTER_TL1
			cout<<"\n**TL1-TL2 Slave Adapter \""<<name()<< "\" :";
			cout<<"\nMasterResponse: sends a TL2 response at time t="<<sc_time_stamp()<<endl;
			cout<<"SResp= "<<m_response_buffer[burst_response_index[current_processed_response]].SResp<<endl;
			cout<<"SThreadID= "<<m_response_buffer[burst_response_index[current_processed_response]].SThreadID<<endl;
			cout<<"SRespInfo= "<<m_response_buffer[burst_response_index[current_processed_response]].SRespInfo<<endl;
            cout<<"Length= "<<burst_response_length[current_processed_response]<<endl;
           if( chunk_is_last_of_burst[current_processed_response] )
                cout<<"Last chunk of the burst"<<endl;
            else cout<<"Not the last chunk of the burst"<<endl;
		#endif

		//send response
		SlaveP->sendOCPResponseBlocking(resp, burst_response_length[current_processed_response], chunk_is_last_of_burst[current_processed_response] );

        if((read_response_not_finished) && ( chunk_is_last_of_burst[current_processed_response] ))  {
            read_response_not_finished=false;
            response_index=(response_index+1)%adapter_depth;
            nb_response[response_index]=0;
            SlaveP->putSCmdAccept();
        }

                
		//processing next TL2 response
		current_processed_response=(current_processed_response+1)%adapter_depth;

		//release the response thread, if it was not released before
		if(release_response_thread)  {
			m_response_number=(m_response_number+1)%adapter_depth;
			MasterP->putMRespAccept();
			release_response_thread=false;
		}

        				
		if(m_pending_response!=0)
			m_pending_response--;

	} //end of infinite loop

} //end of MasterResponse method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::SlaveRequest
//get the TL2 request from the master
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SlaveRequest()   {

	OCPRequestGrp<Td,Ta> req;
	unsigned int chunk_length=0;
	bool last_chunk=false;
	unsigned int i=0;

	bool first_request=true;

	while(true)    {      
		
		//get TL2 request from the master 		
		if(SlaveP->getOCPRequestBlocking(req, false, chunk_length, last_chunk))   {
			//transfer request fields from TL2 request channel to the adapter
			m_request_buffer[m_request_number].copy(req);
			request_length[m_request_number]=chunk_length;
			last_of_a_burst[response_index]=last_chunk;

			#ifdef DEBUG_SLAVE_ADAPTER_TL1
				cout<<"\n***TL1-TL2 Slave Adapter \""<<name()<< "\" :";
				cout<<"\nMaster has sent a TL2 request at time t="<<sc_time_stamp()<<endl;
				cout<<"MAddr= "<<m_request_buffer[m_request_number].MAddr<<endl;
				cout<<"MCmd= "<<m_request_buffer[m_request_number].MCmd<<endl;
				cout<<"MThreadID= "<<m_request_buffer[m_request_number].MThreadID<<endl;
				cout<<"Chunk length= "<<request_length[m_request_number]<<endl;
				cout<<"Last chunk? ";
				if(last_of_a_burst[response_index])
					cout<<"Yes"<<"\n***"<<endl;
				else cout<<"No"<<"\n***"<<endl;
		   #endif

            //check if the address sequence is INCR or STRM or ends the simulation with a error message
            if(   (m_request_buffer[m_request_number].MBurstSeq!=OCP_MBURSTSEQ_INCR)
               && (m_request_buffer[m_request_number].MBurstSeq!=OCP_MBURSTSEQ_STRM) )   {
                   cerr<<"\n** TL1-TL2 Slave adapter \""<<name()<<"\" has received a request chunk with an address sequence which is not STRM or INCR: unsupported **\n"<<endl;
                   sc_stop();
            }
			
			//specify the beginning of the current chunk
            if(m_request_number==0)  {
				//if this is the first received TL2 request, set the beginning index to 0
				if(first_request )  {
					request_index[m_request_number]=0;
					first_request=false;
				}
				//or set the beginning index to the next index after the last request
				else request_index[0]=(request_index[adapter_depth-1]+request_length[adapter_depth-1])%buffer_size;		
			}
			else request_index[m_request_number]=(request_index[m_request_number-1]+request_length[m_request_number-1])%buffer_size;
			

			//check if the burst size does not exceed the maximum size, and copy the data
			if(m_request_buffer[m_request_number].MCmd==OCP_MCMD_WR)  {
                if(request_length[m_request_number]>max_burst_length)   {
				    cout<<"\n-------------\nTL1-TL2 Slave Adapter \""<<name()<< "\" :";
				    cout<<"\nError: size of one burst ("<<request_length[m_request_number]<<") exceeds the maximum burst length\n-------------"<<endl;
			    	for(i=0; i<buffer_size-request_index[m_request_number]; i++)   
			    		m_data[(i+request_index[m_request_number])%buffer_size] =req.MDataPtr[i];
		    	}
		    	else {
			    	for(i=0; i<request_length[m_request_number]; i++)   
			    		m_data[(i+request_index[m_request_number])%buffer_size] =req.MDataPtr[i];
		    	}
            }

			//computes the number of expected response:
			switch(m_request_buffer[m_request_number].MCmd)  {

				case OCP_MCMD_RD:		
                    //increment the number of expected responses
				    nb_response[response_index]+=request_length[m_request_number];
                    	
                    //if it is the last chunk of the burst, proceed with the following response chunk, if it is possible
                    if(last_chunk)  {
						if((response_index+1)%adapter_depth==current_response_burst)   
                            read_response_not_finished=true;
                        else {
                            response_index=(response_index+1)%adapter_depth;
                            //reset value nb_response, so that it can store a new response burst
							nb_response[response_index]=0;
                        }
                    }

					break;
				
				case OCP_MCMD_WR:
					//check if the slave sends responses to write requests
					if(write_response_enable)    {
                        //increment the number of expected responses
				        nb_response[response_index]+=request_length[m_request_number];
							 
                        //if it is the last chunk of the burst, proceed with the following response chunk, if it is possible
						if(last_chunk)  {
							if((response_index+1)%adapter_depth==current_response_burst)   
								read_response_not_finished=true;
							else {
								response_index=(response_index+1)%adapter_depth;
								//reset value nb_response, so that it can store a new response burst
								nb_response[response_index]=0;
							}
						}
					}
					else {
						//no response for a write request
						nb_response[response_index]=0;
                        
						//if it is the last chunk of the burst, proceed with the following response chunk, if it is possible
						if(last_chunk)  {
                            if((response_index+1)%adapter_depth==current_response_burst)   {
                                //next response burst is currently being processed and is a READ burst (or WR with response enable): must wait
								if(nb_response[current_response_burst]!=0)  
									read_response_not_finished=true;
                                else {
                                    response_index=(response_index+1)%adapter_depth;
									//reset value nb_response, so that it can store a new response burst
									nb_response[response_index]=0;
                                }
                            }
                            else {
                                response_index=(response_index+1)%adapter_depth;
                                //reset value nb_response, so that it can store a new response burst
								nb_response[response_index]=0;
                            }
                        }
					}
					break;

				default:
					break;
			}			

			send_TL1_request_event.notify();

			//increment number of pending requests that the adapter has to send
			m_pending_request++;

			//increment next request number
            if(read_response_not_finished )  
                m_request_number=(m_request_number+1)%adapter_depth;
		    else if((m_request_number+1)%adapter_depth==current_processed_request)  {
				m_request_number=(m_request_number+1)%adapter_depth;
				release_request_thread=true;
            }
			else {
				release_request_thread=false;
				m_request_number=(m_request_number+1)%adapter_depth;
				//release request channel
				SlaveP->putSCmdAccept();
			}

            wait();
			
		}  //end of get Request
	} //end of infinite loop
} //end of SlaveRequest method


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::MasterRequest
//send the group of TL1 requests to the slave, made from the TL2 request from the master 
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MasterRequest()   {
	OCPRequestGrp<Td,Ta> req;
	int i=0;
	
	while(true)    {
		if(m_pending_request==0)
			wait();  //wait for notification of send_TL1_request_event

		wait(clk->posedge_event());

        while(MasterP->MgetSbusy())  
			wait(clk->posedge_event());

		//send atomic TL1 requests
		for(i=request_index[current_processed_request]; i<request_index[current_processed_request]+(int)request_length[current_processed_request]; i++)  {

			//specify TL1 request fields
			req.copy(m_request_buffer[current_processed_request]);
            //specify the address
            if(req.MBurstSeq==OCP_MBURSTSEQ_INCR)
			    req.MAddr=m_request_buffer[current_processed_request].MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord;
            else if(req.MBurstSeq==OCP_MBURSTSEQ_STRM)
                req.MAddr=m_request_buffer[current_processed_request].MAddr;

			req.MData=m_data[i%buffer_size];
	
			//specify a precise burst
            req.MBurstPrecise=1;
            req.MBurstLength=request_length[current_processed_request];
		                    
			//send the request
			MasterP->startOCPRequest(req);

			#ifdef DEBUG_SLAVE_ADAPTER_TL1
			switch(req.MCmd)   {
				case OCP_MCMD_WR:
					cout<<"\n**TL1-TL2 Slave Adapter \""<<name()<< "\" :";
					if(i==request_index[current_processed_request])
						cout<<"\nMasterRequest: sends the first TL1 write request of a chunk at time t="<<sc_time_stamp()<<endl;
					else if(i==request_index[current_processed_request]+(int)request_length[current_processed_request]-1)
						cout<<"\nMasterRequest: sends the last TL1 write request of a chunk at time t="<<sc_time_stamp()<<endl;
					else cout<<"\nMasterRequest: sends a TL1 write request at time t="<<sc_time_stamp()<<endl;
					cout<<"Data= "<<m_data[i%buffer_size]<<endl;
					cout<<"Addr= "<<m_request_buffer[current_processed_request].MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord<<endl;
					break;

				case OCP_MCMD_RD:
					cout<<"\n**TL1-TL2 Slave Adapter \""<<name()<< "\" :";
					if(i==request_index[current_processed_request])
						cout<<"\nMasterRequest: sends the first TL1 read request of a chunk at time t="<<sc_time_stamp()<<endl;
					else if(i==request_index[current_processed_request]+(int)request_length[current_processed_request]-1)
						cout<<"\nMasterRequest: sends the last TL1 read request of a chunk at time t="<<sc_time_stamp()<<endl;
					else cout<<"\nMasterRequest: sends a TL1 read request at time t="<<sc_time_stamp()<<endl;
					cout<<"Addr= "<<m_request_buffer[current_processed_request].MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord<<endl;
					break;

				case OCP_MCMD_RDEX:
					cout<<"\n**TL1-TL2 Slave Adapter \""<<name()<< "\" :";
					if(i==request_index[current_processed_request])
						cout<<"\nMasterRequest: sends the first TL1 exclusive read request of a chunk at time t="<<sc_time_stamp()<<endl;
					else if(i==(int)(request_index[current_processed_request]+(int)request_length[current_processed_request]-1))
						cout<<"\nMasterRequest: sends the last TL1 exclusive read request of a chunk at time t="<<sc_time_stamp()<<endl;
					else cout<<"\nMasterRequest: sends a TL1 exclusive read request at time t="<<sc_time_stamp()<<endl;
					cout<<"Addr= "<<m_request_buffer[current_processed_request].MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord<<endl;
					break;

				default:
					cout<<"++++ TL1-TL2 Slave Adapter \""<<name()<< "\" :default case in MasterRequest() ++++"<<endl;
					break;
			} //end of switch
			#endif

			while(MasterP->MgetSbusy())  
				wait(clk->posedge_event());  //wait for next clock edge to send next TL1 request;

        }

        current_processed_request=(current_processed_request+1)%adapter_depth;

		//release the request thread, if it was not released before
		if(release_request_thread)  {
		    SlaveP->putSCmdAccept();
			release_request_thread=false;
            thread_released_event.notify(); 
		}

		if(m_pending_request!=0)
			m_pending_request--;

	} //end of infinite loop
} //end of MasterRequest method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::define_adapter_task
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1 >
SlaveAdapterTask OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::define_adapter_task()   {
	
	count_response++;  //increments number of received TL1 responses
    
	//checks for the next burst of response
	while(nb_response[current_response_burst]==0)        
       		current_response_burst=(current_response_burst+1)%adapter_depth;

	//compare with response parameters in the internal buffer
	if( !new_chunk_transaction)   {
		bool same_fields=same_response_fields(m_response_buffer[current_index], m_response_buffer[beginning_burst_index]);

		//the response fields are the same		
		if( same_fields)  {
			if(count_response==nb_response[current_response_burst])   {
				//chunk is the last of the burst: send the responses 
				if(last_of_a_burst[current_response_burst])  {
					count_response=0;
					chunk_is_last_of_burst[m_response_number]=true;
					current_response_burst=(current_response_burst+1)%adapter_depth;

					return SEND_CHUNK_WITH_DATA;
				}
				//or wait for the following chunk
				else return STORE_NEW_DATA;
			}
			else {
				if(current_index<beginning_burst_index)   {
					if((int)(current_index+buffer_size-beginning_burst_index+1)==max_chunk_length)  {
						chunk_is_last_of_burst[m_response_number]=false;
						return SEND_CHUNK_WITH_DATA;
					}
					else return STORE_NEW_DATA;
				}
				else if((int)(current_index-beginning_burst_index+1)==max_chunk_length)  {
						chunk_is_last_of_burst[m_response_number]=false;
						return SEND_CHUNK_WITH_DATA;
				}
				else return STORE_NEW_DATA;
			}
		}
		//the fields have changed, check if the last response is the last of the burst
		else {
            if(count_response==nb_response[current_response_burst])  {
                //chunk is the last of the burst: send the responses
                if(last_of_a_burst[current_response_burst])  {
                    count_response=0;
                    chunk_is_last_of_burst[m_response_number]=false;
                    chunk_is_last_of_burst[(m_response_number+1)%adapter_depth]=true;
					current_response_burst=(current_response_burst+1)%adapter_depth;
                    
					return SEND_CHUNK_AND_SINGLE_RESPONSE;
                }
                //or wait for the following chunk
                else return STORE_NEW_DATA;
            } 
            else return STORE_NEW_DATA;
        }
	}
	//new chunk transaction
	else  {
        //is it a single response: send it
        if(count_response==nb_response[current_response_burst])  {
            count_response=0;
            chunk_is_last_of_burst[m_response_number]=true;
			current_response_burst=(current_response_burst+1)%adapter_depth;
            
			return SEND_CHUNK_WITH_DATA;
        }
        else return STORE_NEW_DATA;
    }	
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::same_response_fields
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
bool OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::same_response_fields(OCPResponseGrp<Td>& new_res_param, OCPResponseGrp<Td>& current_res_param )  {

	//multiple threads are not supported
	if(new_res_param.SThreadID != current_res_param.SThreadID) {
		cerr<<"\n****************************\nTL1-TL2 Slave Adapter \""<<name()<< "\"Error:\n";
		cerr<<"Multiple threads are not supported\n****************************"<<endl;
		sc_stop();
	}
	//check if the SRespInfo field has changed
	else if(new_res_param.SRespInfo != current_res_param.SRespInfo)
		return false;

	//SRespInfo field has not changed:
	return true;

}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Slave_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::end_of_elaboration()
{

  sc_module::end_of_elaboration();

  // Get data structure
  m_DataCl_tl1 = MasterP->GetDataCl();


  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();
 
  //multiple threads are not supported
  if(m_ParamCl->threads>1)  {
	cerr<<"\n****************************\nTL1-TL2 Slave Adapter \""<<name()<< "\", Error:\n";
	cerr<<"Multiple threads are not supported\n****************************"<<endl;
	sc_stop();
  }

  //response to write requests?
  if(m_ParamCl->writeresp_enable)  
	  write_response_enable=true;
  else write_response_enable=false;

  //check for threadbusy compliances
  if((m_ParamCl->mthreadbusy_exact) || (m_ParamCl->sthreadbusy_exact))  {
	  cerr<<"\n****************************\nTL1-TL2 Slave Adapter \""<<name()<< "\", Error:\n";
	  cerr<<"Threadbusy compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for single request/multiple data
  if((m_ParamCl->burstsinglereq))  {
	  cerr<<"\n****************************\nTL1-TL2 Slave Adapter \""<<name()<< "\", Error:\n";
	  cerr<<"Single Request/Multiple Data compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for datahandshake 
  if((m_ParamCl->datahandshake))  {
	  cerr<<"\n****************************\nTL1-TL2 Slave Adapter \""<<name()<< "\", Error:\n";
	  cerr<<"Single Request/Multiple Data compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

 //check for cmdaccept 
  if(!m_ParamCl->cmdaccept)  {
	  cerr<<"\n****************************\nTL1-TL2 Slave Adapter \""<<name()<< "\", Error:\n";
	  cerr<<"cmdaccept parameters must be set\n****************************"<<endl;
	  sc_stop();
  }

  //check for srespaccept 
  if(!m_ParamCl->respaccept)  {
	  cerr<<"\n****************************\nTL1-TL2 Slave Adapter \""<<name()<< "\", Error:\n";
	  cerr<<"respaccept parameters must be set\n****************************"<<endl;
	  sc_stop();
  }

  // Put parameter into Channel
  // No parameters needed

}


// ----------------------------------------------------------------------------
//
//  Instantiation of the adapter
//
// ----------------------------------------------------------------------------

template class OCP_TL1_TL2_Slave_Adapter<OCP_TL1_DataCl<int , int> >; 

