// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//		 modified by: Stephane Guntz, Prosilog, guntz@prosilog.com
//					  Yann Bajot, Prosilog, bajot@prosilog.com
//					  
//
//  Description : Transaction Level - Layer-1 example Slave
//  $Id: ocp_tl1_slave_sync.cpp,v 1.1.1.1 2004/09/24 09:26:19 sguntz Exp $
//  
//    Features:
//    - Synchronous slave using OCP API
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//                
// Parameter    :
//   Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the SLave.
//           ID must be unique among all Slaves attached to the same Bus.
//     - StartAddress: First address of the Slaves memory.
//     - EndAddress: Last address of the Slaves memory.
//                   The address range of all Slaves in a system
//                   must be mutually distinct.
//
//
// ============================================================================

#include "ocp_tl1_slave_sync.h"

//#define DEBUG_G1

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Slave_Sync::OCP_TL1_Slave_Sync
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Slave_Sync<TdataCl>::OCP_TL1_Slave_Sync
       (
          sc_module_name name_
        , int ID
        , Ta StartAddress
        , Ta EndAddress
        )
          : sc_module          (name_)
          , SlaveP             ("SlaveP")
          , m_ID               (ID)
          , m_StartAddress     (StartAddress)
          , m_EndAddress       (EndAddress)
{
  int i;
  m_NumWordsPerBeat = sizeof(Td);
  m_SmemoryLength = (m_EndAddress - m_StartAddress + 1) / m_NumWordsPerBeat;

  // allocate memory
  if (m_SmemoryLength > 0)
  {
    try
    {
      m_SlaveMemory = new Td[m_SmemoryLength];
    }
    catch(std::bad_alloc ex)
    {
      cerr << "ERROR: Memory allocation for Slave memory failed "
            << name() << ">" << endl;
      cerr << "       Size  is: " << m_SmemoryLength << endl;
      cerr << "       Error is: " << ex.what() << endl;
      sc_stop();
      return;
    }
  }

  for (i = 0; i < m_SmemoryLength; i++)
    m_SlaveMemory[i] = i;

  for (i = 0; i < TL_SLAVE_FIFO_DEPTH; i++)
    m_RespEventS[i] = false;

  SC_THREAD(Request);
  sensitive_pos(clk);
  SC_THREAD(Response);
  sensitive << m_RespEvent;
}


// ----------------------------------------------------------------------------
// Process : OCP_TL1_Slave_Sync::~OCP_TL1_Slave_Sync
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Slave_Sync<TdataCl>::~OCP_TL1_Slave_Sync()
{
  delete [] m_SlaveMemory;
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::Request()
//
//  Top level synchronous data sampling method
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_Sync<TdataCl>::Request()
{
  Td Data;
  Ta Addr;
  OCPRequestGrp<Td,Ta> req;

  int i = 0;
  int ReqIdx = 0;
  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  int cnt = 0;

  while(true)
  {
    wait(); // for rising clock edge

    if (cnt == 0)
    {
		if (SlaveP->getOCPRequest(req, false))		{
			// check if a request was posted
			if (req.MCmd != OCP_MCMD_IDLE)
			{
				Data=req.MData;
				Addr=req.MAddr;
				m_MThreadID=req.MThreadID;
				
				if (req.MCmd == OCP_MCMD_RD)
				{
				// send response in same delta cycle
					m_RespEventS[ReqIdx] = true;
					m_RespEvent.notify();
					response_data=m_SlaveMemory[(Addr - m_StartAddress + 1) / m_NumWordsPerBeat];
					
					// get new index (only for those requests, where the
					// response thread is called)
					ReqIdx++;
					if (ReqIdx >= TL_SLAVE_FIFO_DEPTH)
					  ReqIdx = 0;
					}
				else {
					m_SlaveMemory[(Addr - m_StartAddress + 1) / m_NumWordsPerBeat]=Data;
				}
			
				#ifdef DEBUG_G1
						cout << "Slave got request "
						 << "  delta = " << simcontext()->delta_count()
						 << "  time  = " << sc_time_stamp().to_seconds()
						 << "  data  = " << Data 
						<< " address = "<< Addr<<endl;
				#endif
						i++; cnt = Pause[i%10];
				} // end of get request branch
			
			//release channel
			SlaveP->putSCmdAccept();
        }
    }
    else cnt--;

  } // end of while(true) loop
} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::Response()
//
//  Top level response method (synchronious)
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_Sync<TdataCl>::Response()
{
	
  OCPResponseGrp<Td> resp;
  int ResIdx = 0;

  while(true)
  {
    if (!m_RespEventS[ResIdx])
      wait(); // for m_RespEvent event

	resp.SResp=OCP_SRESP_DVA;
	resp.SThreadID=m_MThreadID;
	resp.SData=response_data;

    // reset + get new index
    m_RespEventS[ResIdx] = false;
    ResIdx++;
    if (ResIdx >= TL_SLAVE_FIFO_DEPTH)
      ResIdx = 0;

    // send response to Master
	while (!SlaveP->startOCPResponse(resp))
      wait(clk->posedge_event());
  }
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_Sync<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  SlaveP->SregisterDirectIF(this);

  // Get data structure
  m_DataCl = SlaveP->GetDataCl();


  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Sync::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Slave_Sync<TdataCl>::MputDirect(
         int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{
  int i;
  int NumBeats = NumWords / m_NumWordsPerBeat;
  int Offset = (Address - m_StartAddress) / m_NumWordsPerBeat;

  if ((Offset >= 0) && ((Offset + NumBeats) <= m_SmemoryLength))
  {
    if (IsWrite)
    {
      // write request
      for (i = 0; i < NumBeats; i++)
        m_SlaveMemory[Offset + i] = Data[i];
    }
    else
    {
      // read request
      for (i = 0; i < NumBeats; i++)
        Data[i] = m_SlaveMemory[Offset + i];
    }
    return(true);
  }
  else
  {
    cout << "Error: Address out of range at Slave " << m_ID << endl
         << "       Address=" << Address << "  StartAddr=" << m_StartAddress
         << "  _EndAddr=" << m_EndAddress
         << "  NumWords=" << NumWords
         << endl;
    return(false);
  }
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Slave_Sync<OCP_TL1_DataCl<int , int> >; 
