// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//			      modified by Yann Bajot, Prosilog, bajot@prosilog.com,
//			      Stephane Guntz, Prosilog, guntz@prosilog.com
//
//  Description : Transaction Level - Layer-1 example Master
//  $Id: ocp_tl1_master.cpp,v 1.1.1.1 2004/09/24 09:26:19 sguntz Exp $
//
// The master sends a READ precise burst, with a total length of 15 elements, at each rising edge of the clock
// The start address is 0, and the address increment is 4.
// The sequence is decomposed as follows:
// At the 4th element (address=12): the MReqInfo field changes and is 1 now..
// 
// For the remaining READ requests: the address increment is still 4, MBurstLength=15, MReqInfo=1
// 
// The adapter separates that 15 TL1-element burst into 4 TL2 chunks:
// The first chunk has a length of 3 (start address=0, end address=8, read data are 0,1,2)
// The second chunk has a length of 5, which is the maximum chunk size: it begins when the MReqInfo has changed (start address=12, end address= 28, read data=3,4,5,6,7)
// The third chunk has a length of 5 (start address=32, end address= 48, read data=8,9,10,11,12)
// The fourth and last chunk has a length of 2 (start address=52, end address= 56, read data=13,14).
//
// On the response side, for each response, the master releases the response channel after a delay of 2 clock cycles
//
// ============================================================================


#include "ocp_tl1_master.h"

#define DEBUG_G1

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master::OCP_TL1_Master
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl_tl1> OCP_TL1_Master<TdataCl_tl1>::OCP_TL1_Master
	  (
        sc_module_name name_
      , int ID
      , int Priority
      , double DelayPct
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_DelayPct        (DelayPct)
{
  // InitParameters();

  SC_THREAD(MasterRequest);
  sensitive_pos(clk); 
 
  SC_THREAD(MasterResponse);
  sensitive<<MasterP.ResponseStartEvent(); 
 }

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master::~OCP_TL1_Master
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl_tl1> OCP_TL1_Master<TdataCl_tl1>::~OCP_TL1_Master()
{
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master::MasterRequest()
//
//  Top level request method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl_tl1> void OCP_TL1_Master<TdataCl_tl1>::MasterRequest()
{
  int i = 0;
  int Pause[] = {0, 0, 0, 0}; // # of cycles between requests;
 // int Pause[] = {0, 1, 2, 3}; // # of cycles between requests;
  bool tmp = 0;
  Td Data = 0;
  Ta Addr = 0;
  int cnt = 0;
  
  OCPRequestGrp<Td, Ta> req;

  while(true)
  {
    wait(); // wait for rising edge of clock

    if (cnt == 0)
    {
      // check if channel is free
      tmp = !MasterP->MgetSbusy();
      if (tmp)
      {
        // get new Pause value
        cnt = Pause[i%4]; i++;

        // Put data into the channel
		req.MAddr=Addr;
		req.MData=Data;
		req.MBurstPrecise=true;
		req.MThreadID=0;
		req.MBurstSeq=OCP_MBURSTSEQ_INCR;
		req.MCmd=OCP_MCMD_RD;
		req.MBurstLength=15;
		req.MReqInfo=0;
		
		if(Addr>=12)  
			req.MReqInfo=1;
		else if(Addr==40)
			req.MReqInfo=0;

		//send request
		MasterP->startOCPRequest(req); 	

		#ifdef DEBUG_G1
		  cout << "Master sent TL1 request "
			   << " delta " << simcontext()->delta_count()
			   << " time " << sc_time_stamp().to_seconds()
			   << " data " << Data << endl;
		#endif

		if(Data==14)
			break;

		Addr += 4;
		Data++;
      }  //end of test if channel is free
    } //enf of if cnt==0
    else
    {
      cnt--;
    }
  } // end of while loop        
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master::MasterResponse()
//
//  Top level response method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl_tl1> void OCP_TL1_Master<TdataCl_tl1>::MasterResponse()
{
 Td Rdata;
 int SThreadID;
 OCPResponseGrp<Td> resp;

  while(true)
  {
    wait(); 
    if(MasterP->getOCPResponseBlocking(resp))    //get a pending response
    {
      Rdata=resp.SData;
	  SThreadID=resp.SThreadID;
	  
#ifdef DEBUG_G1
      cout << "Master got response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " data " << Rdata 
		   << " SThreadID "<< SThreadID<<endl;
#endif
	  //release response channel after a delay of two clock cycles
	  wait(clk->posedge_event());
      wait(clk->posedge_event());

      //Accept the response
      MasterP->putMRespAccept();
    }

  } // end of while(true) loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl_tl1> bool OCP_TL1_Master<TdataCl_tl1>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl_tl1> void OCP_TL1_Master<TdataCl_tl1>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed

}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Master<OCP_TL1_DataCl<int, int> >; 
