///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc.
//          Date: 12/23/2005
//
//  Description :  OCP Channel Transaction Recording Monitor
//	  This channel monitor is based on the transaction recording 
//	  API in the SystemC Verification (SCV) Library. It targets 
//	  performance analysis for architectural modeling.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL1_PERF_MONITOR_H
#define _OCP_TL1_PERF_MONITOR_H

#include <systemc.h>

#include "ocp_perf_monitor_channel_scv.h"
#include "ocp_perf_monitor_system_registry.h"
#include "ocp_tl1_monitor_port.h"
#include "ocp_tl1_perf_monitor_wrapper.h"


template <typename TDataCl>
class OCP_Tl1_Perf_Monitor :
  public sc_module
{
public:
  typedef typename TDataCl::DataType	Td;
  typedef typename TDataCl::AddrType	Ta;
  typedef OCP_TL1_MonitorPort<TDataCl> tl1_port_type;
  typedef typename tl1_port_type::request_type tl1_request_type;
  typedef typename tl1_port_type::response_type tl1_response_type;

  SC_HAS_PROCESS(OCP_Tl1_Perf_Monitor);

  /** this port shall be bound to the monitor interface of the OCP channel */
  tl1_port_type p_mon;
  sc_in<bool> p_clk;
  
  OCP_Tl1_Perf_Monitor(sc_module_name name,
		       bool channel_recording  = true,
		       bool system_recording   = true,
		       bool master_is_node     = false,
		       bool slave_is_node      = false,
		       unsigned int  max_nbr_of_threads = 0,
		       bool burst_recording    = false,
		       bool attribute_recording= false) :
    sc_module(name),
    p_mon("p_mon"),
    p_clk("p_clk"),
    m_channel_monitor(NULL),
    m_system_monitor(NULL),
    m_channel_recording(channel_recording),
    m_system_recording(system_recording),
    m_master_is_node(master_is_node),
    m_slave_is_node(slave_is_node),
    m_req_start(false),
    m_req_end(false),
    m_resp_start(false),
    m_resp_end(false)
  {
     if (m_channel_recording) {
      m_channel_monitor = new OCP_Perf_Monitor_Channel_SCV<Td,Ta>(name,max_nbr_of_threads,burst_recording,attribute_recording);
    }
    
    if (m_system_recording) {
      m_system_monitor = OCP_Perf_Monitor_System_Registry<Td,Ta>::instance();
      assert(m_system_monitor);
    }

    SC_METHOD(OCPClockTick);
    sensitive << p_clk.pos();
  }
  
  virtual ~OCP_Tl1_Perf_Monitor()
  {
    delete m_channel_monitor;
    OCP_Perf_Monitor_System_Registry<Td,Ta>::delete_system_monitor();
  }

  void end_of_elaboration()
  {
    m_tl1_perf_monitor_wrapper = new OCP_Tl1_Perf_Monitor_Wrapper<TDataCl>(p_mon[0]);
     
    if (m_channel_recording) {
      m_channel_monitor->registerChannel(m_tl1_perf_monitor_wrapper);
    }
    if (m_system_recording) {
      m_system_monitor->registerChannel(m_tl1_perf_monitor_wrapper,
					m_master_is_node, 
					m_slave_is_node);
    }
  }

  void start_of_simulation()
  {
    if (m_system_recording) {
      m_system_monitor->start_of_simulation();
    }
  }

  void
  OCPClockTick(void)
  {
    bool new_req_start  = p_mon->peekRequestStart();
    bool new_req_end    = p_mon->peekRequestEnd();
    bool new_resp_start = p_mon->peekResponseStart();
    bool new_resp_end   = p_mon->peekResponseEnd() ;

    if ( new_req_end || (new_req_start && m_req_start)) {
      if (m_channel_recording) {
	m_channel_monitor->NotifyRequestEnd(m_tl1_perf_monitor_wrapper);
      }
      if (m_system_recording) {
	m_system_monitor->NotifyRequestEnd(m_tl1_perf_monitor_wrapper);
      }
    }

    if ( new_req_start ) {
      m_tl1_perf_monitor_wrapper->setRequest(p_mon->peekOCPRequest());
      if (m_channel_recording) {
	m_channel_monitor->NotifyRequestStart(m_tl1_perf_monitor_wrapper);
      }
      if (m_system_recording) {
	m_system_monitor->NotifyRequestStart(m_tl1_perf_monitor_wrapper);
      }
    }

    if ( new_resp_end  || (new_resp_start && m_resp_start)) {
      if (m_channel_recording) {
	m_channel_monitor->NotifyResponseEnd(m_tl1_perf_monitor_wrapper);
      }
      if (m_system_recording) {
	m_system_monitor->NotifyResponseEnd(m_tl1_perf_monitor_wrapper);
      }
    }

    if ( new_resp_start ) {
       m_tl1_perf_monitor_wrapper->setResponse(p_mon->peekOCPResponse());
      if (m_channel_recording) {
	m_channel_monitor->NotifyResponseStart(m_tl1_perf_monitor_wrapper);
      }
      if (m_system_recording) {
	m_system_monitor->NotifyResponseStart(m_tl1_perf_monitor_wrapper);
      }
    }

    m_req_start  = new_req_start; 
    m_req_end    = new_req_end;   
    m_resp_start = new_resp_start;
    m_resp_end   = new_resp_end;  
  }

protected:
  OCP_TL2_Monitor_ObserverIF<Td,Ta>* m_channel_monitor;
  OCP_TL2_Monitor_ObserverIF<Td,Ta>* m_system_monitor;
  OCP_Tl1_Perf_Monitor_Wrapper<TDataCl>*   m_tl1_perf_monitor_wrapper;

  const bool m_channel_recording;
  const bool m_system_recording;
  const bool m_master_is_node;
  const bool m_slave_is_node;

  bool m_req_start;
  bool m_req_end;
  bool m_resp_start;
  bool m_resp_end;  
};

#endif
