// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: StlParser.cc,v 1.2 2007/10/08 23:46:44 halexan Exp $

#include "StlParser.h"
#include "ocp_globals.h"
#include "ocp_utils.h"
#include "ocp_param.h"
#include <iostream>
#include <sstream>
#include <string>

////////////////////////////////////////////////////////////////////////////
using namespace std;
using namespace boost::spirit;
using namespace OcpIp;

static StlParser::MyTransferCommand::DataType genFullOnes() {
    StlParser::MyTransferCommand::DataType ret;
    ret.set();
    return ret;
}

boost::rand48 StlParser::s_randGenerator;
const OcpIp::StlParser::MyTransferCommand::DataType
StlParser::s_fullOneData( genFullOnes () );
//////////////////////////////////////////////////////////////////
// FUNCTION: StlParser::StlParser
// DESCRIPTION: constructor
//////////////////////////////////////////////////////////////////
StlParser::StlParser( const OcpParams& params, bool coreSystem,
                      ostream& err ) :
    m_ocpParams( params ),
    m_pErr( &err ),
    m_commandType( STL_INVALID ),
    m_transferCmd     ( m_ocpParams ),
    m_waitCmd         ( m_ocpParams ),
    m_signalCmd       ( m_ocpParams ),
    m_resetCmd        ( m_ocpParams ),
    m_controlStatusCmd( m_ocpParams, coreSystem ),
    m_versionChecked  ( false ),
    m_versionWarned   ( false ),
    m_pCommandConverter( CommandConverter::create( m_ocpParams,
                                                   m_transferCmd ) )
{
}

//////////////////////////////////////////////////////////////////
// FUNCTION: promoteWarning
// DESCRIPTION: raise warnings to errors on request
//////////////////////////////////////////////////////////////////
static bool promoteWarning()
{
    static int w_as_e = 0;

    if (0 == w_as_e) {
        w_as_e = getenv("STL_WARNING_IS_ERROR") ?  1 
                                                : -1;
    }
    return w_as_e > 0;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserTransferCommand::reportError
// DESCRIPTION: checks the command for OCP validity
// ARGUMENTS: string reference to receive error messages
// RETURNS: true if errors were reported, false if clean
// NOTE: this function belongs in StlTransfer.cc but since
// StlParserTransferCommand is a templated class, and this is fully
// specialized, we would get duplicate definitions when making
// explicit instantiations
//////////////////////////////////////////////////////////////////
namespace {
// Utility function to make a lsb-mask of a given width 
template<typename Td>
Td makeDataMask( uint32_t width )
{
    if ( width == 8*sizeof(Td) )
        return numeric_limits<Td>::max();
    else
        return Td( ( 1ULL << width ) - 1 );
}
}

template<>
int
StlParserTransferCommand<Sonics::BigInt<256>,snx_uint64_t>::reportError(
    string& errors ) const
{
    uint32_t dataWdth = ocpParam( P_data_wdth );

#define MAXVAL( param ) \
( ocpParam( param ) ? ( makeDataMask<snx_uint64_t>( ocpParam( param##_wdth ) ) ) : 0 )
    uint32_t numErrors( 0 );
    uint32_t numWarnings( 0 );
    ostringstream errorStr;
#define CHECKRANGE( signal, min, max )                              \
    if ( request. signal > static_cast<snx_uint64_t> ( max ) ) {        \
        ++numErrors;                                                \
        errorStr << #signal << "(" << request. signal << ") out of" \
        << " range (" << min << "," << max << ")" << "\n";          \
    }
    CHECKRANGE( MThreadID,   0,  ocpParam( P_threads ) - 1 );
    CHECKRANGE( MConnID,     0,  MAXVAL( P_connid ) );
    CHECKRANGE( MReqInfo,    0,  MAXVAL( P_reqinfo ) );
    CHECKRANGE( MAddr,       0,  MAXVAL( P_addr ) );
    CHECKRANGE( MAddrSpace,  0,  MAXVAL( P_addrspace ) );

    // check range for expanded width commands for ByteEn and DataInfo
    if ( request.MByteEn != static_cast<ByteEnType>( -1 ) ) {
        ByteEnType maxByteEn = ( 1ULL <<
                                 (int)( ceil( (double) m_transferWidth ) * 1.0 / 8 ) );
        if ( ocpParam( P_byteen ) > 0 ) {
           CHECKRANGE( MByteEn,  0,  maxByteEn );
        } else if ( ocpParam( P_mdatabyteen ) > 0 ) {
	   // need to check elsewhere ???
        } else {
	   maxByteEn--; // 1000... -> fff...
	   ByteEnType be = request.MByteEn & maxByteEn;
           if (be != maxByteEn) { // OK if all enabled
             ++numErrors;
             errorStr << "ByteEn/MDataByteEn not implemented on this interface" << "\n"; 
	   }
	}
    }


    if ( ocpParam( P_mdatainfo ) ) {
        DataInfoType maxDataInfo =
            ( 1ULL << ( ( ocpParam( P_mdatainfo_wdth ) *
                          (int)( m_transferWidth / dataWdth ) ) ) );
        if ( m_currentDataInfo != static_cast<DataInfoType>( -1 ) &&
             m_currentDataInfo > maxDataInfo ) {
            ++numErrors;
            errorStr << "MDataInfo is out of range (" << maxDataInfo
                     << ")" << "\n";
        }            
    } else if (m_currentDataInfo != static_cast<DataInfoType>( -1 ) && 0 == numErrors) {
        ++numWarnings;
        errorStr << "MDataInfo not supported \n";
    }

    // check data range on the BigInt<256>
    for ( uint32_t bit = m_transferWidth; bit<256; ++bit ) {
        if ( m_currentData.test( bit ) ) {
            ++numErrors;
            errorStr << "MData is out of range (" << m_transferWidth
                     << ")" << "\n";
            break;
        }
    }
    if ( m_currentDataMask != StlParser::s_fullOneData ) {
        for ( uint32_t bit = m_transferWidth; bit<256; ++bit ) {
            if ( m_currentDataMask.test( bit ) ) {
                ++numErrors;
                ostringstream mask;
	        mask << m_currentDataMask;
                const char *mp = mask.str().c_str();
                while ('0' == *mp) mp++;
                errorStr << "Data mask ("         << mp
                         << ") is out of range (" << m_transferWidth
                         << ")" << "\n";
                break;
            }
        }
    }

    if ( m_transferWidth != dataWdth && (
             ocpParam( P_endian ) == Definitions::ENDIAN_NEUTRAL ||
             ocpParam( P_endian ) == Definitions::ENDIAN_BOTH ) ) {
        ++numErrors;
        errorStr << "Explicit width macros are not allowed "
            "when endianness is neutral or both" << "\n";
    }
        
    // check for explicit width narrower than dataWdth
    if ( m_transferWidth < dataWdth && !ocpParam( P_byteen ) ) {
        // used to be an ocpasm warning. 2.0 stl spec says "must"
        ++numErrors;
        errorStr << "Byte-enabled transfer when byteen=0" << "\n";
    }

    // check alignment
    int dataWidth = ocpParam( P_data_wdth );
    int byteenWidth = ( dataWidth <= 8 ) ? 1 :
        ( 1 << OcpIp::ceilLog2( dataWidth ) - 3 );
    int numAddrBits0 = OcpIp::ceilLog2( byteenWidth ) -1;

    if ( request.MAddr & ( 1 << numAddrBits0 ) ) {
        ++numErrors;
        errorStr << "MAddr must be aligned to a " << byteenWidth
                 << " byte boundary" << "\n";
    }

    // check if burst macro is used with a DFLT1, DFLT2 or UNKN
    //   sequence.
    if (ocpParam( P_burstseq ) && request.DataLength > 1) {
        if ((request.MBurstSeq == OCP_MBURSTSEQ_DFLT1) ||
            (request.MBurstSeq == OCP_MBURSTSEQ_DFLT2) ||
            (request.MBurstSeq == OCP_MBURSTSEQ_UNKN)) {
            ++numErrors;
            errorStr << "Burst macro must not be used with "
                     << "DFLT1, DFLT2 and UNKN burst sequences." 
                     << "\n";
        }
    }


#undef CHECKRANGE
#undef MAXVAL

    if ( numErrors > 0 || numWarnings > 0) {
	if (numWarnings > 0 && promoteWarning()) {
            numErrors  += numWarnings;
	    numWarnings = 0;
	}
        errors = errorStr.str();
        return ( numErrors > 0 ) ?  1
	                         : -1;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParserTransferCommand::wrapup()
// DESCRIPTION: final processing. What is more practical to do now
// than during parsing.
// ARGUMENTS: None
// RETURNS: Nothing
// NOTE: this function belongs in StlTransfer.cc but since
// StlParserTransferCommand is a templated class, and this is fully
// specialized, we would get duplicate definitions when making
// explicit instantiations
//////////////////////////////////////////////////////////////////
template<>
void
StlParserTransferCommand<Sonics::BigInt<256>,snx_uint64_t>::wrapup()
{
    bool isRead = ( request.MCmd == OCP_MCMD_RD ||
                    request.MCmd == OCP_MCMD_RDEX ||
                    request.MCmd == OCP_MCMD_RDL );
    // add the data
    if ( request.DataLength == 1 && isRead && (
             m_currentDataMask.any() || m_currentDataInfoMask != 0 ) ) {
        addData( m_currentData, m_currentDataInfo );
    } else {
        // process the data spec
        for ( uint32_t i=0; i < request.DataLength; ++i ) {
            DataType data;
            DataInfoType dataInfo(0);
            uint32_t wordWdth = 8 * sizeof (uint32_t);
            switch ( m_currentDataSpec ) {
            case NDATA: data = i;                       break;
            case RDATA: data = StlParser::getRandom();  
              for (uint32_t j = wordWdth; j < (uint32_t)ocpParam( P_data_wdth ); j += wordWdth) {
              data <<= wordWdth;
              data |= StlParser::getRandom();
              }
              break;
            case CDATA: data = m_currentData;           break;
            case ADATA: {
	      if ( request.MBurstSeq == OCP_MBURSTSEQ_STRM )
		data = request.MAddr;
	      else
		data = request.MAddr + (i*ocpParam( P_byteen_wdth ) );
	      break;
	    }
            default: data = m_currentData;
            }
            switch ( m_currentDataInfoSpec ) {
            case NDI: dataInfo = i;                       break;
            case RDI: dataInfo = StlParser::getRandom();  break;
            case CDI: dataInfo = m_currentDataInfo;       break;
            default: dataInfo = m_currentDataInfo;
            }
        addData( data, dataInfo );
        }
    }

    assert( !dataMask.empty() );
    _dataMask()[0] = m_currentDataMask;
    if ( ocpParam( P_sdatainfo ) ) {
        assert( !dataInfoMask.empty() );
        // adjust the mask to the datainfo width
        m_currentDataInfoMask &= static_cast<ByteEnType>(
            ( 1ULL << ocpParam( P_sdatainfo_wdth ) ) - 1 );
        _dataInfoMask()[0] = m_currentDataInfoMask;
    }

    if ( m_randByteEn )
        _request().MByteEn = StlParser::getRandom();

    // adjust the byte enable to the width
    int byteenWdth = (int)( ceil( (double) m_transferWidth ) * 1.0 / 8 );
    _request().MByteEn &= static_cast<ByteEnType>( ( 1ULL << byteenWdth ) - 1 );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParser::readLine
// DESCRIPTION: Processes one line of STL input
// ARGUMENTS: const string, must be one full line of STL
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
void
StlParser::readLine( const string& stlLine )
{
    m_commandType  = STL_INVALID;
    m_commandDelay = 0;

    // skip blank lines
    if ( stlLine.find_first_not_of( " \r\t" ) == string::npos ) {
        return;
    }

    // Parsing
    m_transferCmd.clear();
    m_signalCmd.clear();
    m_waitCmd.clear();
    m_controlStatusCmd.clear();

    parse_info<> info = boost::spirit::parse( stlLine.c_str(), *this, space_p );


    // Check version
    if ( m_commandType != STL_INVALID && !m_versionChecked && !m_versionWarned ) {
        string error( "Correct STL version (2.0) has not been specified\n" );
        reportError( stlLine, error );
	m_versionWarned = true;
    }

    if ( !info.full ) {
        m_commandType = STL_INVALID;
        m_transferCmd.clear();
        m_signalCmd.clear();
        m_waitCmd.clear();
        m_controlStatusCmd.clear();
        reportError( stlLine, info );
        return;
    }

    // Semantic validation
    string errors;
    switch ( m_commandType ) {
    case STL_TRANSFER:
        switch ( m_transferCmd.reportError( errors ) ) {
        default: m_commandType = STL_INVALID;
        case -1: reportError( stlLine, errors );
        case  0: if ( STL_INVALID != m_commandType) {
                     m_transferCmd.wrapup();
                     m_pCommandConverter->convert();
	         }
        }
        break;
    case STL_SIGNAL:
        switch ( m_signalCmd.reportError( errors ) ) {
        default: m_commandType = STL_INVALID;
        case -1: reportError( stlLine, errors );
        case  0: if ( STL_INVALID != m_commandType) {
                     m_signalCmd.wrapup();
                 }
	}
        break;
    case STL_WAIT:
        switch ( m_waitCmd.reportError( errors ) ) {
        default: m_commandType = STL_INVALID;
        case -1: reportError( stlLine, errors );
        case  0: if ( STL_INVALID != m_commandType) {
	             // OK
                 }
	}
        break;
    case STL_RESET:
        if ( !ocpParam( P_mreset ) ) {
            reportError( stlLine, string( "mreset is not enabled\n" ) );
            m_commandType = STL_INVALID;
        }
        break;
    case STL_CONTROLSTATUS:
        switch ( m_controlStatusCmd.reportError( errors ) ) {
        default: m_commandType = STL_INVALID;
        case -1: reportError( stlLine, errors );
        case  0: if ( STL_INVALID != m_commandType) {
                     m_controlStatusCmd.wrapup();
                 }
	}
        break;
    default:
        break;
    }
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParser::getXCommand
// DESCRIPTION: Retrieves the current STL command
// ARGUMENTS: None
// RETURNS: Current command
// PRECONDITIONS: The requested type of command must match the
// value of getCommandType()
//////////////////////////////////////////////////////////////////
StlWaitCommand
StlParser::getWaitCommand() const {
    assert( m_commandType == STL_WAIT );
    return m_waitCmd;
}

StlSignalCommand
StlParser::getSignalCommand() const {
    assert( m_commandType == STL_SIGNAL );
    return m_signalCmd;
}

StlResetCommand
StlParser::getResetCommand() const {
    assert( m_commandType == STL_RESET );
    return m_resetCmd;
}

StlControlStatusCommand
StlParser::getControlStatusCommand() const {
    assert( m_commandType == STL_CONTROLSTATUS );
    return m_controlStatusCmd;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParser::reportError
// DESCRIPTION: self-explanatory
// ARGUMENTS: line where error occurred, and parse_info result
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
void
StlParser::reportError( const string& stlLine, parse_info<> info )
{
    assert( !info.full );
    assert( m_pErr != NULL );
    ostream& err = *m_pErr;
    err << "Syntax error:" << "\n"
        << stlLine << "\n";
    const string where = string( strlen( info.stop )+1, ' ' );
    err << where << "^" << endl;
}

void
StlParser::reportError( const string& stlLine, const string& errors )
{
    // errors should terminate in "\n"
    assert( m_pErr != NULL );
    ostream& err = *m_pErr;
    err << errors << stlLine  << "\n";
}

//////////////////////////////////////////////////////////////////
// FUNCTION:convertChipDMInterfaceToParamSet
// DESCRIPTION: creates an ocp parameter set from an ocp interface
// configuration structure
// ARGUMENTS: OCPParameters structure
// RETURNS: ParamSet<OcpIp::ParamName>
// SIDE EFFECTS: None
//////////////////////////////////////////////////////////////////
OcpParams
OcpIp::convertInterfaceToParamSet( const OCPParameters& intParams )
{
    OcpParams params;
#define COPYPARAM( name )                               \
    params.setParam( P_##name, intParams. name );


    COPYPARAM( broadcast_enable );
    COPYPARAM( burst_aligned );
    COPYPARAM( burstseq_dflt1_enable );
    COPYPARAM( burstseq_dflt2_enable );
    COPYPARAM( burstseq_incr_enable );
    COPYPARAM( burstseq_strm_enable );
    COPYPARAM( burstseq_unkn_enable );
    COPYPARAM( burstseq_wrap_enable );
    COPYPARAM( burstseq_xor_enable );
    //COPYPARAM( burstseq_blck_enable );
    COPYPARAM( endian );
    COPYPARAM( force_aligned );
    COPYPARAM( mthreadbusy_exact );
    COPYPARAM( rdlwrc_enable );
    COPYPARAM( read_enable );
    COPYPARAM( readex_enable );
    COPYPARAM( sdatathreadbusy_exact );
    COPYPARAM( sthreadbusy_exact );
    COPYPARAM( write_enable );
    COPYPARAM( writenonpost_enable );
    COPYPARAM( datahandshake );
    COPYPARAM( reqdata_together );
    COPYPARAM( writeresp_enable );
    COPYPARAM( addr );
    COPYPARAM( addr_wdth );
    COPYPARAM( addrspace );
    COPYPARAM( addrspace_wdth );
    COPYPARAM( atomiclength );
    COPYPARAM( atomiclength_wdth );
    COPYPARAM( burstlength );
    COPYPARAM( burstlength_wdth );
    //COPYPARAM( blockheight );
    //COPYPARAM( blockheight_wdth );
    //COPYPARAM( blockstride );
    //COPYPARAM( blockstride_wdth );
    COPYPARAM( burstprecise );
    COPYPARAM( burstseq );
    COPYPARAM( burstsinglereq );
    COPYPARAM( byteen );
    COPYPARAM( cmdaccept );
    COPYPARAM( connid );
    COPYPARAM( connid_wdth );
    COPYPARAM( dataaccept );
    COPYPARAM( datalast );
    //COPYPARAM( datarowlast );
    COPYPARAM( data_wdth );
    COPYPARAM( mdata );
    COPYPARAM( mdatabyteen );
    COPYPARAM( mdatainfo );
    COPYPARAM( mdatainfo_wdth );
    COPYPARAM( mdatainfobyte_wdth );
    COPYPARAM( sdatathreadbusy );
    COPYPARAM( mthreadbusy );
    COPYPARAM( reqinfo );
    COPYPARAM( reqinfo_wdth );
    COPYPARAM( reqlast );
    //COPYPARAM( reqrowlast );
    COPYPARAM( resp );
    COPYPARAM( respaccept );
    COPYPARAM( respinfo );
    COPYPARAM( respinfo_wdth );
    COPYPARAM( resplast );
    //COPYPARAM( resprowlast );
    COPYPARAM( sdata );
    COPYPARAM( sdatainfo );
    COPYPARAM( sdatainfo_wdth );
    COPYPARAM( sdatainfobyte_wdth );
    COPYPARAM( sthreadbusy );
    COPYPARAM( threads );
    //COPYPARAM( tags );
    //COPYPARAM( taginorder );
    COPYPARAM( control );
    COPYPARAM( controlbusy );
    COPYPARAM( control_wdth );
    COPYPARAM( controlwr );
    COPYPARAM( interrupt );
    COPYPARAM( merror );
    COPYPARAM( mflag );
    COPYPARAM( mflag_wdth );
    COPYPARAM( mreset );
    COPYPARAM( serror );
    COPYPARAM( sflag );
    COPYPARAM( sflag_wdth );
    COPYPARAM( sreset );
    COPYPARAM( status );
    COPYPARAM( statusbusy );
    COPYPARAM( statusrd );
    COPYPARAM( status_wdth );
    COPYPARAM( sthreadbusy_pipelined );
    COPYPARAM( mthreadbusy_pipelined );
    COPYPARAM( sdatathreadbusy_pipelined );
#undef COPYPARAM

    params.setParam( P_byteen_wdth, int( ceil( params.getParam( P_data_wdth )
                                               * 1.0 / 8 ) ) );
    return params;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlParser::setRandomSeed
// DESCRIPTION: static function, there is only one generator for
// the system. Self explanatory
//////////////////////////////////////////////////////////////////
void
StlParser::setRandomSeed( uint32_t seed )
{
  s_randGenerator.seed( (uint64_t)seed );
}

uint32_t
StlParser::getRandom()
{
    uint32_t ret = s_randGenerator();
    return ret;
}
