// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//      Authors : Yann Bajot, PROSILOG
//         $Id: simple_example.cpp,v 1.2 2004/09/05 17:57:56 halexan Exp $
//
//  Description:  Simple Point to point TL2 example using the OCP-specific TL2
//                API. See section 7 of the OCP API documentation for more
//                details
// ============================================================================

//#define DEBUG_C
//#define TRACE_C

#include <systemc.h>
#include "ocp_tl2_channel.h"  
#include "ocp_tl2_master_port.h"  
#include "ocp_tl2_slave_port.h"  

// ----------------------------------------------------------------------------
// SIMPLE MASTER MODULE:
//
//  * non-pipelined module: uses one thread for request and response
//  * uses an OCP-specific TL2 master port
//  * uses a'OCPRequestGrp' structure to pass all the request signals to the
//      channel, and a 'OCPResponseGrp' structure to store the response
//      signals
//  * uses a single OCP thread
//  * performs the following actions:
//      - sends a 10-length WRITE burst to the slave using
//        sendOCPRequestBlocking(). Only one chunk is used (i.e. transaction is
//        atomic).
//      - sends a 10-length READ burst to the slave using
//        sendOCPRequestBlocking(). Only one chunk is used (i.e. transaction is
//        atomic).
//      - waits and get the corresponding response using two successive
//        getOCPResponseBlocking() calls catching 5-length chunks.
//      - performs a complete 20-length WRITE transaction using the serialized
//        method 'OCPWriteTransfer()'. This call includes the following phases:
//        request send, request acknowledge.
//      - performs a complete 20-length READ transaction using the serialized
//        method 'OCPReadTransfer()'. This call includes the followig phases:
//        request send, request acknowledge, response get and response
//        acknowledge
// 
// ----------------------------------------------------------------------------


SC_MODULE(master) {

    // OCP-specific TL2 specialized master port 
    OCP_TL2_MasterPort<unsigned int, unsigned int> ocp;

    // this module has SC processes
    SC_HAS_PROCESS(master);

    // ----------------------------------------------------------------------------
    // Request/Response thread  
    // ----------------------------------------------------------------------------

    void request_response_thread () {
            bool ret_value = true;
            unsigned int ReqChunkLen = 1;
            bool ReqChunkLast = true;
            unsigned int RespChunkLen = 1;
            bool RespChunkLast = true;

            // ---------------------------------------------------------
            // (1) Prepares the first request (10-length WRITE burst)
            // ---------------------------------------------------------
            
            // Reset the request structure
            m_req.reset();
            
            // Set the members of the request group stucture
            m_req.MCmd            =     OCP_MCMD_WR;    // this is a WRITE transaction
            m_req.MDataPtr        =     wdata;          // pass a pointer to the master data array

            // Sets TL2-specific fields
            // this request is sent using a single TL2 chunk
            ReqChunkLen = 10; ReqChunkLast = true;

            // ---------------------------------------------------------
            // (2) Sends the request and test everything is OK
            // ---------------------------------------------------------

            cout << "Master starts sending request 1 -- " << sc_time_stamp() << endl;
            ret_value=ocp->sendOCPRequestBlocking(m_req,ReqChunkLen,ReqChunkLast);
            assert(ret_value);
            cout << "Master ends sending request 1 -- " << sc_time_stamp() << endl;

            // wait some time
            wait(100,SC_NS);

            // ---------------------------------------------------------
            // (3) Prepares the second request (10-length READ burst): 
            // ---------------------------------------------------------
            
            // Set the members of the request group stucture that have changed
            m_req.MCmd            =     OCP_MCMD_RD;    // this is a READ transaction

            // Sets TL2-specific fields
            ReqChunkLen = 10; ReqChunkLast = true;

            // ---------------------------------------------------------
            // (4) Sends the request and test everything is OK
            // ---------------------------------------------------------

            cout << "Master starts sending request 2 -- " << sc_time_stamp() << endl;
            ret_value=ocp->sendOCPRequestBlocking(m_req,ReqChunkLen,ReqChunkLast);
            assert(ret_value);
            cout << "Master ends sending request 2 -- " << sc_time_stamp() << endl;

            // ---------------------------------------------------------
            // (5) Waits and gets the first response chunk
            // ---------------------------------------------------------
            
            // The response will be not released immediatly 
            bool release_immediatly = false;

            // get the first response chunk and retrieves chunk parameters
            cout << "Master starts getting response chunk 1 -- " << sc_time_stamp() << endl;
            ret_value=ocp->getOCPResponseBlocking(m_resp,release_immediatly,RespChunkLen,RespChunkLast);
            assert(ret_value);

            // This a small regression test. 
            assert(RespChunkLen == 5); assert(RespChunkLast == false);

            // Display the received data
            for ( unsigned int i=0; i< RespChunkLen; i+=1 ) {
                cout << m_resp.SDataPtr[i] << " " ;
            }
            cout << endl;

            // release the response after some delay
            wait(100,SC_NS);
            ocp->putMRespAccept();
            cout << "Master ends getting response chunk 1 -- " << sc_time_stamp() << endl;

            // ---------------------------------------------------------
            // (6) Waits and gets the second response chunk
            // ---------------------------------------------------------

            cout << "Master starts getting response chunk 2 -- " << sc_time_stamp() << endl;
            ret_value=ocp->getOCPResponseBlocking(m_resp,release_immediatly,RespChunkLen,RespChunkLast);
            
            // This a small regression test. 
            assert(ret_value); assert(RespChunkLen == 5); assert(RespChunkLast == true);

            // Display the received data
            for (unsigned int  i=0; i< RespChunkLen; i+=1 ) {
                cout << m_resp.SDataPtr[i] << " " ;
            }
            cout << endl;

            // release the response after some delay
            wait(100,SC_NS);
            ocp->putMRespAccept();
            cout << "Master ends getting response chunk 2 -- " << sc_time_stamp() << endl;

            // wait some time
            wait(100,SC_NS);


            // --------------------------------------------------------------------
            // (7) Prepares the WRITE complete transaction (20-length WRITE burst) 
            // --------------------------------------------------------------------
            
            // Set the members of the request group stucture that have changed
            m_req.MCmd            =     OCP_MCMD_WR;    // this is a WRITE transaction
            m_req.MDataPtr        =     wdata;          // pass a pointer to the master data array
                
            // Sets the WRITE transaction length 
            unsigned int Transaction_length = 20;

            // --------------------------------------------------------------------
            // (8) Sends the WRITE transaction and test everything is OK
            // --------------------------------------------------------------------

            cout << "Master starts sending WRITE TRANSACTION -- " << sc_time_stamp() << endl;
            ret_value=ocp->OCPWriteTransfer(m_req,Transaction_length);
            assert(ret_value);
            cout << "Master ends sending WRITE TRANSACTION -- " << sc_time_stamp() << endl;

            // wait some time
            wait(100,SC_NS);

            // --------------------------------------------------------------------
            // (9) Prepares the READ complete transaction (20-length READ burst): 
            // --------------------------------------------------------------------
            
            // Set the members of the request group stucture that have changed
            m_req.MCmd            =     OCP_MCMD_RD;    // this is a WRITE transaction
                
            // Sets the READ transaction length 
            Transaction_length = 20;

            // --------------------------------------------------------------------
            // (10) Sends the READ transaction and test everything is OK
            // --------------------------------------------------------------------

            cout << "Master starts sending READ TRANSACTION -- " << sc_time_stamp() << endl;
            ret_value=ocp->OCPReadTransfer(m_req,m_resp,Transaction_length);
            assert(ret_value);
            cout << "Master ends sending READ TRANSACTION -- " << sc_time_stamp() << endl;

            // Display the received data
            for (unsigned int  i=0; i< Transaction_length; i+=1 ) {
                cout << m_resp.SDataPtr[i] << " " ;
            }
            cout << endl;


            cout << "MASTER THREAD finished -- " << sc_time_stamp() << endl;

    }

    // ----------------------------------------------------------------------------
    // Constuctor
    // ----------------------------------------------------------------------------
    master(sc_module_name mod):
        sc_module(mod),
    ocp("Master_Port")
    {
        // Init Master Data array
        for (unsigned int i=0;i<100;i++)
            wdata[i] = i;
        SC_THREAD(request_response_thread);
    }

    // ----------------------------------------------------------------------------
    // Internal class members 
    // ----------------------------------------------------------------------------
    private:
    
    // Array storing data to be sent
    unsigned int wdata[100];

    // OCP Request/Response structures
    OCPRequestGrp<unsigned int, unsigned int> m_req;
    OCPResponseGrp<unsigned int> m_resp;

};

// ----------------------------------------------------------------------------
// SIMPLE SLAVE MODULE:
// 
//  * non-pipelined module: uses one thread for request and response
//  * uses an OCP-specific TL2 slave port
//  * uses an'OCPRequestGrp' structure to store all the request signals from the
//      channel, and an 'OCPResponseGrp' structure to send response signals
//  * performs the following actions:
//      - receives a 10-length WRITE burst from the master, and stores the
//          received data in an internal array.
//      - receives a 10-length READ burst from the master,and sends the
//        response using two consecutive response chunks (5-length each) with a
//        different 'SRespInfo' signal value.
//      - receives a 20-length WRITE burst from the master, and stores the
//          received data in an internal array.
//      - receives a 20-length READ burst from the master,and sends the
//        response using one response call
// 
// ----------------------------------------------------------------------------

SC_MODULE(slave) {

    // OCP-specific TL2 specialized master port 
    OCP_TL2_SlavePort<unsigned int,unsigned int> ocp;

    // this module has SC processes
    SC_HAS_PROCESS(slave);


    // ----------------------------------------------------------------------------
    // Request/Response thread
    // ----------------------------------------------------------------------------
    void request_response_thread () {
            bool ret_value = true;
            unsigned int ReqChunkLen = 1;
            bool ReqChunkLast = true;
            unsigned int RespChunkLen = 1;
            bool RespChunkLast = true;

            
            // ---------------------------------------------------------
            // (1) Waits and gets the first request (10-length WRITE)
            // ---------------------------------------------------------
            
            //  We will release the request channel after some delay
            bool release_immediatly = false;

            // get the first request chunk and retrieves chunk parameters
            cout << "Slave starts getting request 1 -- " << sc_time_stamp() << endl;
            ret_value=ocp->getOCPRequestBlocking(m_req,release_immediatly,ReqChunkLen,ReqChunkLast);

            // This a small regression test. 
            assert(ret_value); assert(m_req.MCmd == OCP_MCMD_WR);
            assert(ReqChunkLen == 10); assert(ReqChunkLast == true);

            // Display the received data
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                cout << m_req.MDataPtr[i] << " " ;
            }
            cout << endl;

            // Copy the received data into the internal buffer
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                data[i] = m_req.MDataPtr[i];
            }

            // releases the request after some delay 
            wait(100,SC_NS);
            ocp->putSCmdAccept();
            cout << "Slave ends getting request 1 -- " << sc_time_stamp() << endl;


            // ---------------------------------------------------------
            // (2) Waits and gets the second request (10-length READ) 
            // ---------------------------------------------------------
            
            // Now we release the request immediatly
            release_immediatly = true;

            // get the first request chunk and retrieves the chunk parameters
            cout << "Slave starts getting request 2 -- " << sc_time_stamp() << endl;
            ret_value=ocp->getOCPRequestBlocking(m_req,release_immediatly,ReqChunkLen,ReqChunkLast);
            cout << "Slave ends getting request 2 -- " << sc_time_stamp() << endl;

            // This a small regression test. 
            assert(ret_value); assert(m_req.MCmd == OCP_MCMD_RD);
            assert(ReqChunkLen == 10); assert(ReqChunkLast == true);

            // ---------------------------------------------------------
            // (3) Sends the response using 2 chunks
            // ---------------------------------------------------------
            
            // Reset the response structure
            m_resp.reset();

            // Set the members of the response group stucture
            m_resp.SResp          =     OCP_SRESP_DVA;      // Response is VALID
            m_resp.SDataPtr       =     data;               // pass a pointer to the slave data
            m_resp.SRespInfo      =     1;                  // Info for the first chunk

            // Sends the first chunk 
            RespChunkLen = 5; RespChunkLast = false;

            cout << "Slave starts sending Response 1 chunk 1 -- " << sc_time_stamp() << endl;
            ret_value=ocp->sendOCPResponseBlocking(m_resp,RespChunkLen,RespChunkLast);
            cout << "Slave ends sending Response 1 chunk 1 -- " << sc_time_stamp() << endl;
            assert(ret_value);

            // Sends the second chunk 
            m_resp.SDataPtr       =     &data[5];           // pass a pointer to the second part of the slave data
            m_resp.SRespInfo      =     2;                  // Info for the second chunk
            RespChunkLen = 5; RespChunkLast = true;         // this is the last chunk of the response burst

            cout << "Slave starts sending Response 1 chunk 2 -- " << sc_time_stamp() << endl;
            ret_value=ocp->sendOCPResponseBlocking(m_resp,RespChunkLen,RespChunkLast);
            cout << "Slave ends sending Response 1 chunk 2 -- " << sc_time_stamp() << endl;
            assert(ret_value);

            // ---------------------------------------------------------
            // (4) Waits and gets the third request (20-length WRITE)
            // ---------------------------------------------------------
            
            //  We will release the request channel after some delay
            release_immediatly = false;

            // get the first request chunk and retrieves chunk parameters
            cout << "Slave starts getting request 3 -- " << sc_time_stamp() << endl;
            ret_value=ocp->getOCPRequestBlocking(m_req,release_immediatly,ReqChunkLen,ReqChunkLast);
            assert(ret_value);

            // Regression test
            assert(m_req.MCmd == OCP_MCMD_WR);
            assert(ReqChunkLen == 20); assert(ReqChunkLast == true);

            // Display the received data
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                cout << m_req.MDataPtr[i] << " " ;
            }
            cout << endl;

            // Copy the received data into the internal buffer
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                data[i] = m_req.MDataPtr[i];
            }

            // releases the request after some delay 
            wait(100,SC_NS);
            ocp->putSCmdAccept();
            cout << "Slave ends getting request 3 -- " << sc_time_stamp() << endl;


            // ---------------------------------------------------------
            // (5) Waits and gets the fourth request (20-length READ) 
            // ---------------------------------------------------------
            
            // Now we release the request immediatly
            release_immediatly = true;

            // gets the first request chunk and retrieves chunk parameters
            cout << "Slave starts getting request 4 -- " << sc_time_stamp() << endl;
            ret_value=ocp->getOCPRequestBlocking(m_req,release_immediatly,ReqChunkLen,ReqChunkLast);
            assert(ret_value);

            // Regression test
            assert(m_req.MCmd == OCP_MCMD_RD);
            assert(ReqChunkLen == 20); assert(ReqChunkLast == true);

            cout << "Slave ends getting request 4 -- " << sc_time_stamp() << endl;

            // Delay before issuing the response
            wait(100,SC_NS);
            
            // ---------------------------------------------------------
            // (4) Sends the response using only one chunk 
            // ---------------------------------------------------------
            
            // Set the members of the response group stucture
            m_resp.SResp          =     OCP_SRESP_DVA;      // Response is VALID
            m_resp.SDataPtr       =     data;               // pass a pointer to the slave data
            m_resp.SRespInfo      =     1;                  // Info for the first chunk

            // Sends the first and only chunk 
            RespChunkLen = 20; RespChunkLast = true;

            cout << "Slave starts sending Response 2 chunk 1 -- " << sc_time_stamp() << endl;
            ret_value=ocp->sendOCPResponseBlocking(m_resp,RespChunkLen,RespChunkLast);
            cout << "Slave ends sending Response 2 chunk 1 -- " << sc_time_stamp() << endl;
            assert(ret_value);
            
            cout << "SLAVE THREAD finished -- " << sc_time_stamp() << endl;

    }

    // ----------------------------------------------------------------------------
    // constuctor
    // ----------------------------------------------------------------------------
    slave(sc_module_name mod):
        sc_module(mod),
    ocp("Sale_Port")
    {
        SC_THREAD(request_response_thread);
    }

    // ----------------------------------------------------------------------------
    // Internal class members 
    // ----------------------------------------------------------------------------
    private:
    // slave memory 
    unsigned int data[100];

    // OCP Request/Response structures
    OCPRequestGrp<unsigned int, unsigned int> m_req;
    OCPResponseGrp<unsigned int> m_resp;
};

// ----------------------------------------------------------------------------
// MAIN PROGRAM
// ----------------------------------------------------------------------------

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}

  // Performs some basic checks on the parameter values
  // To be completed....
  template <class TdataCl> 
  bool ConfigChecking(ParamCl<TdataCl>* pcl) {
     bool configError = false;
      
     // 'Exact ThreadBusy' conditions (OCP 2.0 spec. page 49)
     if (pcl->sthreadbusy_exact==1 && !(pcl->cmdaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sthreadbusy_exact==1 and cmdaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sthreadbusy_exact==1 && pcl->datahandshake==1 && pcl->sdatathreadbusy==0 && !(pcl->dataaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sthreadbusy_exact==1,datahandshake==1,sdatathreadbusy==0 and dataaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sdatathreadbusy_exact==1 && !(pcl->dataaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sdatathreadbusy_exact==1 and dataaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->mthreadbusy_exact==1 && !(pcl->respaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      mthreadbusy_exact==1 and respaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sthreadbusy_exact==0 && pcl->cmdaccept== 0 && pcl->sthreadbusy==1 )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sthreadbusy_exact is set to 0, cmdaccept is set to 0, but sthreadbusy is set to 1" ;
         configError = true;
     }
     if (pcl->mthreadbusy_exact==0 && pcl->respaccept== 0 && pcl->mthreadbusy==1 )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      mthreadbusy_exact is set to 0, respaccept is set to 0, but mthreadbusy is set to 1" ;
         configError = true;
     }

     return !configError;
      
  };



int sc_main(int, char*[])
{

    // Creates the OCP TL2 channel
    OCP_TL2_Channel<unsigned int, unsigned int>  ch0("ch0");

    // Set the OCP parameters for this channel
    MapStringType  ocpParamMap;
    readMapFromFile("ocpParams", ocpParamMap);
    ch0.setConfiguration(ocpParamMap);

    // performs some basic checks on the OCP parameter values
    ParamCl<OCP_TL2_DataCl<unsigned int, unsigned int> > *pcl = ch0.GetParamCl();
    assert(ConfigChecking(pcl));

    // Creates masters and slaves 
    slave sl1("sl1");
    master ms1("ms1");

    // Connect masters and slaves using channels
    ms1.ocp(ch0);
    sl1.ocp(ch0);

    // Starts simulation 
    sc_start(20000, SC_MS);

    return(0);
}



