// 
//  (c) Copyright OCP-IP 2003, 2004, 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapters
//      Authors : Herve Alexanian, SONICS, herve@sonicsinc.com
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//		  Yann Bajot, PROSILOG, bajot@prosilog.com
//
//  Description : Transaction Level - Layer-1 to Layer-2 OCP Slave Adapter
//  $Id:
//	 
//  The TL1/TL2 slave adapter is a systemc module with a OCP TL1 master port
//  port and a OCP TL2 slave port.
//  It must be connected to:
//      - a TL2 master through a TL2 channel
//      - a TL1 slave  through a TL1 channel
//
//  The adapter is defined in a templated class OCP_TL1_TL2_Slave_Adapter,
//  and the template argument is the TL1 channel data class.
//
//  Constructor arguments
//   - module name
//   - max_chunk_length: the maximum number of TL1 requests pertaining to the
//     same burst that the adapter aggregates before passing to TL2
//   - max_burst_length and adapter_depth are still present for code
//     compatibility but are deprecated and have no effect
//
// ============================================================================

#ifndef _OCP_TL1_TL2_SLAVE_ADAPTER_H
#define _OCP_TL1_TL2_SLAVE_ADAPTER_H

#include <deque>
#include <vector>
#include "systemc.h"
#include "ocp_globals.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl2_slave_port.h"
#include "ocp_tl1_master_port.h"
#include "ocp_tl_param_cl.h"
#include "ocp_utils.h"
#include "async_signal_relay.h"

//for debugging purposes, print messages on screen
//comment for no messages
#define DEBUG_SLAVE_ADAPTER_TL1

template <typename Td, typename Ta>
class Tl2RequestExpander
{
  public:
    Tl2RequestExpander( const OCPTL2RequestGrp<Td, Ta>& req );
    void setParamCl( const OCPParameters* pParams ) { m_pParams = pParams; }
    void setSequenceCalc( OcpIp::BurstSequence<Ta>& calc ) {
        m_pBurstSequence = &calc;
    }
    bool finished() const { return m_numExpanded == m_tl2Req.DataLength; }
    bool isRead()  const { return m_isRead; }
    bool isWrite() const { return !m_isRead; }
    OCPRequestGrp<Td, Ta> next(); // the workhorse

  private:    
    unsigned int  m_numExpanded;
    bool m_isRead;
    OCPTL2RequestGrp<Td, Ta>  m_tl2Req;
    OCPRequestGrp<Td, Ta>     m_tl1Req;
    OcpIp::BurstSequence<Ta>* m_pBurstSequence;
    const OCPParameters*      m_pParams;
};

template <typename Td, typename Ta>
class Tl2RequestExpanderDh
{
  public:
    Tl2RequestExpanderDh( const OCPTL2RequestGrp<Td, Ta>& req );
    bool finished() const { return m_numExpanded == m_tl2Req.DataLength; }
    OCPDataHSGrp<Td> next(); // the workhorse
    // For request/dh phase ordering
    bool             ready() const {
        return ( m_numExpanded < m_requestsSent );
    }
    void             sentRequest( int number=1 ) {
        if ( m_tl2Req.MBurstSingleReq ) {
            number = m_tl2Req.MBurstLength *
                ( ( m_tl2Req.MBlockHeight > 1 ) ? m_tl2Req.MBlockHeight : 1 );
        }
        m_requestsSent += number;
    }

  private:

    unsigned int  m_numExpanded;
    unsigned int  m_requestsSent;

    OCPTL2RequestGrp<Td, Ta> m_tl2Req;
    OCPDataHSGrp<Td>         m_tl1Dh;
};


template <typename Td, typename Ta>
class Tl1ResponseAggregator
{
  public:
    Tl1ResponseAggregator( const OCPTL2RequestGrp<Td, Ta>& req,
                           Td* pDataPtr );
    bool finished() const { return m_numAggregated == m_tl2Resp.DataLength; }
    void next( const OCPResponseGrp<Td>& tl1Resp ); // the workhorse
    OCPTL2ResponseGrp<Td>  get() {
        return m_tl2Resp;
    }
  private:
    unsigned int           m_numAggregated;
    OCPTL2ResponseGrp<Td>  m_tl2Resp;
};

template <class TdataCl_tl1> class OCP_TL1_TL2_Slave_Adapter
: public sc_module
{
  public:  
    // TL1 data type and address type
    typedef typename TdataCl_tl1::DataType Td;
    typedef typename TdataCl_tl1::AddrType Ta;
    typedef OCP_TL1_MasterIF<TdataCl_tl1>  MasterIF;
    typedef OCP_TL2_SlaveIF<Td, Ta>        SlaveIF;

    // TL2 Slave Port and TL1 Master port
    OCP_TL1_MasterPort< TdataCl_tl1> MasterP;
    OCP_TL2_SlavePort< Td, Ta> SlaveP;
	  
    //clock port
    sc_in_clk clk;

    SC_HAS_PROCESS(OCP_TL1_TL2_Slave_Adapter);

    // constructor. Keep compatible for now
    OCP_TL1_TL2_Slave_Adapter(sc_module_name name, int max_chunk_length,
                              int max_burst_length = -1, int adapter_depth = -1);

    // destructor
    ~OCP_TL1_TL2_Slave_Adapter();

    // SystemC processes
    void MasterRequest();
    void MasterDataHs();
    void SlaveRequest();
    void MasterResponse();
    void SlaveResponse();
    void SCmdAccept();
    void MRespAccept();
    void MputFlags();
    void SputSFlag();
    void SputSError();
    void SputSInterrupt();
    void MputReset();
    void SputReset();

  private:
    void end_of_elaboration();

    //pointer to param class
    const OCPParameters* m_params;
    MTimingGrp m_times;  

    // TL2-TL1 request path
    OCPTL2RequestGrp<Td,Ta> m_tl2Request;
    typedef deque<Tl2RequestExpander<Td, Ta> >    Tl2RequestExpanderQueue;
    vector<Tl2RequestExpanderQueue>               m_tl2RequestExpanderQueues;
    vector<sc_event*>                             m_newTl2ReqEvent;
    vector<sc_event*>                             m_requestSentEvent;
    vector<OcpIp::BurstCounter>                   m_tl2BurstCounter;
    vector<OcpIp::BurstSequence<Ta> >             m_burstSequence;
    sc_event                                      m_tl2CmdAcceptEvent;

    // TL2-TL1 datahandshake path
    typedef deque<Tl2RequestExpanderDh<Td, Ta> >  Tl2RequestExpanderDhQueue;
    vector<Tl2RequestExpanderDhQueue>             m_tl2RequestExpanderDhQueues;
    sc_event                                      m_tl2DataAcceptEvent;

    // TL1-TL2 response path
    OCPResponseGrp<Td> m_tl1Response;
    typedef deque<Tl1ResponseAggregator<Td, Ta> > Tl1ResponseAggregatorQueue;
    vector<Tl1ResponseAggregatorQueue>            m_tl1ResponseAggregatorQueues;
    vector<Td*>                                   m_tl2SDataPool;
    vector<sc_event*>                             m_newTl2RespEvent;
    sc_event                                      m_tl1RespAcceptEvent;

    // parameters given as constructor parameters
    int m_maxChunkLength;   // maximum length of a response chunk

    // only to assign the right ocp thread to each master/datahs SC_THREAD
    // do not use after the threads have started
    int m_tl1RequestThreadId;
    int m_tl1DataHsThreadId;
    int m_tl2ResponseThreadId;

    // pass threadbusy signals through
    OcpIp::PortVoidFunctor< MasterIF, unsigned int,
        &MasterIF::getSThreadBusy> m_bindGetSThreadBusy;
    OcpIp::PortFunctor< SlaveIF, void, unsigned int,
        &SlaveIF::putSThreadBusy> m_bindPutSThreadBusy;
    OcpIp::AsyncSignalRelay<unsigned int>  m_sThreadBusyRelay;

    OcpIp::PortVoidFunctor< SlaveIF, unsigned int,
        &SlaveIF::getMThreadBusy> m_bindGetMThreadBusy;
    OcpIp::PortFunctor< MasterIF, void, unsigned int,
        &MasterIF::putMThreadBusy> m_bindPutMThreadBusy;
    OcpIp::AsyncSignalRelay<unsigned int>  m_mThreadBusyRelay;
};

#endif // _OCP_TL1_TL2_SLAVE_ADAPTER_H
