// 
// Copyright 2003 OCP-IP
// OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//         Date : 10/16/2003
//
//  Description:  Regression test 1 for the OCP-specific TL2 API 
//                Tests that all separate members of the TL2 data class  are
//                correctly sent and read on each side, for both request and
//                response phases and sideband signals. This test use the
//                separate member access methods available in the specialized
//                interfaces 'ocp_tl2_master_if' and 'ocp_tl2_slave_if'
// ============================================================================


//#define DEBUG_C
//#define TRACE_C

#include <systemc.h>
#include "ocp_tl2_channel.h"  
#include "ocp_tl2_master_port.h"  
#include "ocp_tl2_slave_port.h"  

// ----------------------------------------------------------------------------
// MASTER MODULE:
// ----------------------------------------------------------------------------
SC_MODULE(master) {

    // Port definitions
    OCP_TL2_MasterPort<unsigned int, unsigned int> ocp;

    SC_HAS_PROCESS(master);


    // ----------------------------------------------------------------------------
    // Send all the request signals to the slave
    // ----------------------------------------------------------------------------
    void request_thread () {
        bool read_cycle = 0;
        int shift = 0;
        request_trans_nb = 0;

        while (1) {

            ocp->MputMAddr           ( 0+request_trans_nb  ) ;
            ocp->MputMAddrSpace      ( 1+request_trans_nb  ) ;
            ocp->MputMAtomicLength   ( 2+request_trans_nb  ) ;
            ocp->MputMBurstSeq       ( (OCPMBurstSeqType) ((3+request_trans_nb)%8)  ) ;
            ocp->MputMByteEn         ( 4+request_trans_nb  ) ;
            ocp->MputMConnID         ( 6+request_trans_nb  ) ;
            ocp->MputMDataInfo       ( 7+request_trans_nb  ) ;
            ocp->MputMThreadID       ( 8+request_trans_nb  ) ;
            ocp->MputMBurstLength    ( 9+request_trans_nb  ) ;
            ocp->MputMBurstPrecise   ( (10+request_trans_nb)%2 ) ;
            ocp->MputMBurstSingleReq ( (11+request_trans_nb)%2 ) ;
            ocp->MputMReqInfo        ( 12+request_trans_nb ) ;
            ocp->MputMReqLast        ( (13+request_trans_nb)%2 ) ;

            // WRITE
            if (!read_cycle) {
                read_cycle = 1; // Toggle write and read cycles

                bool last_of_a_burst=false;

                if(shift==60)
                    last_of_a_burst=true;

                ocp->MputMData(&wdata[shift], 10, last_of_a_burst); 

                sc_assert(ocp->MputWriteRequestBlocking());

#ifdef DEBUG_C
                cout << "Master write request finished";
                cout << ", data = ";
                for (int i=shift;i<shift+10;i++)
                    cout << wdata[i] <<",";
                cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
                if (shift<90)
                    shift += 10;
                else
                    shift = 0;
            } // READ
            else {
                read_cycle = 0; // Toggle write and read cycles
                ocp->MputMReqChunkLen(10);
                ocp->MputMReqChunkLast( ((request_trans_nb%3)==0) );
                sc_assert(ocp->MputReadRequestBlocking());

#ifdef DEBUG_C
                cout << "Master read request finished";
                cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
            }
            request_trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Check all the received response signals from the slave
    // ----------------------------------------------------------------------------
    void response_thread () { 
        unsigned int len;
        response_trans_nb=0;
        bool last_of_a_burst;

        while (1) {
            if (ocp->MgetResponseBlocking(0)) {
                ocp->Mrelease(sc_time(1000, SC_MS)); 

                sc_assert(ocp->MgetSDataInfo()==0+response_trans_nb);
                sc_assert(ocp->MgetSResp()==(OCPSRespType)((1+response_trans_nb)%4));
                sc_assert(ocp->MgetSRespInfo()==2+response_trans_nb);
                sc_assert(ocp->MgetSRespLast()==(3+response_trans_nb)%2);
                sc_assert(ocp->MgetSThreadID()==4+response_trans_nb);

                if(response_trans_nb%2 == 0) {
                    rdata = ocp->MgetSData(len,last_of_a_burst);
                    sc_assert(last_of_a_burst==true);
                } else {
                    rdata = ocp->MgetSData();
                    last_of_a_burst = ocp->MgetSRespChunkLast();
                    len = ocp->MgetSRespChunkLen();
                    sc_assert(last_of_a_burst==false);
                }
                sc_assert(len==10);

#ifdef DEBUG_C
                cout << "Master gets read response";
                cout << ", data = ";
                for (unsigned int i=0;i<len;i++)
                    cout << rdata[i] <<",";
                cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
            }

            response_trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Check all the received sideband signals from the slave
    // ----------------------------------------------------------------------------
    void get_slave_sb_thread() {  
        slave_sb_trans_nb = 0;

        while(1) {

            wait(slave_sb_start_event);

            // 'SReset_n' test
            wait(ocp->SidebandResetEvent());
            sc_assert(ocp->MgetSReset_n() == slave_sb_trans_nb%2 == 0);

            // 'SFlag' test
            wait(ocp->SidebandSlaveEvent());
            sc_assert(ocp->MgetSFlag() == slave_sb_trans_nb+1);

            // 'SError' test
            wait(ocp->SidebandSlaveEvent());
            sc_assert(ocp->MgetSError() == (slave_sb_trans_nb+2)%2 );

            // 'SInterrupt' test
            wait(ocp->SidebandSlaveEvent());
            sc_assert(ocp->MgetSInterrupt() == (slave_sb_trans_nb+3)%2 );

            // 'SThreadBusy' test
            wait(ocp->SThreadBusyEvent());
            sc_assert(ocp->MgetSThreadBusy() == slave_sb_trans_nb%2 == 0);

            // 'CputStatus' test
            wait(ocp->SidebandStatusEvent());
            sc_assert(ocp->SysgetStatus() == slave_sb_trans_nb + 4);

            // 'CputStatusBusy' test
            wait(ocp->SidebandStatusEvent());
            sc_assert(ocp->SysgetStatusBusy() == slave_sb_trans_nb%2 == 0);

            // 'CputControlBusy' test
            wait(ocp->SidebandControlEvent());
            sc_assert(ocp->SysgetControlBusy() == slave_sb_trans_nb%2 == 0);

#ifdef DEBUG_C
            cout  << "get_slave_sb_thread test passed... " << slave_sb_trans_nb << "\n" ;
#endif

            slave_sb_trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Check all the received sideband signals from the slave
    // ----------------------------------------------------------------------------
    void put_master_sb_thread() {  
        master_sb_trans_nb = 0;

        while(1) {

            wait(master_sb_start_event);

            // 'MReset_n' test
            wait(sc_time(10, SC_MS));
            ocp->MputMReset_n(master_sb_trans_nb%2 == 0);

            // 'MFlag' test
            wait(sc_time(10, SC_MS));
            ocp->MputMFlag(master_sb_trans_nb+1);

            // 'MError' test
            wait(sc_time(10, SC_MS));
            ocp->MputMError(master_sb_trans_nb%2);

            // 'MThreadBusy' test
            wait(sc_time(10, SC_MS));
            ocp->MputMThreadBusy(master_sb_trans_nb%2 == 0);

            // 'SysputControlWr' test
            wait(sc_time(10, SC_MS));
            ocp->SysputControlWr(master_sb_trans_nb%2 == 0);

            // 'SysputControl' test
            wait(sc_time(10, SC_MS));
            ocp->SysputControl(master_sb_trans_nb+3);

            // 'SysputStatusRd' test
            wait(sc_time(10, SC_MS));
            ocp->SysputStatusRd(master_sb_trans_nb%2 == 0);

#ifdef DEBUG_C
            cout  << "put_master_sb_thread test passed... " << master_sb_trans_nb << "\n" ;
#endif

            master_sb_trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Synchronizes 'master sideband' and 'slave sideband' communication phases
    // ----------------------------------------------------------------------------
    void sync_sb() {  
        while(true) {
            master_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
            slave_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
        }
    }


    // ----------------------------------------------------------------------------
    // Constructor 
    // ----------------------------------------------------------------------------
    master(sc_module_name mod):
        sc_module(mod),
    ocp("Master_Port")
    {
        for (int i=0;i<100;i++)
            wdata[i] = i;
        SC_THREAD(get_slave_sb_thread);
        SC_THREAD(put_master_sb_thread);
        SC_THREAD(request_thread);
        SC_THREAD(response_thread);
        SC_THREAD(sync_sb);
    }

    unsigned int wdata[100], *rdata;
    sc_event slave_sb_start_event, master_sb_start_event;

    unsigned int request_trans_nb;
    unsigned int master_sb_trans_nb;
    unsigned int response_trans_nb;
    unsigned int slave_sb_trans_nb;

};



// ----------------------------------------------------------------------------
// SLAVE MODULE:
// ----------------------------------------------------------------------------
#include "tl_slave_if.h"
#include "tl_slave_port.h"

SC_MODULE(slave) {

    // Port definitions
    OCP_TL2_SlavePort<unsigned int,unsigned int> ocp;

    SC_HAS_PROCESS(slave);

    // Globals
    unsigned int * data;
    int len;

    sc_event response_event;

    // ----------------------------------------------------------------------------
    // check all the received request signals from the master
    // ----------------------------------------------------------------------------
    void request_thread () {
        bool read_cycle = 0;
        int shift = 0;
        request_trans_nb=0;

        while (true) {

            sc_assert(ocp->SgetRequestBlocking(0));
            ocp->Srelease(sc_time(1000, SC_MS)); 

            // Check request fields
            sc_assert( ocp->SgetMAddr           ()== 0+request_trans_nb  ) ;
            sc_assert( ocp->SgetMAddrSpace      ()== 1+request_trans_nb  ) ;
            sc_assert( ocp->SgetMAtomicLength   ()== 2+request_trans_nb  ) ;
            sc_assert( ocp->SgetMBurstSeq       ()== (OCPMBurstSeqType) ((3+request_trans_nb)%8) ) ;
            sc_assert( ocp->SgetMByteEn         ()== 4+request_trans_nb  ) ;

            if (!read_cycle) {
                sc_assert( ocp->SgetMCmd            ()== OCP_MCMD_WR ) ;
            } else {
                sc_assert( ocp->SgetMCmd            ()== OCP_MCMD_RD ) ;
            }
            sc_assert( ocp->SgetMConnID         ()== 6+request_trans_nb  ) ;
            sc_assert( ocp->SgetMDataInfo       ()== 7+request_trans_nb  ) ;
            sc_assert( ocp->SgetMThreadID       ()== 8+request_trans_nb  ) ;
            sc_assert( ocp->SgetMBurstLength    ()== 9+request_trans_nb  ) ;
            sc_assert( ocp->SgetMBurstPrecise   ()== (10+request_trans_nb)%2 ) ;
            sc_assert( ocp->SgetMBurstSingleReq ()== (11+request_trans_nb)%2 ) ;
            sc_assert( ocp->SgetMReqInfo        ()== 12+request_trans_nb ) ;
            sc_assert( ocp->SgetMReqLast        ()== (13+request_trans_nb)%2 ) ;

            // If Write request, check the received data
            if (!read_cycle) {
                read_cycle = 1; // Toggle write and read cycles

                // First Method
                unsigned int w;
                bool last_of_a_burst;
                data=ocp->SgetMData(w,last_of_a_burst); 
                sc_assert(w==10);

                if(shift==60) 
                    sc_assert(last_of_a_burst==true);
                else
                    sc_assert(last_of_a_burst==false);

                for ( unsigned int i=0; i<10 ;i++  )
                { sc_assert(data[i]==shift+i); }


                if (shift<90)
                    shift += 10;
                else
                    shift = 0;

#ifdef DEBUG_C
                cout << "Slave write request finished";
                cout << ", data = ";
                for (int i=0;i<10;i++)
                    cout << data[i] <<",";
                cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif

            } else 
                // Read Request
            {
                read_cycle = 0; // Toggle write and read cycles

                sc_assert(ocp->SgetMReqChunkLen()==10); 
                sc_assert(ocp->SgetMReqChunkLast()==(request_trans_nb%3)==0); 

                response_event.notify(sc_time(100, SC_MS));

#ifdef DEBUG_C
                cout << "Slave read request finished";
                cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif

            }


            request_trans_nb++;

        }
    }

    // ----------------------------------------------------------------------------
    // send all the response signals to the master
    // ----------------------------------------------------------------------------
    void response_thread () {
        response_trans_nb=0;

        while (true) {

            wait (response_event);

            if ( response_trans_nb%2 == 0 )
            {
                ocp->SputSData(data, 10,true );
            }
            else
            {
                ocp->SputSData(data, 5, true );   // Bad values forced
                // Overwrites 'SDataChunkLen' and 'SDataChunkLast' values to test the
                // following functions :
                ocp->SputSRespChunkLen(10);
                ocp->SputSRespChunkLast(false);
            }
            ocp->SputSDataInfo(0+response_trans_nb);
            ocp->SputSResp((OCPSRespType)((1+response_trans_nb)%4));
            ocp->SputSRespInfo(2+response_trans_nb);
            ocp->SputSRespLast((3+response_trans_nb)%2);
            ocp->SputSThreadID(4+response_trans_nb);

            sc_assert(ocp->SputResponseBlocking());

#ifdef DEBUG_C
            cout << "Slave read response finished";
            cout << ", data = ";
            for (int i=0;i<10;i++)
                cout << data[i] <<",";
            cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif

            response_trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // send all the slave sideband signals to the master
    // ----------------------------------------------------------------------------
    void put_slave_sb_thread() {
        slave_sb_trans_nb = 0;

        while(1) {

            wait(slave_sb_start_event);

            // 'SReset_n' test
            wait(sc_time(5, SC_MS));
            ocp->SputSReset_n(slave_sb_trans_nb%2 == 0);

            // 'SFlag' test
            wait(sc_time(10, SC_MS));
            ocp->SputSFlag(slave_sb_trans_nb+1);

            // 'SError' test
            wait(sc_time(10, SC_MS));
            ocp->SputSError((slave_sb_trans_nb+2)%2);

            // 'SInterrupt' test
            wait(sc_time(10, SC_MS));
            ocp->SputSInterrupt((slave_sb_trans_nb+3)%2);

            // 'SThreadBusy' test
            wait(sc_time(10, SC_MS));
            ocp->SputSThreadBusy(slave_sb_trans_nb%2 == 0);

            // 'CputStatus' test
            wait(sc_time(10, SC_MS));
            ocp->CputStatus(slave_sb_trans_nb + 4);

            // 'CputStatusBusy' test
            wait(sc_time(10, SC_MS));
            ocp->CputStatusBusy(slave_sb_trans_nb%2 == 0);

            // 'CputControlBusy' test
            wait(sc_time(10, SC_MS));
            ocp->CputControlBusy(slave_sb_trans_nb%2 == 0);

#ifdef DEBUG_C
            cout  << "put_slave_sb_thread test passed... " << slave_sb_trans_nb << "\n" ;
#endif

            slave_sb_trans_nb ++ ;
        }
    }

    // ----------------------------------------------------------------------------
    // check all the received sideband signals from the master
    // ----------------------------------------------------------------------------
    void get_master_sb_thread() {
        master_sb_trans_nb = 0;

        while(1) {

            wait(master_sb_start_event);

            // 'MReset_n' test
            wait(ocp->SidebandResetEvent());
            sc_assert(ocp->SgetMReset_n() == (master_sb_trans_nb%2 == 0));

            // 'MFlag' test
            wait(ocp->SidebandMasterEvent());
            sc_assert(ocp->SgetMFlag()==master_sb_trans_nb+1);

            // 'MError' test
            wait(ocp->SidebandMasterEvent());
            sc_assert(ocp->SgetMError()==master_sb_trans_nb%2);

            // 'MThreadBusy' test
            wait(ocp->MThreadBusyEvent());
            sc_assert(ocp->SgetMThreadBusy()== (master_sb_trans_nb%2 == 0));

            // 'SysputControlWr' test
            wait(ocp->SidebandControlEvent());
            sc_assert(ocp->CgetControlWr()== (master_sb_trans_nb%2 == 0));

            // 'SysputControl' test
            wait(ocp->SidebandControlEvent());
            sc_assert(ocp->CgetControl()== (master_sb_trans_nb+3));

            // 'SysputStatusRd' test
            wait(ocp->SidebandStatusEvent());
            sc_assert(ocp->CgetStatusRd()== (master_sb_trans_nb%2 == 0));

#ifdef DEBUG_C
            cout  << "get_master_sb_thread test passed... " << master_sb_trans_nb << "\n" ;
#endif

            master_sb_trans_nb++ ;
        }
    }

    // ----------------------------------------------------------------------------
    // Synchronizes 'master sideband' and 'slave sideband' communication phases
    // ----------------------------------------------------------------------------
    void sync_sb() {  
        while(true) {
            master_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
            slave_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
        }
    }

    // ----------------------------------------------------------------------------
    // Constructor 
    // ----------------------------------------------------------------------------
    slave(sc_module_name mod):
        sc_module(mod),
    ocp("Sale_Port")
    {
        SC_THREAD(put_slave_sb_thread);
        SC_THREAD(get_master_sb_thread);
        SC_THREAD(request_thread);
        SC_THREAD(response_thread);
        SC_THREAD(sync_sb);
    }


    sc_event slave_sb_start_event, master_sb_start_event;

    unsigned int request_trans_nb;
    unsigned int master_sb_trans_nb;
    unsigned int response_trans_nb;
    unsigned int slave_sb_trans_nb;
};

// ----------------------------------------------------------------------------
// MAIN PROGRAM
// ----------------------------------------------------------------------------

int sc_main(int, char*[])
{

    OCP_TL2_Channel<unsigned int, unsigned int>  ch0("ch0");

    slave sl1("sl1");
    master ms1("ms1");

    ms1.ocp(ch0);

    sl1.ocp(ch0);

    sc_start(20000, SC_MS);
    if(ms1.request_trans_nb==19 && ms1.response_trans_nb==10
            && ms1.slave_sb_trans_nb==10 && ms1.master_sb_trans_nb
            && sl1.request_trans_nb==20 && sl1.response_trans_nb==10
            && sl1.slave_sb_trans_nb==10 && sl1.master_sb_trans_nb
       )
        cout << endl << "OCP TL2 API Regression test 1 PASSED...." << endl;
    else
        cout << endl << "OCP TL2 API Regression test 1 FAILED...." << endl;
        
    return(0);
}

