////////////////////////////////////////////////////////////////////////
//                                                                    //
// Copyright (c) 2002 Sonics, Inc.                                    //
//                                                                    //
// Confidential and Proprietary Information of Sonics, Inc.           //
// Use, disclosure, or reproduction is prohibited without             //
// written permission from Sonics, Inc.                               //
//                                                                    //
// $Id: SlaveResponseQueue.h,v 1.1 2003/12/01 16:47:39 ybajot Exp $  //
//                                                                    //
////////////////////////////////////////////////////////////////////////

#ifndef _SlaveResponseQueue_h
#define _SlaveResponseQueue_h

#include "ocp_globals.h"

// Set to same value as Q-Slave (max requests outstanding)
// NOTE: would be more robust to move to deque
#define RESPONSE_QUEUE_DEPTH    256

// model a behavior response queue module with enqueue/dequeue events
template <typename TdataCl>
class ResponseQueue {
  public:
    // type definitions
    typedef typename TdataCl::DataType Td;
    typedef typename TdataCl::AddrType Ta;

  private:
    int      m_Head;
    int      m_Tail;
    sc_event m_WaitOnDeQueueEvent;
    bool     m_WaitOnDeQueue;
    sc_event m_WaitOnEnQueueEvent;
    bool     m_WaitOnEnQueue;

    struct {
        OCPResponseGrp<Td> Resp;
        unsigned int chunklen;
        bool chunklast;
        sc_time      SendTime;
        bool         Valid;
    } m_Buffer[RESPONSE_QUEUE_DEPTH];

  public:
    ResponseQueue()
      : m_Head(0),
        m_Tail(0),
        m_WaitOnDeQueue(false),
        m_WaitOnEnQueue(false)
    {
        reset();
    }

    void reset()
    {
        int i;

        for (i = 0; i < RESPONSE_QUEUE_DEPTH; i++) {
            m_Buffer[i].Valid = false;
            m_Buffer[i].Resp.SResp = OCP_SRESP_NULL;
            m_Buffer[i].Resp.SDataPtr = NULL;
            m_Buffer[i].chunklen = true;
            m_Buffer[i].chunklast = true;
        }
        m_Head =0;
        m_Tail =0;
    }

    void enqueueBlocking(const OCPResponseGrp<Td>& resp, unsigned int chunklen, bool chunklast, const sc_time& send_time)
    {
        // check for full
        if (m_Buffer[m_Tail].Valid) {
            if (!m_WaitOnEnQueue) {
                // go to the enqueue waiting state
                m_WaitOnEnQueue = true;
                wait(m_WaitOnEnQueueEvent);
                m_WaitOnEnQueue = false;
            } else {
                cerr << "ERROR: Cannot handle two processes wait to enqueue in SlaveResponseQueue."
                     << endl;
                sc_stop();
            }
        }

        // insertion
        m_Buffer[m_Tail].Valid = true;
        m_Buffer[m_Tail].Resp = resp;
        m_Buffer[m_Tail].SendTime = send_time;
        m_Buffer[m_Tail].chunklen = chunklen;
        m_Buffer[m_Tail].chunklast = chunklast;

        m_Tail++;
        if (m_Tail >= RESPONSE_QUEUE_DEPTH) m_Tail = 0;

        // check for wake-up
        if (m_WaitOnDeQueue) {
            // wait up the wait-on-dequeue process
            m_WaitOnDeQueueEvent.notify(SC_ZERO_TIME);
        }
    }

    void dequeueBlocking(OCPResponseGrp<Td>& resp, unsigned int& chunklen, bool& chunklast, sc_time& send_time)
    {
        // check for empty
        if (!m_Buffer[m_Head].Valid) {
            if (!m_WaitOnDeQueue) {
                // go to the dequeue waiting state
                m_WaitOnDeQueue = true;
                wait(m_WaitOnDeQueueEvent);
                m_WaitOnDeQueue = false;
            } else {
                cerr << "ERROR: Cannot handle two processes wait to dequeue in SlaveResponseQueue.h."
                     << endl;
                sc_stop();
            }
        }

        // deletion
        m_Buffer[m_Head].Valid = false;
        resp = m_Buffer[m_Head].Resp;
        send_time = m_Buffer[m_Head].SendTime;
        chunklen = m_Buffer[m_Head].chunklen;
        chunklast = m_Buffer[m_Head].chunklast;

        m_Head++;
        if (m_Head >= RESPONSE_QUEUE_DEPTH) m_Head = 0;

        // check for wake-up
        if (m_WaitOnEnQueue) {
            // wait up the wait-on-enqueue process
            m_WaitOnEnQueueEvent.notify(SC_ZERO_TIME);
        }
    }

    bool empty()
    {
       return !(m_Buffer[m_Head].Valid);
    }

    bool getHeadSResp(OCPSRespType& headSResp)
    {
        if (empty()) {
            return false;
        }
        headSResp = m_Buffer[m_Head].Resp.SResp;
        return true;
    }

    bool getHeadSendTime(sc_time& headSendTime)
    {
        if (empty()) {
            return false;
        }
        headSendTime = m_Buffer[m_Head].SendTime;
        return true;
    }

    int length()
    {
        // Is the queue empty?
        if (!m_Buffer[m_Head].Valid) {
            return 0;
        }
        int my_len = m_Tail - m_Head;
        if (my_len < 1) {
            my_len += RESPONSE_QUEUE_DEPTH;
        }
        return my_len;
    }

    void purgePlaceholders()
    {
        // Starting at the head of the list, removes any place holder
        // "dummy" posted write responses that have reached their time out
        
        // TODO: Note that it is possible that expired place holder "dummy"
        //       responses are sitting behind a regular response.
        //       For example, if the queue looked like this:
        //       head  --->  Read_Response
        //                   Place_Holder_Expired
        //                   Place_Holder_Expired
        //       tail  --->  Read_Response
        //       Then just looking at the head would miss the expired placeholders
        //       behind it. This circular buffer really needs to be a List or some
        //       other structure that would let us move through the queue and purge
        //       items in the middle if needed.
        
        // Keep removing expired place holders from the head:
        sc_time CurTime = sc_time_stamp();

        while ( (m_Buffer[m_Head].Valid) && 
                (m_Buffer[m_Head].Resp.SResp==OCP_SRESP_NULL) && 
                (m_Buffer[m_Head].SendTime<=CurTime) ) {

            // This is a place holder - delete it.
            m_Buffer[m_Head].Valid = false;
            m_Head++;
            if (m_Head >= RESPONSE_QUEUE_DEPTH) {
                m_Head = 0;
            }
        }
    }

    sc_event& WaitOnDeQueueEvent() {
        return m_WaitOnDeQueueEvent;
    }
};

#endif // _SlaveResponseQueue_h
