// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//         Date : 02/15/2003
//
//  Description : Transaction Level - Layer-1 example Master
//    Features:
//    - Asynchronous master using OCP data class
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//                
// Parameter    :
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - DelayPct: Non-negative number specifying the delay in percent
//                 of a clock cycle, the data are transmitted.
// ============================================================================

#include "ocp_tl1_master_async.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async::OCP_TL1_Master_Async
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async<TdataCl>::OCP_TL1_Master_Async
      (
        sc_module_name name_
      , int ID
      , int Priority
      , double DelayPct
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_DelayPct        (DelayPct)
{
  // InitParameters();

  SC_THREAD(MasterRequest);
  sensitive_pos(clk); 
  SC_THREAD(MasterResponse);
  sensitive << MasterP; 
}

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async::~OCP_TL1_Master_Async
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async<TdataCl>::~OCP_TL1_Master_Async()
{
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::MasterRequest()
//
//  Top level request method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::MasterRequest()
{
  int i = 0;
  int Pause[] = {0, 0, 0, 0}; // # of cycles between requests;
//  int Pause[] = {0, 1, 2, 3}; // # of cycles between requests;
  bool tmp = 0;
  Td Data = 0;
  Ta Addr = 0;
  int cnt = 0;

  while(true)
  {
    wait(); // wait for rising edge of clock

    if (cnt == 0)
    {
      // get new Pause value
      i++;
      cnt = Pause[i%4];

      // check if channel is free
      tmp = !MasterP->MgetSbusy();
      if (tmp)
      {
        // Delay data signal
	// This must work with or without the next line
        // The delay must be less than 1 clk cycle
	if (m_DelayPct > 0)
          wait(TL_CLK_PERIOD * (m_DelayPct/100.0), TL_TIME_UNIT);

        // Put data into the channel
	m_DataCl->MputMAddr(Addr);
	m_DataCl->MputMData(Data);
        MasterP->MputReadRequest(); // Toggles the request data members

#ifdef DEBUG_G1
      cout << "Master sent request "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " data " << Data << endl;
#endif

        Addr += 4;
        Data++;
      }
    }
    else
    {
      cnt--;
    }
  } // end of while loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::MasterResponse()
//
//  Top level response method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::MasterResponse()
{
  Td Rdata;

  while(true)
  {
    wait(); // for default event

    if (MasterP->MgetResponse(true))
    {
      m_DataCl->MgetSData(Rdata);

#ifdef DEBUG_G1
      cout << "Master got response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " data " << Rdata << endl;
#endif
    }
  } // end of while(true) loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Master_Async<TdataCl>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get communication structure
  m_CommCl = MasterP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Master_Async<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
