// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Albert Chitemyan Sonics Inc.
//                
//         $Id: 
//
//  Description : OCP API - Layer-2 test Master
//    Features:
//    - Synchronous
//    - Non-blocking methods are used
// ============================================================================

using namespace std;

#include "master.h"

/////////////////////////////////////////////////////////////////////////////
// 
// This example demonstrates how to use timing extensions for TL2 modeling
// as well as how to use special transactions to transport timing information 
// from master to slave and visa versa.
//
// In this particular example we demonstrate ReqSndI timing annotation usage,
// other timing annotation should be used in the same way.
// 
/////////////////////////////////////////////////////////////////////////////

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Master<BUSWIDTH, ADDRWIDTH>::Master (sc_core::sc_module_name name_): 
  sc_core::sc_module (name_), ip_socket("ip_socket", ocpip::ocp_master_socket_tl2<32>::mm_txn_with_data()), ongoing_req(false), rsp(NULL) {
  
  // initialize common members
  transaction_count = 0;
  
  SC_THREAD(proc);
      sensitive << clk.pos(); 
      dont_initialize();

  SC_THREAD(process_waiting_for_burst_end);
      sensitive << clk.pos(); 

  ip_socket.register_nb_transport_bw(this, &Master::nb_transport);

  // we will use these burst lengths in this example
  burst_len_list[0] = 4;
  burst_len_list[1] = 8;  
  burst_len_list[2] = 16;  

  wait_for_burst_end = false;

}

// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Master<BUSWIDTH, ADDRWIDTH>::~Master() {}

// ----------------------------------------------------------------------------
//  Method : Master::proc()
//
//  Synchronous Master process
//
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void Master<BUSWIDTH, ADDRWIDTH>::proc() 
{
  // do this at the start
  send_timing_info();
  wait();

  while (true)
  {   
      wait();
      if (!ongoing_req && !wait_for_burst_end) 
      {
        generate_burst();

#ifdef DEBUG_TL2
        // burst ready to send, so we can print it
        print_burst();
#endif
 
        // Send burst 
        time  = sc_core::SC_ZERO_TIME;
        phase = tlm::BEGIN_REQ;
        tlm::tlm_sync_enum slave_answer = ip_socket->nb_transport_fw(*req, phase, time);
       
        switch(slave_answer){
          case tlm::TLM_ACCEPTED: 
            ongoing_req = true; 
            break;
          case tlm::TLM_UPDATED: 
            switch (phase){
              case tlm::END_REQ:
                if (req->get_command() == tlm::TLM_WRITE_COMMAND){
                  ip_socket.release_transaction(req); //we are done now because writes do not have resps in this example
                }
                break;
              default:
                std::cerr<<"Error in "<<name()<<" : got unexpected phase update to "<<phase<<" in response to BEGIN_REQ."<<std::endl;
                exit(1);                
            }
            break;
          case tlm::TLM_COMPLETED:;
        }

       wait_for_burst_end = true;

       transaction_count++;
      } // end of if (!ongoing_req)
    
      if (rsp ) 
      {
        time = sc_core::SC_ZERO_TIME;
        phase = tlm::END_RESP;
        ip_socket->nb_transport_fw(*rsp, phase, time);  
    
        ip_socket.release_transaction(rsp); //we are done with this txn now
        rsp = NULL;
      }
  } // end of while
} // end of method

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum Master<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) 
{
  switch(ph){
    case tlm::END_REQ:
      ongoing_req = false;
      if (wait_for_burst_end) 
         cerr << "Burst is in process but slave send END_REQ"    << endl;

      if (req->get_command() == tlm::TLM_WRITE_COMMAND){
        ip_socket.release_transaction(req); //we are done now because writes do not have resps in this example
      }      
      break;
    case tlm::BEGIN_RESP:      
      if (wait_for_burst_end)
         cerr << "Burst request sending is in process but slave started response"    << endl;

      rsp = &txn;
      break;
    default:
      std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_bw"<<std::endl;
      exit(1);
  }
  return tlm::TLM_ACCEPTED;
}


template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::generate_burst() 
{ 
  scv_smart_ptr< unsigned int > rand;
  rand->next();

  req=ip_socket.get_transaction();
  tlm::tlm_command cmd;

  if ((*rand)%2 == 0){
    cmd = tlm::TLM_WRITE_COMMAND; 
    ip_socket.template validate_extension<ocpip::posted>(*req);
  }
  else
    cmd = tlm::TLM_READ_COMMAND;

  rand->next();
  // burst start address
  unsigned int addr = 0x80 * ( 1 + (*rand)%5 );

  burst_len = burst_len_list[ (*rand)%3 ];

  req->set_command(cmd);
  req->set_address(addr);
  ip_socket.reserve_data_size(*req, burst_len * sizeof(Td));

  ocpip::burst_sequence* b_seq;
  ip_socket.template get_extension<ocpip::burst_sequence>(b_seq, *req);
  b_seq->value.sequence = ocpip::INCR;
  ip_socket.template validate_extension<ocpip::burst_sequence>(*req);

  ocpip::burst_length* b_len; 
  ip_socket.template get_extension<ocpip::burst_length>(b_len, *req);
  b_len->value = burst_len;
  ip_socket.template validate_extension<ocpip::burst_length>(*req);

  // generate burst_length data for entire burst
  if (req->get_command() == tlm::TLM_WRITE_COMMAND) 
  {
      unsigned int localdata = 10000 + ( (*rand)/100) * 100;

      for (unsigned int i = 0; i < burst_len; ++i)
          *(((Td*)(req->get_data_ptr())) + i) = localdata++;
  }  

  burst_duration = burst_len * req_send_interval; 
}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::send_timing_info()
{
  // initialize timing annotation for TL2
  req_send_interval = 3;

  timing_txn = ip_socket.get_tl2_timing_transaction();
  timing_ext = ip_socket.template get_extension<ocpip::tl2_timing>(*timing_txn);
  timing_ext->value.master_timing.RqSndI = req_send_interval;
  timing_ph = ocpip::TL2_TIMING_CHANGE;

  sc_core::sc_time zero_time = sc_core::SC_ZERO_TIME;

  // send special txn for timing change
  ip_socket->nb_transport_fw(*timing_txn, timing_ph, zero_time);
  // male sure that slave didn't change phase parameter
  assert(timing_ph == ocpip::TL2_TIMING_CHANGE);

}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::process_waiting_for_burst_end()
{
    while(true) 
    {
       wait();
       if (wait_for_burst_end)
       {
          // we already waited 1 cycle
          for (unsigned int i = 0; i < burst_duration - 1; i++)
             wait();

          wait_for_burst_end = false;
       }
    }
}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Master<BUSWIDTH, ADDRWIDTH>::print_burst()
{
    cout <<
    endl << "------------------------------------------------------------------------------------ time="
         << sc_core::sc_time_stamp() << endl;

    cout << "Master sent burst with length " <<  burst_len;

    if (req->get_command() == tlm::TLM_WRITE_COMMAND) {
      cout << "   WRITE "
           << "   addr=" << req->get_address();

    } else {
      cout << "   READ "
           << "   addr=" << req->get_address();
    }

    cout << endl << "Master will not send new burst: " << burst_duration << " cycles, until burst finishes"
         << endl;
}
// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class Master< 32,32 >;

