///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  Testbench for TL1/TL0 adapters: Simple with SystemC types
//
// TL1 and TL0 initiators and targets are created with identical behaviour for
// the different abstraction levels.
// Six simulations are run, 4 with adapters and 2 without, for 0, 1 and 2
//   cascaded adapters.
// All sets of results should be the same, allowing for a few ps of timing
// delay in some cases.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "ocpip_adapters.h"
#include <iostream>
#include <queue>


using namespace std;
using namespace ocpip;
using namespace sc_core;
using namespace sc_dt;
using namespace tlm;


// all OCPs in this testbench have the same configuration
ocp_parameters gen_ocp_config() {
  ocp_parameters p;
  p.writeresp_enable = true;
  p.addr_wdth = 32;
  p.data_wdth = 32;
  p.mreset = true;

  p.enableclk = 1;
  p.reqinfo = true;
  p.reqinfo_wdth = 10;
  p.mdatainfo = true;
  p.mdatainfo_wdth = 12;
  p.respinfo = true;
  p.respinfo_wdth = 1;
  p.sdatainfo = true;
  p.sdatainfo_wdth = 12;
  p.mflag = true;
  p.mflag_wdth = 2;
  p.sflag = true;
  p.sflag_wdth = 3;
  p.interrupt = true;
  p.merror = true;
  p.serror = true;
  p.sreset = true;
  p.addrspace = true;
  p.addrspace_wdth = 2;
  p.connid = true;
  p.connid_wdth = 5;
  return p;
}


class BASE_TL0_TYPES:
  public tl1_tl0::SIGNAL_DEFAULT_TL0_TYPES<32, 32, sc_bv> {
public:
  typedef bool ENABLE_CLK_T;
  typedef sc_bv<2> MADDRSPACE_T;
  typedef sc_bv<12> MDATAINFO_T;
  typedef sc_bv<10> MREQINFO_T;
  typedef sc_bv<12> SDATAINFO_T;
  typedef sc_bv<1> SRESPINFO_T;
  typedef sc_bv<5> MCONNID_T;
  typedef sc_bv<1> MERROR_T;
  typedef sc_bv<2> MFLAG_T;
  typedef sc_bv<1> SERROR_T;
  typedef sc_bv<3> SFLAG_T;
  typedef sc_bv<1> SINTERRUPT_T;
  typedef sc_bv<1> SRESET_N_T;
};


// TLM extensions for req/resp/data-info
template<class T, int UNIQUE>  struct info: tlm_extension<info<T,UNIQUE> > {
  void free() {delete this;}
  void copy_from(tlm_extension_base const &) {}
  tlm_extension_base *clone() const {return 0;}
  static T &get(tlm_generic_payload &pl) {
    info *e = pl.get_extension<info>();
    if(e == 0) {e = new info; pl.set_extension<info>(e);}
    return e->value;
  }
  T value;
};
typedef info<unsigned, 0> req_info;
typedef info<unsigned, 1> data_info;
typedef info<unsigned, 2> resp_info;
typedef info<unsigned, 3> mflag;
typedef info<unsigned, 4> sflag;


class ADAPTER01_TL0_TYPES: public BASE_TL0_TYPES {
public:
  // functions for user-defined in-band information conversion
  static void conv_mreqinfo(MREQINFO_T &info, tlm::tlm_generic_payload &pl) {
    req_info::get(pl) = info.to_uint();
  }
  static void conv_mdatainfo(MDATAINFO_T &info, tlm::tlm_generic_payload &pl, unsigned beat) {
    data_info::get(pl) = info.to_uint();
  }
  static void conv_srespinfo(SRESPINFO_T &info, tlm::tlm_generic_payload &pl) {
    info = resp_info::get(pl);
  }
  static void conv_sdatainfo(SDATAINFO_T &info, tlm::tlm_generic_payload &pl, unsigned beat) {
    info = data_info::get(pl);
  }
  static void conv_mflag(MFLAG_T &info, tlm::tlm_generic_payload &pl) {
    mflag::get(pl) = info.to_uint();
  }
  static void conv_sflag(SFLAG_T &info, tlm::tlm_generic_payload &pl) {
    info = sflag::get(pl);
  }
};

class ADAPTER10_TL0_TYPES: public BASE_TL0_TYPES {
public:
  // functions for user-defined in-band information conversion
  static void conv_mreqinfo(MREQINFO_T &info, tlm::tlm_generic_payload &pl) {
    info = req_info::get(pl);
  }
  static void conv_mdatainfo(MDATAINFO_T &info, tlm::tlm_generic_payload &pl, unsigned beat) {
    info = data_info::get(pl);
  }
  static void conv_srespinfo(SRESPINFO_T &info, tlm::tlm_generic_payload &pl) {
    resp_info::get(pl) = info.to_uint();
  }
  static void conv_sdatainfo(SDATAINFO_T &info, tlm::tlm_generic_payload &pl, unsigned beat) {
    data_info::get(pl) = info.to_uint();
  }
  static void conv_mflag(MFLAG_T &info, tlm::tlm_generic_payload &pl) {
    info = mflag::get(pl);
  }
  static void conv_sflag(SFLAG_T &info, tlm::tlm_generic_payload &pl) {
    sflag::get(pl) = info.to_uint();
  }
};


// simulation duration determined by this parameter
#define MEM_SIZE 4000
#include "tb_common.h"


class ocp_tl0_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl0_initiator);
public:
  ocp_tl0_initiator(sc_module_name n):
    sc_module(n),
    cycle(0),
    reqs_sent(0),
    next_req_cycle(1),
    resps_rxd(0),
    req_active(false),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    MReset_n.initialize(sc_bv<1>(true));
    MCmd.initialize(sc_bv<3>(0));

    fill_initiator_memory(memory);
    reset[1] = reset[0] = false;
  }

  sc_in<bool> Clk, EnableClk;
  sc_out<sc_bv<32> > MAddr;
  sc_out<sc_bv<3> > MCmd;
  sc_out<sc_bv<32> > MData;
  sc_in<sc_bv<32> > SData;
  sc_in<sc_bv<1> > SCmdAccept;
  sc_in<sc_bv<2> > SResp;
  sc_out<sc_bv<1> > MReset_n;

  sc_out<sc_bv<2> > MAddrSpace;
  sc_out<sc_bv<12> > MDataInfo;
  sc_out<sc_bv<10> > MReqInfo;
  sc_in<sc_bv<12> > SDataInfo;
  sc_in<sc_bv<1> > SRespInfo;
  sc_out<sc_bv<5> > MConnID;
  sc_out<sc_bv<1> > MError;
  sc_out<sc_bv<2> > MFlag;
  sc_in<sc_bv<1> > SError;
  sc_in<sc_bv<3> > SFlag;
  sc_in<sc_bv<1> > SInterrupt;
  sc_in<sc_bv<1> > SReset_n;

private:
  void on_Clk_rising() {
    if(!EnableClk) return;

    if((reqs_sent * 4 + 3 >= MEM_SIZE) && (resps_rxd == reqs_sent)) {
      stop_sim();
      return;
    }

    cycle++;

    // reset the OCP - start cycle selected when 5 outstanding responses
    if(cycle == 9821) {
      os << cycle << " I: sent OCP reset with  " << reqs_sent - resps_rxd << " outstanding" << endl;
      MReset_n = sc_bv<1>(false);
      while(rd_addr.size() > 0) rd_addr.pop();
    }
    if(cycle == 10032) {
      os << cycle << " I: retracted OCP reset" << endl;
      MReset_n = sc_bv<1>(true);
      MCmd = sc_bv<3>(0);
      reqs_sent = 0;
      resps_rxd = 0;
      req_active = false;
    }
    if((cycle >= 9821) && (cycle <= 10032)) return;

    // resynchronise Slave OCP reset
    // can fall (enter reset) asynchronously but rise (leave reset) is clock-
    // aligned and must be detected immediately.
    bool r_in = (SReset_n->read() == sc_bv<1>(false));
    bool in_r = reset[1] && r_in;
    reset[1] = reset[0];
    reset[0] = r_in;
    if(in_r) {
      os << cycle << " I: in reset" << endl;
      // reset all state
      MCmd = sc_bv<3>(0);
      reqs_sent = 0;
      resps_rxd = 0;
      req_active = false;
      while(rd_addr.size() > 0) rd_addr.pop();
      return;
    }

    // flags
    if((cycle % 300) == 1) {
      MFlag = (cycle /300) % 4;
      MError = (cycle / 300) % 2;
    }
    if((cycle % 300) == 2) {
      os << cycle << " I: SFlag " << (SFlag->read()).to_uint() << endl;
      os << cycle << " I: SError " << (SError->read()).to_uint() << endl;
      os << cycle << " I: SInterrupt " << (SInterrupt->read()).to_uint() << endl;
    }

    // request side
    if(req_active && (SCmdAccept->read() == sc_bv<1>(true)))
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if((SCmdAccept->read() == sc_bv<1>(true)) || !req_active) {
      if((cycle >= next_req_cycle) && (reqs_sent * 4 + 3 < MEM_SIZE)) {
        // incrementing word addresses
        unsigned addr = reqs_sent * 4;
        MAddr = addr;
        // alternate WR and RD
        char dir;
        if(((cycle >> 4) * (~cycle >> 8) * cycle) & 8) {
          MCmd = 1;
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          MData = d;
          MDataInfo = sc_bv<12>(cycle & 4095);
          os << cycle << " I: W data: " <<  d << endl;
          rd_addr.push(-1);
          dir = 'W';
        } else {
          MCmd = 2;
          rd_addr.push(addr);
          dir = 'R';
        }
        MReqInfo = sc_bv<10>(reqs_sent);
        MAddrSpace = sc_bv<2>(reqs_sent);
        MConnID = sc_bv<5>(reqs_sent >> 2);
        reqs_sent++;
        req_active = true;
        os << cycle << " I: " << dir << " req sent @ " << addr << endl;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        MCmd = 0;
        req_active = false;
      }
    }

    // response side
    if(!(SResp.read() == 0)) {
      resps_rxd++;
      int addr = rd_addr.front();
      rd_addr.pop();
      char dir;
      if(addr >= 0) {
        *reinterpret_cast<unsigned *>(memory + addr) = (SData->read()).to_uint();
        dir = 'R';
        os << cycle << " I: R data: " << (SData->read()).to_uint() << endl;
        os << cycle << " I: R data info: " << (SDataInfo->read()).to_uint() << endl;
      } else {
        dir = 'W';
      }
      os << cycle << " I: " << dir << " resp received @ " << addr << endl;
      os << cycle << " I: resp info: " << (SRespInfo->read()).to_uint() << endl;
    }
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, next_req_cycle, resps_rxd;
  bool req_active, reset[2];
  queue<int> rd_addr;
  ofstream os;
};


class ocp_tl1_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl1_initiator);
public:
  ocp_tl1_initiator(sc_module_name n):
    sc_module(n),
    ocp("ocp", this, &ocp_tl1_initiator::ocp_timing_update),
    cycle(0),
    reqs_sent(0),
    next_req_cycle(1),
    resps_rxd(0),
    req_active(false),
    was_not_in_reset(true),
    scmdaccept(false),
    sresp(0),
    local_sflag(0),
    local_serror(0),
    local_sinterrupt(0),
    os(sc_module::name()),
    srespT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_bw(this, &ocp_tl1_initiator::nb_transport_bw);

    fill_initiator_memory(memory);
    r_in = reset[1] = reset[0] = false;
  }

  sc_in<bool> Clk, EnableClk;
  ocp_master_socket_tl1<32> ocp;

private:
  void on_Clk_rising() {
    if(!EnableClk) return;

    if((reqs_sent * 4 + 3 >= MEM_SIZE) && (resps_rxd == reqs_sent)) {
      stop_sim();
      return;
    }

    cycle++;

    // reset the OCP
    if(cycle == 9821) {
      os << cycle << " I: sent OCP reset with  " << reqs_sent - resps_rxd << " outstanding" << endl;
      tlm_generic_payload *pl = ocp.get_reset_transaction();
      tlm_phase ph = BEGIN_RESET;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_fw(*pl, ph, t);
    }
    if(cycle == 10032) {
      os << cycle << " I: retracted OCP reset" << endl;
      ocp.reset();
      tlm_generic_payload *pl = ocp.get_reset_transaction();
      tlm_phase ph = END_RESET;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_fw(*pl, ph, t);
      while(rd_addr.size() > 0) {
        rd_addr.pop();
        ocp.release_transaction(txns_used.front());
        txns_used.pop();
      }
      sresp = 0;
      reqs_sent = 0;
      resps_rxd = 0;
      req_active = false;
    }
    if((cycle >= 9821) && (cycle <= 10032)) return;

    // resynchronise Slave OCP reset
    // can fall (enter reset) asynchronously but rise (leave reset) is clock-
    // aligned and must be detected immediately.
    bool in_r = reset[1] && r_in;
    reset[1] = reset[0];
    reset[0] = r_in;
    if(in_r) {
      os << cycle << " I: in reset" << endl;
      // reset all state
      sresp = 0;
      if(was_not_in_reset) ocp.reset();
      was_not_in_reset = false;
      reqs_sent = 0;
      resps_rxd = 0;
      req_active = false;
      while(rd_addr.size() > 0) {
        rd_addr.pop();
        ocp.release_transaction(txns_used.front());
        txns_used.pop();
      }
      return;
    }
    was_not_in_reset = true;

    // flags
    if((cycle % 300) == 1) {
      tlm_generic_payload *pl;
      tlm_phase ph;
      sc_time t;
      pl = ocp.get_flag_transaction();
      mflag::get(*pl) = (cycle /300) % 4;
      ph = MFLAG_CHANGE;
      t = SC_ZERO_TIME;
      ocp->nb_transport_fw(*pl, ph, t);
      pl = ocp.get_error_transaction();
      if((cycle / 300) & 1) ph = BEGIN_ERROR; else ph = END_ERROR;
      t = SC_ZERO_TIME;
      ocp->nb_transport_fw(*pl, ph, t);
    }
    if((cycle % 300) == 2) {
      os << cycle << " I: SFlag " << local_sflag << endl;
      os << cycle << " I: SError " << local_serror << endl;
      os << cycle << " I: SInterrupt " << local_sinterrupt << endl;
    }

    // request side
    if(req_active && scmdaccept)
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if(scmdaccept || !req_active) {
      if((cycle >= next_req_cycle) && (reqs_sent * 4 + 3 < MEM_SIZE)) {
        tlm_generic_payload *pl = ocp.get_transaction();
        leak_test(*pl);
        pl->set_data_length(4);
        pl->set_streaming_width(4);
        pl->set_byte_enable_ptr(0);
        pl->set_response_status(TLM_INCOMPLETE_RESPONSE);
        // incrementing word addresses
        unsigned addr = reqs_sent * 4;
        pl->set_address(addr);
        pl->set_data_ptr((unsigned char*)memory + addr);
        // mix up WR and RD
        char dir;
        if(((cycle >> 4) * (~cycle >> 8) * cycle) & 8) {
          pl->set_write();
          ocp.validate_extension<posted>(*pl);
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          os << cycle << " I: W data: " <<  d << endl;
          data_info::get(*pl) = (cycle & 4095);
          rd_addr.push(-1);
          dir = 'W';
        } else {
          pl->set_read();
          rd_addr.push(addr);
          dir = 'R';
        }
        req_info::get(*pl) = (reqs_sent & 1023);
        ocp.validate_extension<address_space>(*pl);
        address_space *as;
        ocp.get_extension<address_space>(as, *pl);
        as->value = reqs_sent & 3;
        ocp.validate_extension<conn_id>(*pl);
        conn_id *ci;
        ocp.get_extension<conn_id>(ci, *pl);
        ci->value = (reqs_sent >> 2) & 31;
        txns_used.push(pl);  // for reset
        tlm_phase ph = BEGIN_REQ;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*pl, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_REQ);
          scmdaccept = true;
        } else {
          scmdaccept = false;
        }
        reqs_sent++;
        req_active = true;
        os << cycle << " I: " << dir << " req sent @ " << addr << endl;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        req_active = false;
      }
    }

    // response side
    if(sresp != 0) {
      resps_rxd++;
      int addr = rd_addr.front();
      rd_addr.pop();
      txns_used.pop();
      char dir;
      if(addr >= 0) {
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        dir = 'R';
        os << cycle << " I: R data: " << d << endl;
        os << cycle << " I: R data info: " << data_info::get(*sresp) << endl;
      } else {
        dir = 'W';
      }
      os << cycle << " I: " << dir << " resp received @ " << addr << endl;
      os << cycle << " I: resp info: " << resp_info::get(*sresp) << endl;
      ocp.release_transaction(sresp);
      sresp = 0;
    }
  }

  tlm_sync_enum nb_transport_bw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == BEGIN_INTERRUPT) {
      local_sinterrupt = true;
      return TLM_ACCEPTED;
    }
    if(ph == END_INTERRUPT) {
      local_sinterrupt = false;
      return TLM_ACCEPTED;
    }
    if(ph == SFLAG_CHANGE) {
      local_sflag = sflag::get(pl);
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_ERROR) {
      local_serror = true;
      return TLM_ACCEPTED;
    }
    if(ph == END_ERROR) {
      local_serror = false;
      return TLM_ACCEPTED;
    }
    if(ph == END_RESET) {
      r_in = false;
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_RESET) {
      r_in = true;
      return TLM_ACCEPTED;
    }
    if(ph == END_REQ) {
      scmdaccept = true;
      return TLM_ACCEPTED;
    }
    sc_assert(ph == BEGIN_RESP);
    sresp = &pl;
    ph = END_RESP;
    return TLM_UPDATED;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, next_req_cycle, resps_rxd;
  bool req_active, was_not_in_reset;
  queue<int> rd_addr;
  queue<tlm_generic_payload *> txns_used;
  bool scmdaccept;
  bool r_in, reset[2];
  tlm_generic_payload *sresp;
  unsigned local_sflag;
  bool local_serror, local_sinterrupt;
  ofstream os;
  sc_time srespT;

  // not sure if this is redundant when there is no resp-accept in the OCP
  // configuration.  It still exists in the TLM2
  void ocp_timing_update(ocp_tl1_slave_timing times) {
    if(srespT != times.ResponseGrpStartTime) {
      srespT = times.ResponseGrpStartTime;
      set_master_timing();
    }
  }

  void set_master_timing() {
    ocp_tl1_master_timing mytimes;
    mytimes.MRespAcceptStartTime = srespT + sc_get_time_resolution();
    ocp.set_master_timing(mytimes);
  }
};


class ocp_tl0_target: public sc_module {
SC_HAS_PROCESS(ocp_tl0_target);
public:
  ocp_tl0_target(sc_module_name n):
    sc_module(n),
    cycle(0),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    SResp.initialize(sc_bv<2>(0));
    SCmdAccept.initialize(sc_bv<1>(false));
    SReset_n.initialize(sc_bv<1>(true));

    for(unsigned i = 0; i < 4; i++) fill_target_memory(memory[i], i);
    reset[1] = reset[0] = false;
  }

  sc_in<bool> Clk, EnableClk;
  sc_in<sc_bv<32> > MAddr;
  sc_in<sc_bv<3> > MCmd;
  sc_in<sc_bv<32> > MData;
  sc_out<sc_bv<32> > SData;
  sc_out<sc_bv<1> > SCmdAccept;
  sc_out<sc_bv<2> > SResp;
  sc_in<sc_bv<1> > MReset_n;

  sc_in<sc_bv<2> > MAddrSpace;
  sc_in<sc_bv<12> > MDataInfo;
  sc_in<sc_bv<10> > MReqInfo;
  sc_out<sc_bv<12> > SDataInfo;
  sc_out<sc_bv<1> > SRespInfo;
  sc_in<sc_bv<5> > MConnID;
  sc_in<sc_bv<1> > MError;
  sc_in<sc_bv<2> > MFlag;
  sc_out<sc_bv<1> > SError;
  sc_out<sc_bv<3> > SFlag;
  sc_out<sc_bv<1> > SInterrupt;
  sc_out<sc_bv<1> > SReset_n;

private:
  void on_Clk_rising() {
    if(!EnableClk) return;

    cycle++;

    // reset the OCP - start cycle selected when several outstanding responses
    if(cycle == 19602) {
      os << cycle << " T: sent OCP reset with  " << rd_addr.size() << " outstanding" << endl;
      SReset_n = sc_bv<1>(false);
      while(resp_cycle.size() > 0) {
        resp_cycle.pop();
        rd_addr.pop();
        addr_space.pop();
      }
      SResp = sc_bv<2>(0);
      SCmdAccept = sc_bv<1>(false);
    }
    if(cycle == 20032) {
      os << cycle << " T: retracted OCP reset" << endl;
      SReset_n = sc_bv<1>(true);
    }
    if((cycle >= 19602) && (cycle <= 20032)) return;

    // resynchronise Master OCP reset
    // can fall (enter reset) asynchronously but rise (leave reset) is clock-
    // aligned and must be detected immediately.
    bool r_in = (MReset_n->read() == sc_bv<1>(false));
    bool in_r = reset[1] && r_in;
    reset[1] = reset[0];
    reset[0] = r_in;
    if(in_r) {
      os << cycle << " T: in reset" << endl;
      // reset all state
      while(resp_cycle.size() > 0) {
        resp_cycle.pop();
        rd_addr.pop();
        addr_space.pop();
      }
      SResp = sc_bv<2>(0);
      return;
    }

    // flag, error and interrupt
    if((cycle % 300) == 1) {
      SFlag = (cycle / 300);
      SError = (cycle / 300) & 1;
      SInterrupt = !((cycle / 300) & 1);
    }
    if((cycle % 300) == 2) {
      os << cycle << " T: MFlag " << (MFlag->read()).to_uint() << endl;
      os << cycle << " T: MError " << (MError->read()).to_uint() << endl;
    }

    // response side
    if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
      resp_cycle.pop();
      SResp = 1;
      int addr = rd_addr.front();
      unsigned as = addr_space.front();
      rd_addr.pop();
      addr_space.pop();
      char dir;
      if(addr < 0) {
        dir = 'W';
      } else {
        dir = 'R';
        unsigned d = *reinterpret_cast<unsigned *>(memory[as] + addr);
        SData = d;
        os << cycle << " T: R data: " <<  d << endl;
        SDataInfo = sc_bv<12>(cycle & 4095);
      }
      os << cycle << " T: " << dir << " resp sent @ " << addr << ", " << as << endl;
      SRespInfo = sc_bv<1>(addr & 1);
    } else {
      SResp = 0;
    }

    // request side
    // accept in cycles 3,6,9,12,15,... and impose 16 cycle minimum latency
    if(SCmdAccept->read() == sc_bv<1>(true)) {
      unsigned d;
      switch((MCmd->read()).to_uint()) {
      case 0:
        break;
      case 1:
        rd_addr.push(-1);
        addr_space.push((MAddrSpace->read()).to_uint());
        d = (MData->read()).to_uint();
        *reinterpret_cast<unsigned *>(memory[(MAddrSpace->read()).to_uint()] + (MAddr->read()).to_uint()) = d;
        os << cycle << " T: W data: " << d << endl;
        os << cycle << " T: W req received @ " << (MAddr->read()).to_uint() << ", " << (MAddrSpace->read()).to_uint() << endl;
        os << cycle << " T: W data info: " << (MDataInfo->read()).to_uint() << endl;
        os << cycle << " T: conn-ID: " << (MConnID->read()).to_uint() << endl;
        os << cycle << " T: req info: " << (MReqInfo->read()).to_uint() << endl;
        resp_cycle.push(cycle + 16);
        break;
      case 2:
        rd_addr.push((MAddr->read()).to_uint());
        addr_space.push((MAddrSpace->read()).to_uint());
        os << cycle << " T: R req received @ " << (MAddr->read()).to_uint() << ", " << (MAddrSpace->read()).to_uint() << endl;
        os << cycle << " T: conn-ID: " << (MConnID->read()).to_uint() << endl;
        os << cycle << " T: req info: " << (MReqInfo->read()).to_uint() << endl;
        resp_cycle.push(cycle + 16);
        break;
      }
    }
    SCmdAccept = sc_bv<1>((cycle % 3) == 0);
  }

  unsigned cycle;
  char memory[MEM_SIZE][4];
  queue<int> rd_addr;
  queue<unsigned> addr_space, resp_cycle;
  bool reset[2];
  ofstream os;
};


class ocp_tl1_target: public sc_module {
SC_HAS_PROCESS(ocp_tl1_target);
public:
  ocp_tl1_target(sc_module_name n):
    sc_module(n),
    ocp("ocp", this, &ocp_tl1_target::ocp_timing_update),
    cycle(0),
    scmdaccept(false),
    mcmd(0),
    local_mflag(0),
    local_merror(0),
    os(sc_module::name()),
    cmdT(SC_ZERO_TIME),
    dataT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_fw(this, &ocp_tl1_target::nb_transport_fw);

    for(unsigned i = 0; i < 4; i++) fill_target_memory(memory[i], i);
    r_in = reset[1] = reset[0] = false;
  }

  sc_in<bool> Clk, EnableClk;
  ocp_slave_socket_tl1<32> ocp;

private:
  void on_Clk_rising() {
    if(!EnableClk) return;

    cycle++;

    // reset the OCP - start cycle selected when several outstanding responses
    if(cycle == 19602) {
      os << cycle << " T: sent OCP reset with  " << rd_addr.size() << " outstanding" << endl;
      tlm_generic_payload *pl = ocp.get_reset_transaction();
      tlm_phase ph = BEGIN_RESET;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
    }
    if(cycle == 20032) {
      os << cycle << " T: retracted OCP reset" << endl;
      ocp.reset();
      tlm_generic_payload *pl = ocp.get_reset_transaction();
      tlm_phase ph = END_RESET;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
      while(rd_addr.size() > 0) {
        rd_addr.front()->release();
        rd_addr.pop();
      }
      while(resp_cycle.size() > 0) resp_cycle.pop();
      mcmd = 0;
      scmdaccept = false;
    }
    if((cycle >= 19602) && (cycle <= 20032)) return;

    // resynchronise Master OCP reset
    // can fall (enter reset) asynchronously but rise (leave reset) is clock-
    // aligned and must be detected immediately.
    bool in_r = reset[1] && r_in;
    reset[1] = reset[0];
    reset[0] = r_in;
    if(in_r) {
      os << cycle << " T: in reset" << endl;
      return;
    }

    // flag, error, interrupt
    if((cycle % 300) == 1) {
      tlm_generic_payload *pl;
      tlm_phase ph;
      sc_time t;
      pl = ocp.get_flag_transaction();
      sflag::get(*pl) = (cycle /300) % 8;
      ph = SFLAG_CHANGE;
      t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
      pl = ocp.get_error_transaction();
      if((cycle / 300) & 1) ph = BEGIN_ERROR; else ph = END_ERROR;
      t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
      pl = ocp.get_interrupt_transaction();
      if((cycle / 300) & 1) ph = END_INTERRUPT; else ph = BEGIN_INTERRUPT;
      t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
    }
    if((cycle % 300) == 2) {
      os << cycle << " T: MFlag " << local_mflag << endl;
      os << cycle << " T: MError " << local_merror << endl;
    }

    // response side
    if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
      resp_cycle.pop();
      tlm_generic_payload *pl = rd_addr.front();
      rd_addr.pop();
      pl->set_response_status(TLM_OK_RESPONSE);
      int addr = pl->get_address();
      address_space *as;
      ocp.get_extension<address_space>(as, *pl);
      char dir;
      unsigned *local_addr = reinterpret_cast<unsigned *>(memory[as->value] + addr);
      unsigned *init_addr = reinterpret_cast<unsigned *>(pl->get_data_ptr());
      if(pl->is_write()) {
        dir = 'W';
        addr = -1;
      } else {
        dir = 'R';
        *init_addr = *local_addr;
        os << cycle << " T: R data: " << *local_addr << endl;
        data_info::get(*pl) = (cycle & 4095);
      }
      resp_info::get(*pl) = (addr & 1);
      tlm_phase ph = BEGIN_RESP;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
      pl->release();
      os << cycle << " T: " << dir << " resp sent @ " << addr << ", " << as->value << endl;
    }

    // request side
    // accept in cycles 3,6,9,12,15,... and impose 16 cycle minimum latency
    if(scmdaccept) {
      if(mcmd != 0) {
        tlm_generic_payload *pl = mcmd;
        int addr = pl->get_address();
        address_space *as;
        sc_assert(ocp.get_extension<address_space>(as, *pl));
        conn_id *ci;
        sc_assert(ocp.get_extension<conn_id>(ci, *pl));
        if(pl->is_write()) {
          unsigned *local_addr = reinterpret_cast<unsigned *>(memory[as->value] + addr);
          unsigned *init_addr = reinterpret_cast<unsigned *>(pl->get_data_ptr());
          *local_addr = *init_addr;
          os << cycle << " T: W data: " << *local_addr << endl;
          os << cycle << " T: W req received @ " << addr << ", " << as->value << endl;
          os << cycle << " T: W data info: " << data_info::get(*pl) << endl;
          os << cycle << " T: conn-ID: " << ci->value << endl;
          os << cycle << " T: req info: " << req_info::get(*pl) << endl;
        } else {
          os << cycle << " T: R req received @ " << addr <<  ", " << as->value << endl;
          os << cycle << " T: conn-ID: " << ci->value << endl;
          os << cycle << " T: req info: " << req_info::get(*pl) << endl;
        }
        resp_cycle.push(cycle + 16);
        mcmd = 0;
      }
    }
    scmdaccept = ((cycle % 3) == 0);
    if(scmdaccept && (mcmd != 0)) {
      tlm_phase ph = END_REQ;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mcmd, ph, t);
    }
  }

  tlm_sync_enum nb_transport_fw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == MFLAG_CHANGE) {
      local_mflag = mflag::get(pl);
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_ERROR) {
      local_merror = true;
      return TLM_ACCEPTED;
    }
    if(ph == END_ERROR) {
      local_merror = false;
      return TLM_ACCEPTED;
    }
    if(ph == END_RESET) {
      r_in = false;
      ocp.reset();
      while(rd_addr.size() > 0) {
        rd_addr.front()->release();
        rd_addr.pop();
      }
      while(resp_cycle.size() > 0) resp_cycle.pop();
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_RESET) {
      r_in = true;
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_REQ) {
      mcmd = &pl;
      rd_addr.push(&pl);
      pl.acquire();
      leak_test(pl);
      if(scmdaccept) {
        ph = END_REQ;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    sc_assert(ph == END_RESP);
    return TLM_ACCEPTED;
  }

  unsigned cycle;
  bool scmdaccept;
  tlm_generic_payload *mcmd;
  char memory[MEM_SIZE][4];
  queue<tlm_generic_payload *> rd_addr;
  queue<unsigned> resp_cycle;
  bool r_in, reset[2];
  unsigned local_mflag;
  bool local_merror;
  ofstream os;
  sc_time cmdT, dataT;

  void ocp_timing_update(ocp_tl1_master_timing times) {
    if((cmdT != times.RequestGrpStartTime) ||
       (dataT != times.DataHSGrpStartTime)) {
      cmdT = times.RequestGrpStartTime;
      dataT = times.DataHSGrpStartTime;
      set_slave_timing();
    }
  }

  void set_slave_timing() {
    ocp_tl1_slave_timing mytimes;
    mytimes.SCmdAcceptStartTime = cmdT + sc_get_time_resolution();
    mytimes.SDataAcceptStartTime = dataT + sc_get_time_resolution();
    ocp.set_slave_timing(mytimes);
  }
};


class testbench00: public sc_module {
public:
  testbench00(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    EnableClk("EnableClk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    initiator.EnableClk(EnableClk);
    target.EnableClk(EnableClk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);

    initiator.MAddrSpace(MAddrSpace);
    initiator.MDataInfo(MDataInfo);
    initiator.MReqInfo(MReqInfo);
    initiator.SDataInfo(SDataInfo);
    initiator.SRespInfo(SRespInfo);
    initiator.MConnID(MConnID);
    initiator.MError(MError);
    initiator.MFlag(MFlag);
    initiator.SError(SError);
    initiator.SFlag(SFlag);
    initiator.SInterrupt(SInterrupt);
    initiator.SReset_n(SReset_n);
    target.MAddrSpace(MAddrSpace);
    target.MDataInfo(MDataInfo);
    target.MReqInfo(MReqInfo);
    target.SDataInfo(SDataInfo);
    target.SRespInfo(SRespInfo);
    target.MConnID(MConnID);
    target.MError(MError);
    target.MFlag(MFlag);
    target.SError(SError);
    target.SFlag(SFlag);
    target.SInterrupt(SInterrupt);
    target.SReset_n(SReset_n);
  }

  sc_in<bool> Clk, EnableClk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;

  sc_signal<sc_bv<32> > MAddr;
  sc_signal<sc_bv<3> > MCmd;
  sc_signal<sc_bv<32> > MData;
  sc_signal<sc_bv<32> > SData;
  sc_signal<sc_bv<1> > SCmdAccept;
  sc_signal<sc_bv<2> > SResp;
  sc_signal<sc_bv<1> > MReset_n;

  sc_signal<sc_bv<2> > MAddrSpace;
  sc_signal<sc_bv<12> > MDataInfo;
  sc_signal<sc_bv<10> > MReqInfo;
  sc_signal<sc_bv<12> > SDataInfo;
  sc_signal<sc_bv<1> > SRespInfo;
  sc_signal<sc_bv<5> > MConnID;
  sc_signal<sc_bv<1> > MError;
  sc_signal<sc_bv<2> > MFlag;
  sc_signal<sc_bv<1> > SError;
  sc_signal<sc_bv<3> > SFlag;
  sc_signal<sc_bv<1> > SInterrupt;
  sc_signal<sc_bv<1> > SReset_n;
};


class testbench01: public sc_module {
public:
  testbench01(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    EnableClk("EnableClk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);
    initiator.EnableClk(EnableClk);
    target.EnableClk(EnableClk);
    adapter.EnableClk(EnableClk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);

    initiator.MAddrSpace(MAddrSpace);
    initiator.MDataInfo(MDataInfo);
    initiator.MReqInfo(MReqInfo);
    initiator.SDataInfo(SDataInfo);
    initiator.SRespInfo(SRespInfo);
    initiator.MConnID(MConnID);
    initiator.MError(MError);
    initiator.MFlag(MFlag);
    initiator.SError(SError);
    initiator.SFlag(SFlag);
    initiator.SInterrupt(SInterrupt);
    initiator.SReset_n(SReset_n);
    adapter.MAddrSpace(MAddrSpace);
    adapter.MDataInfo(MDataInfo);
    adapter.MReqInfo(MReqInfo);
    adapter.SDataInfo(SDataInfo);
    adapter.SRespInfo(SRespInfo);
    adapter.MConnID(MConnID);
    adapter.MError(MError);
    adapter.MFlag(MFlag);
    adapter.SError(SError);
    adapter.SFlag(SFlag);
    adapter.SInterrupt(SInterrupt);
    adapter.SReset_n(SReset_n);

    // TL1 interface
    adapter.ocpTL1(target.ocp);
  }

  sc_in<bool> Clk, EnableClk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl1_target target;
  tl0_initiator_to_tl1_target<32, ADAPTER01_TL0_TYPES> adapter;

  sc_signal<sc_bv<32> > MAddr;
  sc_signal<sc_bv<3> > MCmd;
  sc_signal<sc_bv<32> > MData;
  sc_signal<sc_bv<32> > SData;
  sc_signal<sc_bv<1> > SCmdAccept;
  sc_signal<sc_bv<2> > SResp;
  sc_signal<sc_bv<1> > MReset_n;

  sc_signal<sc_bv<2> > MAddrSpace;
  sc_signal<sc_bv<12> > MDataInfo;
  sc_signal<sc_bv<10> > MReqInfo;
  sc_signal<sc_bv<12> > SDataInfo;
  sc_signal<sc_bv<1> > SRespInfo;
  sc_signal<sc_bv<5> > MConnID;
  sc_signal<sc_bv<1> > MError;
  sc_signal<sc_bv<2> > MFlag;
  sc_signal<sc_bv<1> > SError;
  sc_signal<sc_bv<3> > SFlag;
  sc_signal<sc_bv<1> > SInterrupt;
  sc_signal<sc_bv<1> > SReset_n;
};


class testbench10: public sc_module {
public:
  testbench10(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    EnableClk("EnableClk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);
    initiator.EnableClk(EnableClk);
    target.EnableClk(EnableClk);
    adapter.EnableClk(EnableClk);

    // TL1 interface
    initiator.ocp(adapter.ocpTL1);

    // TL0 interface
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);

    adapter.MAddrSpace(MAddrSpace);
    adapter.MDataInfo(MDataInfo);
    adapter.MReqInfo(MReqInfo);
    adapter.SDataInfo(SDataInfo);
    adapter.SRespInfo(SRespInfo);
    adapter.MConnID(MConnID);
    adapter.MError(MError);
    adapter.MFlag(MFlag);
    adapter.SError(SError);
    adapter.SFlag(SFlag);
    adapter.SInterrupt(SInterrupt);
    adapter.SReset_n(SReset_n);
    target.MAddrSpace(MAddrSpace);
    target.MDataInfo(MDataInfo);
    target.MReqInfo(MReqInfo);
    target.SDataInfo(SDataInfo);
    target.SRespInfo(SRespInfo);
    target.MConnID(MConnID);
    target.MError(MError);
    target.MFlag(MFlag);
    target.SError(SError);
    target.SFlag(SFlag);
    target.SInterrupt(SInterrupt);
    target.SReset_n(SReset_n);
  }

  sc_in<bool> Clk, EnableClk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl0_target target;
  tl1_initiator_to_tl0_target<32, ADAPTER10_TL0_TYPES> adapter;

  sc_signal<sc_bv<32> > MAddr;
  sc_signal<sc_bv<3> > MCmd;
  sc_signal<sc_bv<32> > MData;
  sc_signal<sc_bv<32> > SData;
  sc_signal<sc_bv<1> > SCmdAccept;
  sc_signal<sc_bv<2> > SResp;
  sc_signal<sc_bv<1> > MReset_n;

  sc_signal<sc_bv<2> > MAddrSpace;
  sc_signal<sc_bv<12> > MDataInfo;
  sc_signal<sc_bv<10> > MReqInfo;
  sc_signal<sc_bv<12> > SDataInfo;
  sc_signal<sc_bv<1> > SRespInfo;
  sc_signal<sc_bv<5> > MConnID;
  sc_signal<sc_bv<1> > MError;
  sc_signal<sc_bv<2> > MFlag;
  sc_signal<sc_bv<1> > SError;
  sc_signal<sc_bv<3> > SFlag;
  sc_signal<sc_bv<1> > SInterrupt;
  sc_signal<sc_bv<1> > SReset_n;
};


class testbench11: public sc_module {
public:
  testbench11(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    EnableClk("EnableClk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    initiator.EnableClk(EnableClk);
    target.EnableClk(EnableClk);

    // TL1 interface
    initiator.ocp(target.ocp);
  }

  sc_in<bool> Clk, EnableClk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;
};


class testbench101: public sc_module {
public:
  testbench101(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    EnableClk("EnableClk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter10("adapter10", &t_print),
    adapter01("adapter01", &t_print)
  {
    // OCP configuration
    adapter10.set_ocp_config(gen_ocp_config());
    adapter01.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter10.Clk(Clk);
    adapter01.Clk(Clk);
    initiator.EnableClk(EnableClk);
    target.EnableClk(EnableClk);
    adapter10.EnableClk(EnableClk);
    adapter01.EnableClk(EnableClk);

    // TL1 interface
    initiator.ocp(adapter10.ocpTL1);
    adapter01.ocpTL1(target.ocp);

    // TL0 interface
    adapter10.MAddr(MAddr);
    adapter10.MCmd(MCmd);
    adapter10.MData(MData);
    adapter10.SData(SData);
    adapter10.SCmdAccept(SCmdAccept);
    adapter10.SResp(SResp);
    adapter10.MReset_n(MReset_n);
    adapter01.MAddr(MAddr);
    adapter01.MCmd(MCmd);
    adapter01.MData(MData);
    adapter01.SData(SData);
    adapter01.SCmdAccept(SCmdAccept);
    adapter01.SResp(SResp);
    adapter01.MReset_n(MReset_n);

    adapter10.MAddrSpace(MAddrSpace);
    adapter10.MDataInfo(MDataInfo);
    adapter10.MReqInfo(MReqInfo);
    adapter10.SDataInfo(SDataInfo);
    adapter10.SRespInfo(SRespInfo);
    adapter10.MConnID(MConnID);
    adapter10.MError(MError);
    adapter10.MFlag(MFlag);
    adapter10.SError(SError);
    adapter10.SFlag(SFlag);
    adapter10.SInterrupt(SInterrupt);
    adapter10.SReset_n(SReset_n);
    adapter01.MAddrSpace(MAddrSpace);
    adapter01.MDataInfo(MDataInfo);
    adapter01.MReqInfo(MReqInfo);
    adapter01.SDataInfo(SDataInfo);
    adapter01.SRespInfo(SRespInfo);
    adapter01.MConnID(MConnID);
    adapter01.MError(MError);
    adapter01.MFlag(MFlag);
    adapter01.SError(SError);
    adapter01.SFlag(SFlag);
    adapter01.SInterrupt(SInterrupt);
    adapter01.SReset_n(SReset_n);

    // timing:  TL1 initiator and target produce transport calls at +0 ps
    //   but they aren't seen by adapters until +1 ps because of synch protection
    //   and only at this time are TL0 signals written
    adapter01.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter10 writes MCmd always during TLM-transport-from-initiator at +1 ps
      sc_time(3.0, SC_PS)
      // same for MDataValid but sampling is delayed to be after MCmd sampling,
      // which would be automatic but shown here for clarity
    );
    adapter10.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter01 writes SResp always during TLM-transport-from-target at +1 ps
      sc_time(5.0, SC_PS),
      // TL1 target may signal SCmdAccept during its clocked process with an
      // explicit transport call.  This would be seen at +1 ps by adapter01 and
      // the TL0 SCmdAccept signal would be stable for sampling at +2 ps.  But
      // TL1 target might return SCmdAccept to a fw transport call, which it
      // would see at the MCmd sampling time of adapter01 plus 1, ie 3 ps.  The
      // socket PEQ will convert this into a separate backward transport call
      // seen by adapter01 at 4 ps.  Thus SCmdAccept should be sampled at 5 ps.
      sc_time(6.0, SC_PS)
      // data trails command by 1 ps
    );
  }

  sc_in<bool> Clk, EnableClk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;

  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl1_initiator_to_tl0_target<32, ADAPTER10_TL0_TYPES> adapter10;
  tl0_initiator_to_tl1_target<32, ADAPTER01_TL0_TYPES> adapter01;

  sc_signal<sc_bv<32> > MAddr;
  sc_signal<sc_bv<3> > MCmd;
  sc_signal<sc_bv<32> > MData;
  sc_signal<sc_bv<32> > SData;
  sc_signal<sc_bv<1> > SCmdAccept;
  sc_signal<sc_bv<2> > SResp;
  sc_signal<sc_bv<1> > MReset_n;

  sc_signal<sc_bv<2> > MAddrSpace;
  sc_signal<sc_bv<12> > MDataInfo;
  sc_signal<sc_bv<10> > MReqInfo;
  sc_signal<sc_bv<12> > SDataInfo;
  sc_signal<sc_bv<1> > SRespInfo;
  sc_signal<sc_bv<5> > MConnID;
  sc_signal<sc_bv<1> > MError;
  sc_signal<sc_bv<2> > MFlag;
  sc_signal<sc_bv<1> > SError;
  sc_signal<sc_bv<3> > SFlag;
  sc_signal<sc_bv<1> > SInterrupt;
  sc_signal<sc_bv<1> > SReset_n;
};


class testbench010: public sc_module {
public:
  testbench010(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    EnableClk("EnableClk"),
    initiator("initiator"),
    target("target"),
    adapter01("adapter01"),
    adapter10("adapter10")
  {
    // OCP configuration
    adapter01.set_ocp_config(gen_ocp_config());
    adapter10.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter01.Clk(Clk);
    adapter10.Clk(Clk);
    initiator.EnableClk(EnableClk);
    target.EnableClk(EnableClk);
    adapter01.EnableClk(EnableClk);
    adapter10.EnableClk(EnableClk);

    // TL1 interface
    adapter01.ocpTL1(adapter10.ocpTL1);

    // TL0 interfaces
    initiator.MAddr(MAddr[0]);
    initiator.MCmd(MCmd[0]);
    initiator.MData(MData[0]);
    initiator.SData(SData[0]);
    initiator.SCmdAccept(SCmdAccept[0]);
    initiator.SResp(SResp[0]);
    initiator.MReset_n(MReset_n[0]);
    adapter01.MAddr(MAddr[0]);
    adapter01.MCmd(MCmd[0]);
    adapter01.MData(MData[0]);
    adapter01.SData(SData[0]);
    adapter01.SCmdAccept(SCmdAccept[0]);
    adapter01.SResp(SResp[0]);
    adapter01.MReset_n(MReset_n[0]);

    adapter10.MAddr(MAddr[1]);
    adapter10.MCmd(MCmd[1]);
    adapter10.MData(MData[1]);
    adapter10.SData(SData[1]);
    adapter10.SCmdAccept(SCmdAccept[1]);
    adapter10.SResp(SResp[1]);
    adapter10.MReset_n(MReset_n[1]);
    target.MAddr(MAddr[1]);
    target.MCmd(MCmd[1]);
    target.MData(MData[1]);
    target.SData(SData[1]);
    target.SCmdAccept(SCmdAccept[1]);
    target.SResp(SResp[1]);
    target.MReset_n(MReset_n[1]);

    initiator.MAddrSpace(MAddrSpace[0]);
    initiator.MDataInfo(MDataInfo[0]);
    initiator.MReqInfo(MReqInfo[0]);
    initiator.SDataInfo(SDataInfo[0]);
    initiator.SRespInfo(SRespInfo[0]);
    initiator.MConnID(MConnID[0]);
    initiator.MError(MError[0]);
    initiator.MFlag(MFlag[0]);
    initiator.SError(SError[0]);
    initiator.SFlag(SFlag[0]);
    initiator.SInterrupt(SInterrupt[0]);
    initiator.SReset_n(SReset_n[0]);
    adapter01.MAddrSpace(MAddrSpace[0]);
    adapter01.MDataInfo(MDataInfo[0]);
    adapter01.MReqInfo(MReqInfo[0]);
    adapter01.SDataInfo(SDataInfo[0]);
    adapter01.SRespInfo(SRespInfo[0]);
    adapter01.MConnID(MConnID[0]);
    adapter01.MError(MError[0]);
    adapter01.MFlag(MFlag[0]);
    adapter01.SError(SError[0]);
    adapter01.SFlag(SFlag[0]);
    adapter01.SInterrupt(SInterrupt[0]);
    adapter01.SReset_n(SReset_n[0]);

    adapter10.MAddrSpace(MAddrSpace[1]);
    adapter10.MDataInfo(MDataInfo[1]);
    adapter10.MReqInfo(MReqInfo[1]);
    adapter10.SDataInfo(SDataInfo[1]);
    adapter10.SRespInfo(SRespInfo[1]);
    adapter10.MConnID(MConnID[1]);
    adapter10.MError(MError[1]);
    adapter10.MFlag(MFlag[1]);
    adapter10.SError(SError[1]);
    adapter10.SFlag(SFlag[1]);
    adapter10.SInterrupt(SInterrupt[1]);
    adapter10.SReset_n(SReset_n[1]);
    target.MAddrSpace(MAddrSpace[1]);
    target.MDataInfo(MDataInfo[1]);
    target.MReqInfo(MReqInfo[1]);
    target.SDataInfo(SDataInfo[1]);
    target.SRespInfo(SRespInfo[1]);
    target.MConnID(MConnID[1]);
    target.MError(MError[1]);
    target.MFlag(MFlag[1]);
    target.SError(SError[1]);
    target.SFlag(SFlag[1]);
    target.SInterrupt(SInterrupt[1]);
    target.SReset_n(SReset_n[1]);

    // timing: nothing needed as adapters' TL0 interfaces are
    //  directly connected to TL0-native initiator and target
  }

  sc_in<bool> Clk, EnableClk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;
  tl0_initiator_to_tl1_target<32, ADAPTER01_TL0_TYPES> adapter01;
  tl1_initiator_to_tl0_target<32, ADAPTER10_TL0_TYPES> adapter10;

  sc_signal<sc_bv<32> > MAddr[2];
  sc_signal<sc_bv<3> > MCmd[2];
  sc_signal<sc_bv<32> > MData[2];
  sc_signal<sc_bv<32> > SData[2];
  sc_signal<sc_bv<1> > SCmdAccept[2];
  sc_signal<sc_bv<2> > SResp[2];
  sc_signal<sc_bv<1> > MReset_n[2];

  sc_signal<sc_bv<2> > MAddrSpace[2];
  sc_signal<sc_bv<12> > MDataInfo[2];
  sc_signal<sc_bv<10> > MReqInfo[2];
  sc_signal<sc_bv<12> > SDataInfo[2];
  sc_signal<sc_bv<1> > SRespInfo[2];
  sc_signal<sc_bv<5> > MConnID[2];
  sc_signal<sc_bv<1> > MError[2];
  sc_signal<sc_bv<2> > MFlag[2];
  sc_signal<sc_bv<1> > SError[2];
  sc_signal<sc_bv<3> > SFlag[2];
  sc_signal<sc_bv<1> > SInterrupt[2];
  sc_signal<sc_bv<1> > SReset_n[2];
};


class testbench: public sc_module {
SC_HAS_PROCESS(testbench);
public:
  testbench(sc_module_name n, bool run_01 = true, bool run_10 = true):
    sc_module(n),
    tb00("00"),
    tb01("01"),
    tb10("10"),
    tb11("11"),
    tb101("101"),
    tb010("010"),
    Clk("Clk", sc_time(10.0, SC_NS))
  {
    SC_METHOD(toggle);
    sensitive << Clk.posedge_event();
    dont_initialize();
    count = 0;

    tb00.Clk(Clk);
    tb11.Clk(Clk);

    if(run_01) tb01.Clk(Clk);
    else tb01.Clk(tieoff);

    if(run_10) tb10.Clk(Clk);
    else tb10.Clk(tieoff);

    if(run_01 && run_10) {
      tb010.Clk(Clk);
      tb101.Clk(Clk);
    } else {
     tb101.Clk(tieoff);
      tb010.Clk(tieoff);
    }

    tb00.EnableClk(EnableClk);
    tb11.EnableClk(EnableClk);
    tb10.EnableClk(EnableClk);
    tb01.EnableClk(EnableClk);
    tb101.EnableClk(EnableClk);
    tb010.EnableClk(EnableClk);
  }
private:
  void toggle() {
    if(++count >= 5) count = 0;
    EnableClk = (count > 2);
  }

  testbench00 tb00;
  testbench01 tb01;
  testbench10 tb10;
  testbench11 tb11;
  testbench101 tb101;
  testbench010 tb010;
  sc_clock Clk;
  sc_signal<bool> tieoff, EnableClk;
  unsigned count;
};


#ifndef OMIT_SC_MAIN
int sc_main(int argc, char **argv) {
  sc_report_handler::set_actions(SC_ERROR, SC_DISPLAY | SC_ABORT);
  bool run_01 = ((argc < 2) || ('A' == *argv[1]) || ('0' == *argv[1]));
  bool run_10 = ((argc < 2) || ('A' == *argv[1]) || ('1' == *argv[1]));
  testbench theTestbench("testbench", run_01, run_10);
  sc_start();
  return 0;
}
#endif

