///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  Testbench for TL1/TL0 adapters: imprecise aligned MRMD with
//                 varied burst types
//
// TL1 and TL0 initiators and targets are created with identical behaviour for
// the different abstraction levels.
// Six simulations are run, 4 with adapters and 2 without, for 0, 1 and 2
//   cascaded adapters.
// All sets of results should be the same, allowing for a few ps of timing
// delay in some cases.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


#include "ocpip_adapters.h"
#include <iostream>
#include <queue>


using namespace std;
using namespace ocpip;
using namespace sc_core;
using namespace tlm;


// all OCPs in this testbench have the same configuration
ocp_parameters gen_ocp_config() {
  ocp_parameters p;
  // imprecise bursts INCR, UNKN, DFLT1, DFLT2
  p.burstprecise = true; // means the signal exists, not that bursts are precise
  p.burstlength = true;
  p.burstlength_wdth = 4;
  p.burstseq = true;
  p.burstseq_unkn_enable = true;
  p.burstseq_dflt1_enable = true;
  p.burstseq_dflt2_enable = true;
  p.byteen = true;
  // base config
  p.addr_wdth = 32;
  p.data_wdth = 32;
  p.mreset = true;
  return p;
}


// traits class for TL0 signal types
class MRMDBURST_TYPES: public tl1_tl0::POD_DEFAULT_TL0_TYPES<> {
public:
  typedef unsigned MBURSTLENGTH_T;
  typedef unsigned MBURSTSEQ_T;
  typedef bool MBURSTPRECISE_T;
  typedef unsigned MBYTEEN_T;
};


// simulation duration determined by these parameters - leave space for
// final burst to complete
#define MAX_ADDR 4000
#define MEM_SIZE 5000
#include "tb_common.h"


class ocp_tl0_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl0_initiator);
public:
  ocp_tl0_initiator(sc_module_name n):
    sc_module(n),
    cycle(0),
    reqs_sent(0),
    rd_reqs_sent(0),
    curr_addr(0),
    local_addr(0),
    curr_seq(INCR),
    next_req_cycle(1),
    resps_rxd(0),
    precise_burst_len(1),
    bytes_left_in_burst(0),
    req_active(false),
    curr_is_read(true),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    MReset_n.initialize(true);
    MCmd.initialize(0);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  sc_out<unsigned> MAddr;
  sc_out<unsigned> MCmd;
  sc_out<unsigned> MData;
  sc_out<unsigned> MBurstLength;
  sc_out<unsigned> MBurstSeq;
  sc_out<bool> MBurstPrecise;
  sc_out<unsigned> MByteEn;
  sc_in<unsigned> SData;
  sc_in<bool> SCmdAccept;
  sc_in<unsigned> SResp;
  sc_out<bool> MReset_n;

private:
  enum burstseq {INCR = 0, UNKN = 6, DFLT1 = 1, DFLT2 = 3};

  void on_Clk_rising() {
    if((curr_addr >= MAX_ADDR) && (resps_rxd == rd_reqs_sent)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && SCmdAccept)
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if(SCmdAccept || !req_active) {
      if((cycle >= next_req_cycle) && (curr_addr < MAX_ADDR)) {

        // new bus burst
        if(bytes_left_in_burst == 0) {
          switch(curr_seq) {
            case INCR: curr_seq = DFLT1; break;
            case DFLT1: curr_seq = DFLT2; break;
            case DFLT2: curr_seq = UNKN; break;
            case UNKN: curr_seq = INCR; break;
          }
          if(cycle & 5) curr_is_read = !curr_is_read ;
          curr_addr += cycle % 4;
          local_addr = curr_addr & 0xFFFFFFFC;
          bytes_left_in_burst = 1 + (cycle % 56);  // max burst length is 15 words
          if(curr_seq == INCR) {
            if(precise_burst_len == 0) precise_burst_len = 1 +
              (((curr_addr + bytes_left_in_burst - 1) & 0xFFFFFFFC) - (curr_addr & 0xFFFFFFFC)) / 4;
            else precise_burst_len = 0;
          }
        }
        unsigned end_addr = (curr_addr & 0xFFFFFFFC) + 4;
        unsigned bytes = end_addr - curr_addr;
        if(bytes > bytes_left_in_burst) {
          bytes = bytes_left_in_burst;
          end_addr = curr_addr + bytes;
        }

        MAddr = (curr_addr & 0xFFFFFFFC);
        unsigned be = (0xF >> (4 - bytes)) << (curr_addr & 3);
        MByteEn = be;
        MBurstSeq = curr_seq;

        switch(curr_seq) {
          // arbitrary address patterns for each burst type
          case INCR: curr_addr = end_addr; break;
          case DFLT1: curr_addr = end_addr + 8; break;
          case DFLT2: curr_addr = end_addr + 12; break;
          case UNKN: curr_addr = end_addr + 16; break;
        }

        bytes_left_in_burst -= bytes;
        if((precise_burst_len > 0) && (curr_seq == INCR)) {
          MBurstPrecise = true;
          MBurstLength = precise_burst_len;
        } else {
          MBurstPrecise = false;
          MBurstLength = (bytes_left_in_burst == 0 ? 1 : 2 + bytes_left_in_burst/4);
        }

        char dir;
        if(!curr_is_read) {
          MCmd = 1;
          unsigned d = *reinterpret_cast<unsigned *>(memory + local_addr);
          MData = d;
          os << cycle << " I: W data: " << d << ", BE = " << be << endl;
          dir = 'W';
        } else {
          MCmd = 2;
          rd_addr.push(local_addr);
          rd_be.push(be);
          rd_reqs_sent++;
          dir = 'R';
        }
        local_addr += 4;
        reqs_sent++;
        req_active = true;
        os << cycle << " I: " << dir << " req sent @ " << curr_addr << endl;
        os << cycle << " I: " << dir << " req length " << bytes_left_in_burst << endl;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        MCmd = 0;
        req_active = false;
      }
    }

    // response side
    if(SResp != 0) {
      resps_rxd++;
      int addr = rd_addr.front();
      unsigned be = rd_be.front();
      rd_addr.pop();
      rd_be.pop();
      char *local = memory + addr;
      unsigned sdata = SData;
      unsigned mask = 0;
      if(be & 1) {local[0] = sdata; mask |= 0xFF;}
      if(be & 2) {local[1] = sdata >> 8; mask |= 0xFF00;}
      if(be & 4) {local[2] = sdata >> 16; mask |= 0xFF0000;}
      if(be & 8) {local[3] = sdata >> 24; mask |= 0xFF000000;}
      os << cycle << " I: R data: " << (sdata & mask) << ", BE = " << be << endl;
      os << cycle << " I: R resp received @ " << addr << endl;
    }
    cycle++;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, rd_reqs_sent, curr_addr, local_addr, curr_seq;
  unsigned next_req_cycle, resps_rxd, precise_burst_len, bytes_left_in_burst;
  bool req_active, curr_is_read;
  queue<int> rd_addr, rd_be;
  ofstream os;
};


class ocp_tl1_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl1_initiator);
public:
  ocp_tl1_initiator(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_initiator::ocp_timing_update,
        ocp_master_socket_tl1<>::mm_txn_with_be()),
    cycle(0),
    reqs_sent(0),
    rd_reqs_sent(0),
    curr_addr(0),
    local_addr(0),
    curr_seq(ocpip::INCR),
    next_req_cycle(1),
    resps_rxd(0),
    precise_burst_len(1),
    bytes_left_in_burst(0),
    req_active(false),
    curr_is_read(true),
    scmdaccept(false),
    sresp(0),
    curr(0),
    os(sc_module::name()),
    srespT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_bw(this, &ocp_tl1_initiator::nb_transport_bw);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_master_socket_tl1<32> ocp;

private:
  void on_Clk_rising() {
    if((curr_addr >= MAX_ADDR) && (resps_rxd == rd_reqs_sent)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && scmdaccept)
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if(scmdaccept || !req_active) {
      if((cycle >= next_req_cycle) && (curr_addr < MAX_ADDR)) {

        // new bus burst
        if(bytes_left_in_burst == 0) {
          switch(curr_seq) {
            case INCR: curr_seq = DFLT1; break;
            case DFLT1: curr_seq = DFLT2; break;
            case DFLT2: curr_seq = UNKN; break;
            case UNKN: curr_seq = INCR; break;
          }
          if(cycle & 5) curr_is_read = !curr_is_read ;
          curr_addr += cycle % 4;
          local_addr = curr_addr & 0xFFFFFFFC;
          bytes_left_in_burst = 1 + (cycle % 56);  // max burst length is 15 words
          unsigned ocp_length = 1
                + (((curr_addr + bytes_left_in_burst - 1) & 0xFFFFFFFC)
                - local_addr) / 4;
          if(curr_seq == INCR) {
            if(precise_burst_len == 0) precise_burst_len = ocp_length;
            else precise_burst_len = 0;
          }
          curr = ocp.get_transaction();
          leak_test(*curr);
          curr->set_byte_enable_length(64);
          ocp.reserve_be_size(*curr, 64);
          curr->set_streaming_width(64);
          curr->set_response_status(TLM_INCOMPLETE_RESPONSE);
          if(curr_is_read) curr->set_read();
          else {
            curr->set_write();
            ocp.validate_extension<posted>(*curr);
          }
          burst_length *bl;
          ocp.get_extension<burst_length>(bl, *curr);
          burst_sequence *bs;
          ocp.get_extension<burst_sequence>(bs, *curr);
          if(curr_seq == INCR) {
            // special case for INCR - simple TLM2 transaction with minimal extensions
            curr->set_byte_enable_ptr(0);
            curr->set_data_length(bytes_left_in_burst);
            curr->set_address(curr_addr);
            if(curr_is_read) rd_orig_addr.push(curr_addr);
            curr->set_data_ptr((unsigned char *)memory + curr_addr);
            ocp.invalidate_extension<burst_sequence>(*curr);
            if(precise_burst_len > 0) {
              ocp.invalidate_extension<burst_length>(*curr);
              ocp.invalidate_extension<imprecise>(*curr);
            } else {
              ocp.validate_extension<burst_length>(*curr);
              bl->value = ocp_length;
              ocp.validate_extension<imprecise>(*curr);
            }
          } else {
            // Non-INCR burst types
            // addresses must be aligned to 4 which will be the unkn-dflt-bytes-per-addr
            curr->set_address(local_addr);
            if(curr_is_read) rd_orig_addr.push(local_addr);
            curr->set_data_ptr((unsigned char *)memory + local_addr);
            bs->value.sequence = (burst_seqs)curr_seq;
            bs->value.unkn_dflt_bytes_per_address = 4;
            bs->value.unkn_dflt_addresses_valid = true;
            if(bs->value.unkn_dflt_addresses.size() < ocp_length)
              bs->value.unkn_dflt_addresses.resize(ocp_length);
            ocp.validate_extension<burst_sequence>(*curr);
            ocp.validate_extension<burst_length>(*curr);
            ocp.validate_extension<imprecise>(*curr);
            curr->set_data_length(4 * ocp_length);
          }
        }
        unsigned end_addr = (curr_addr & 0xFFFFFFFC) + 4;
        unsigned bytes = end_addr - curr_addr;
        if(bytes > bytes_left_in_burst) {
          bytes = bytes_left_in_burst;
          end_addr = curr_addr + bytes;
        }

        burst_length *bl;
        ocp.get_extension<burst_length>(bl, *curr);
        burst_sequence *bs;
        ocp.get_extension<burst_sequence>(bs, *curr);

        bytes_left_in_burst -= bytes;

        unsigned be = (0xF >> (4 - bytes)) << (curr_addr & 3);
        if((precise_burst_len == 0) || (curr_seq != INCR)) {
          bl->value = (bytes_left_in_burst == 0 ? 1 : 2 + bytes_left_in_burst/4);
          if(curr_seq != INCR) {
            // BROKEN - can not assume curr->get_address() is stable
            unsigned beat = (local_addr - curr->get_address()) / 4;
            bs->value.unkn_dflt_addresses[beat] = curr_addr & 0xFFFFFFFC;
            for(unsigned x = 0; x < 4; x++)
              *(curr->get_byte_enable_ptr() + 4 * beat + x) =
                ((be & (1 << x)) ? TLM_BYTE_ENABLED : TLM_BYTE_DISABLED);
          }
        }

        switch(curr_seq) {
          // arbitrary address patterns for each burst type
          case INCR: curr_addr = end_addr; break;
          case DFLT1: curr_addr = end_addr + 8; break;
          case DFLT2: curr_addr = end_addr + 12; break;
          case UNKN: curr_addr = end_addr + 16; break;
        }

        char dir;
        if(!curr_is_read) {
          unsigned d = *reinterpret_cast<unsigned *>(memory + local_addr);
          os << cycle << " I: W data: " << d << ", BE = " << be << endl;
          dir = 'W';
        } else {
          rd_addr.push(local_addr);
          rd_be.push(be);
          rd_reqs_sent++;
          dir = 'R';
        }
        local_addr += 4;
        tlm_phase ph = BEGIN_REQ;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*curr, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_REQ);
          scmdaccept = true;
        } else {
          scmdaccept = false;
        }
        if((curr->is_write()) && (bytes_left_in_burst == 0))
          ocp.release_transaction(curr);
        reqs_sent++;
        req_active = true;
        os << cycle << " I: " << dir << " req sent @ " << curr_addr << endl;
        os << cycle << " I: " << dir << " req length " << bytes_left_in_burst << endl;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        req_active = false;
      }
    }

    // response side
    if(sresp != 0) {
      resps_rxd++;
      int addr = rd_addr.front();
      unsigned be = rd_be.front();
      unsigned orig_addr = rd_orig_addr.front();
      rd_addr.pop();
      rd_be.pop();
      unsigned *sdata = reinterpret_cast<unsigned *>(memory + addr);
      unsigned mask = 0;
      if(be & 1) mask |= 0xFF;
      if(be & 2) mask |= 0xFF00;
      if(be & 4) mask |= 0xFF0000;
      if(be & 8) mask |= 0xFF000000;
      os << cycle << " I: R data: " << (*sdata & mask) << ", BE = " << be << endl;
      os << cycle << " I: R resp received @ " << addr << endl;
      if((4 + addr) - orig_addr >= sresp->get_data_length()) {
        ocp.release_transaction(sresp);
        rd_orig_addr.pop();
      }
      sresp = 0;
    }
    cycle++;
  }

  tlm_sync_enum nb_transport_bw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_REQ) {
      scmdaccept = true;
      return TLM_ACCEPTED;
    }
    sc_assert(ph == BEGIN_RESP);
    sresp = &pl;
    ph = END_RESP;
    return TLM_UPDATED;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, rd_reqs_sent, curr_addr, local_addr, curr_seq;
  unsigned next_req_cycle, resps_rxd, precise_burst_len, bytes_left_in_burst;
  bool req_active, curr_is_read;
  queue<int> rd_addr;
  queue<unsigned> rd_be, rd_orig_addr;;
  bool scmdaccept;
  tlm_generic_payload *sresp, *curr;
  ofstream os;
  sc_time srespT;

  // not sure if this is redundant when there is no resp-accept in the OCP
  // configuration.  It still exists in the TLM2
  void ocp_timing_update(ocp_tl1_slave_timing times) {
    if(srespT != times.ResponseGrpStartTime) {
      srespT = times.ResponseGrpStartTime;
      set_master_timing();
    }
  }

  void set_master_timing() {
    ocp_tl1_master_timing mytimes;
    mytimes.MRespAcceptStartTime = srespT + sc_get_time_resolution();
    ocp.set_master_timing(mytimes);
  }
};


class ocp_tl0_target: public sc_module {
SC_HAS_PROCESS(ocp_tl0_target);
public:
  ocp_tl0_target(sc_module_name n):
    sc_module(n),
    cycle(0),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos() << MReset_n.neg();

    SResp.initialize(0);
    SCmdAccept.initialize(false);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  sc_in<unsigned> MAddr;
  sc_in<unsigned> MCmd;
  sc_in<unsigned> MData;
  sc_in<unsigned> MBurstLength;
  sc_in<unsigned> MBurstSeq;
  sc_in<bool> MBurstPrecise;
  sc_in<unsigned> MByteEn;
  sc_out<unsigned> SData;
  sc_out<bool> SCmdAccept;
  sc_out<unsigned> SResp;
  sc_in<bool> MReset_n;

private:
  void on_Clk_rising() {
    if(!MReset_n) {
      // reset all state
      while(resp_cycle.size() > 0) {
        resp_cycle.pop();
        rd_addr.pop();
      }
      SResp = 0;
    } else {
      // response side
      if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
        resp_cycle.pop();
        SResp = 1;
        int addr = rd_addr.front();
        rd_addr.pop();
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        SData = d;
        os << cycle << " T: R data: " <<  d << endl;
        os << cycle << " T: R resp sent @ " << addr << endl;
      } else {
        SResp = 0;
      }

      // request side
      // accept in cycles 3,6,9,12,15,... and impose 4 cycle minimum latency
      if(SCmdAccept) {
        unsigned mdata;
        unsigned be;
        char *local;
        unsigned mask;
        switch(MCmd) {
        case 0:
          break;
        case 1:
          mdata = MData;
          be = MByteEn;
          local = memory + MAddr;
          mask = 0;
          if(be & 1) {mask |= 0xFF; local[0] = mdata;}
          if(be & 2) {mask |= 0xFF00; local[1] = mdata >> 8;}
          if(be & 4) {mask |= 0xFF0000; local[2] = mdata >> 16;}
          if(be & 8) {mask |= 0xFF000000; local[3] = mdata >> 24;}
          os << cycle << " T: W data: " << (mdata & mask) << endl;
          os << cycle << " T: W req received @ " << MAddr << endl;
          os << cycle << " T: W req length " << MBurstLength << endl;
          break;
        case 2:
          rd_addr.push(MAddr);
          os << cycle << " T: R req received @ " << MAddr << endl;
          os << cycle << " T: R req length " << MBurstLength << endl;
          resp_cycle.push(cycle + 4);
          break;
        }
      }
      SCmdAccept = ((cycle % 3) == 0);

      cycle++;
    }
  }

  unsigned cycle;
  char memory[MEM_SIZE];
  queue<int> rd_addr;
  queue<unsigned> resp_cycle;
  ofstream os;
};


class ocp_tl1_target: public sc_module {
SC_HAS_PROCESS(ocp_tl1_target);
public:
  ocp_tl1_target(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_target::ocp_timing_update),
    cycle(0),
    in_reset(false),
    scmdaccept(false),
    mcmd(0),
    req_in_burst(0),
    resp_in_burst(0),
    os(sc_module::name()),
    cmdT(SC_ZERO_TIME),
    dataT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_fw(this, &ocp_tl1_target::nb_transport_fw);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_slave_socket_tl1<32> ocp;

private:
  struct ext: tlm_extension<ext> {
    void free() {delete this;}
    void copy_from(tlm_extension_base const &) {}
    tlm_extension_base *clone() const {return 0;}
    static ext &get(tlm_generic_payload &pl) {
      ext *e = pl.get_extension<ext>();
      if(e == 0) {e = new ext; pl.set_extension<ext>(e);}
      return *e;
    }
    bool precise;
    unsigned req_in_burst, resp_in_burst, length, addr;
    unsigned orig_len;
    burst_sequence *seq;
    ext(): req_in_burst(0) {}
  };

  void on_Clk_rising() {
    if(in_reset) return;

    // response side
    if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
      resp_cycle.pop();
      tlm_generic_payload *pl = rd_addr.front();
      rd_addr.pop();
      pl->set_response_status(TLM_OK_RESPONSE);
      ext &e(ext::get(*pl));

      // calculate OCP addr and byte enables for this beat of the burst
      unsigned addr;
      unsigned bypb = 4;
      if((e.seq != 0) && (e.seq->value.sequence == UNKN)) {
        unsigned wdth = e.seq->value.unkn_dflt_bytes_per_address;
        if(wdth < 4) {
          // original source of UNKN burst was narrow and UNKN burst has
          // not been repacked.  only part of each beat is used
          addr = e.seq->value.unkn_dflt_addresses[e.resp_in_burst];
          // addr is not OCP-aligned but bypb-aligned
          bypb = 4 / wdth;
        } else {
          // original source of UNKN burst was maybe wider and UNKN burst has
          // been cut into a sequence of INCR.
          unsigned s = wdth / 4;
          addr = e.seq->value.unkn_dflt_addresses[e.resp_in_burst / s]
            + 4 * (e.resp_in_burst % s);
        }
      } else {
        // for INCR and DLFTx bursts in MRMD, the per-beat addresses can be
        // ignored as they _must_ match the target's understanding
        // default is INCR
        unsigned m = 4;
        if((e.seq != 0) && (e.seq->value.sequence == DFLT1)) m = 12;
        if((e.seq != 0) && (e.seq->value.sequence == DFLT2)) m = 16;
        addr = (e.addr & 0xFFFFFFFC) + e.resp_in_burst * m;
      }

      unsigned sdata = 0;
      unsigned char *bep = pl->get_byte_enable_ptr();
      // byte enables are set for whatever is within the initiator buffer
      // and if there's a BE array, enabled in it
      int idx = (e.addr & 0xFFFFFFFC) + e.resp_in_burst * bypb - e.addr;
      char *local = memory + addr;
      unsigned char *init = pl->get_data_ptr() + idx;
      for(unsigned x = 0; x < bypb; x++) {
        if((0 <= idx + x) && (idx + x < pl->get_data_length()) && (
          (bep == 0) ||
          (bep[(idx + x) % pl->get_byte_enable_length()] == TLM_BYTE_ENABLED)))
        {
          init[x] = local[x];
        }
        sdata |= ((unsigned char)(local[x]) << (x * 8));
      }

      tlm_phase ph = BEGIN_RESP;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);

      ++e.resp_in_burst;
      pl->release();
      os << cycle << " T: R data: " << sdata << endl;
      os << cycle << " T: R resp sent @ " << addr << endl;
    }

    // request side
    // accept in cycles 3,6,9,12,15,... and impose 4 cycle minimum latency
    if(scmdaccept) {
      if(mcmd != 0) {
        ext &e(ext::get(*mcmd));
        if(e.req_in_burst == 0) {
          // new transaction
          leak_test(*mcmd);
          e.resp_in_burst = 0;
          e.addr = mcmd->get_address();
          e.precise = !ocp.get_extension<imprecise>(*mcmd);
          if(e.precise) {
            e.length = (e.orig_len > 0 ? e.orig_len :
              (3 + mcmd->get_data_length() + (mcmd->get_address() & 3)) / 4);
          }
          if(!ocp.get_extension<burst_sequence>(e.seq, *mcmd)) e.seq = 0;
        }

        // current burst length
        unsigned cbl = e.length;
        if((!e.precise) && (e.orig_len > 0)) cbl = e.orig_len;

        // calculate OCP addr and byte enables for this beat of the burst
        unsigned addr;
        unsigned bypb = 4;
        if((e.seq != 0) && (e.seq->value.sequence == UNKN)) {
          unsigned wdth = e.seq->value.unkn_dflt_bytes_per_address;
          if(wdth < 4) {
            // original source of UNKN burst was narrow and UNKN burst has
            // not been repacked.  only part of each beat is used
            addr = e.seq->value.unkn_dflt_addresses[e.req_in_burst];
            // addr is not OCP-aligned but bypb-aligned
            bypb = 4 / wdth;
          } else {
            // original source of UNKN burst was maybe wider and UNKN burst has
            // been cut into a sequence of INCR.
            unsigned s = wdth / 4;
            addr = e.seq->value.unkn_dflt_addresses[e.req_in_burst / s]
              + 4 * (e.req_in_burst % s);
          }
        } else {
          // for INCR and DLFTx bursts in MRMD, the per-beat addresses can be
          // ignored as they _must_ match the target's understanding
          // default is INCR
          unsigned m = 4;
          if((e.seq != 0) && (e.seq->value.sequence == DFLT1)) m = 12;
          if((e.seq != 0) && (e.seq->value.sequence == DFLT2)) m = 16;
          addr = (e.addr & 0xFFFFFFFC) + e.req_in_burst * m;
        }

        if(mcmd->is_write()) {
          unsigned mdatam = 0;
          unsigned char *bep = mcmd->get_byte_enable_ptr();
          // byte enables are set for whatever is within the initiator buffer
          // and if there's a BE array, enabled in it
          int idx = (e.addr & 0xFFFFFFFC) + e.req_in_burst * bypb - e.addr;
          char *local = memory + addr;
          unsigned char *init = mcmd->get_data_ptr() + idx;
          for(int x = 0; x < int(bypb); x++) {
            if(((0 <= (idx + x)) && ((idx + x) < int(mcmd->get_data_length())))
              && ((bep == 0) ||
              (bep[(idx + x) % mcmd->get_byte_enable_length()] == TLM_BYTE_ENABLED)))
            {
               local[x] = init[x];
               mdatam |= init[x] << (x * 8);
            }
          }
          os << cycle << " T: W data: " << mdatam << endl;
          os << cycle << " T: W req received @ " << addr << endl;
          os << cycle << " T: W req length " << cbl << endl;
          mcmd->release();
        } else {
          os << cycle << " T: R req received @ " << addr << endl;
          os << cycle << " T: R req length " << cbl << endl;
          resp_cycle.push(cycle + 4);
        }
        ++e.req_in_burst;
        if((e.precise && (e.length == e.req_in_burst)) || (cbl == 1)) {
          e.req_in_burst = 0;
        }
        mcmd = 0;
      }
    }
    scmdaccept = ((cycle % 3) == 0);
    if(scmdaccept && (mcmd != 0)) {
      tlm_phase ph = END_REQ;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mcmd, ph, t);
    }

    cycle++;
  }

  tlm_sync_enum nb_transport_fw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_RESET) {
      in_reset = false;
      while(rd_addr.size() > 0) {
        rd_addr.front()->release();
        rd_addr.pop();
      }
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_RESET) {
      in_reset = true;
      mcmd = 0;
      ocp.reset();
      while(resp_cycle.size() > 0) resp_cycle.pop();
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_REQ) {
      mcmd = &pl;
      ext &e(ext::get(*mcmd));
      burst_length *bl;
      bool has_bl = ocp.get_extension<burst_length>(bl, *mcmd);
      e.orig_len = (has_bl ? bl->value : 0);
      pl.acquire();
      if(pl.is_read()) rd_addr.push(&pl);
      if(scmdaccept) {
        ph = END_REQ;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    sc_assert(ph == END_RESP);
    return TLM_ACCEPTED;
  }

  unsigned cycle;
  bool in_reset;
  bool scmdaccept;
  tlm_generic_payload *mcmd;
  unsigned req_in_burst, resp_in_burst;
  char memory[MEM_SIZE];
  queue<tlm_generic_payload *> rd_addr;
  queue<unsigned> resp_cycle;
  ofstream os;
  sc_time cmdT, dataT;

  void ocp_timing_update(ocp_tl1_master_timing times) {
    if((cmdT != times.RequestGrpStartTime) ||
       (dataT != times.DataHSGrpStartTime)) {
      cmdT = times.RequestGrpStartTime;
      dataT = times.DataHSGrpStartTime;
      set_slave_timing();
    }
  }

  void set_slave_timing() {
    ocp_tl1_slave_timing mytimes;
    mytimes.SCmdAcceptStartTime = cmdT + sc_get_time_resolution();
    mytimes.SDataAcceptStartTime = dataT + sc_get_time_resolution();
    ocp.set_slave_timing(mytimes);
  }
};


class testbench00: public sc_module {
public:
  testbench00(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.MBurstLength(MBurstLength);
    initiator.MBurstSeq(MBurstSeq);
    initiator.MBurstPrecise(MBurstPrecise);
    initiator.MByteEn(MByteEn);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.MBurstLength(MBurstLength);
    target.MBurstSeq(MBurstSeq);
    target.MBurstPrecise(MBurstPrecise);
    target.MByteEn(MByteEn);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> MBurstLength;
  sc_signal<unsigned> MBurstSeq;
  sc_signal<bool> MBurstPrecise;
  sc_signal<unsigned> MByteEn;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench01: public sc_module {
public:
  testbench01(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.MBurstLength(MBurstLength);
    initiator.MBurstSeq(MBurstSeq);
    initiator.MBurstPrecise(MBurstPrecise);
    initiator.MByteEn(MByteEn);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.MBurstLength(MBurstLength);
    adapter.MBurstSeq(MBurstSeq);
    adapter.MBurstPrecise(MBurstPrecise);
    adapter.MByteEn(MByteEn);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);

    // TL1 interface
    adapter.ocpTL1(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl1_target target;
  tl0_initiator_to_tl1_target<32, MRMDBURST_TYPES> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> MBurstLength;
  sc_signal<unsigned> MBurstSeq;
  sc_signal<bool> MBurstPrecise;
  sc_signal<unsigned> MByteEn;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench10: public sc_module {
public:
  testbench10(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter.ocpTL1);

    // TL0 interface
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.MBurstLength(MBurstLength);
    adapter.MBurstSeq(MBurstSeq);
    adapter.MBurstPrecise(MBurstPrecise);
    adapter.MByteEn(MByteEn);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.MBurstLength(MBurstLength);
    target.MBurstSeq(MBurstSeq);
    target.MBurstPrecise(MBurstPrecise);
    target.MByteEn(MByteEn);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl0_target target;
  tl1_initiator_to_tl0_target<32, MRMDBURST_TYPES> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> MBurstLength;
  sc_signal<unsigned> MBurstSeq;
  sc_signal<bool> MBurstPrecise;
  sc_signal<unsigned> MByteEn;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench11: public sc_module {
public:
  testbench11(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL1 interface
    initiator.ocp(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;
};


class testbench101: public sc_module {
public:
  testbench101(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter10("adapter10", &t_print),
    adapter01("adapter01", &t_print)
  {
    // OCP configuration
    adapter10.set_ocp_config(gen_ocp_config());
    adapter01.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter10.Clk(Clk);
    adapter01.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter10.ocpTL1);
    adapter01.ocpTL1(target.ocp);

    // TL0 interface
    adapter10.MAddr(MAddr);
    adapter10.MCmd(MCmd);
    adapter10.MData(MData);
    adapter10.MBurstLength(MBurstLength);
    adapter10.MBurstSeq(MBurstSeq);
    adapter10.MBurstPrecise(MBurstPrecise);
    adapter10.MByteEn(MByteEn);
    adapter10.SData(SData);
    adapter10.SCmdAccept(SCmdAccept);
    adapter10.SResp(SResp);
    adapter10.MReset_n(MReset_n);
    adapter01.MAddr(MAddr);
    adapter01.MCmd(MCmd);
    adapter01.MData(MData);
    adapter01.MBurstLength(MBurstLength);
    adapter01.MBurstSeq(MBurstSeq);
    adapter01.MBurstPrecise(MBurstPrecise);
    adapter01.MByteEn(MByteEn);
    adapter01.SData(SData);
    adapter01.SCmdAccept(SCmdAccept);
    adapter01.SResp(SResp);
    adapter01.MReset_n(MReset_n);

    // timing:  TL1 initiator and target produce transport calls at +0 ps
    //   but they aren't seen by adapters until +1 ps because of synch protection
    //   and only at this time are TL0 signals written
    adapter01.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter10 writes MCmd always during TLM-transport-from-initiator at +1 ps
      sc_time(3.0, SC_PS)
      // same for MDataValid but sampling is delayed to be after MCmd sampling,
      // which would be automatic but shown here for clarity
    );
    adapter10.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter01 writes SResp always during TLM-transport-from-target at +1 ps
      sc_time(5.0, SC_PS),
      // TL1 target may signal SCmdAccept during its clocked process with an
      // explicit transport call.  This would be seen at +1 ps by adapter01 and
      // the TL0 SCmdAccept signal would be stable for sampling at +2 ps.  But
      // TL1 target might return SCmdAccept to a fw transport call, which it
      // would see at the MCmd sampling time of adapter01 plus 1, ie 3 ps.  The
      // socket PEQ will convert this into a separate backward transport call
      // seen by adapter01 at 4 ps.  Thus SCmdAccept should be sampled at 5 ps.
      sc_time(6.0, SC_PS)
      // data trails command by 1 ps
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;

  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl1_initiator_to_tl0_target<32, MRMDBURST_TYPES> adapter10;
  tl0_initiator_to_tl1_target<32, MRMDBURST_TYPES> adapter01;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> MBurstLength;
  sc_signal<unsigned> MBurstSeq;
  sc_signal<bool> MBurstPrecise;
  sc_signal<unsigned> MByteEn;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench010: public sc_module {
public:
  testbench010(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter01("adapter01"),
    adapter10("adapter10")
  {
    // OCP configuration
    adapter01.set_ocp_config(gen_ocp_config());
    adapter10.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter01.Clk(Clk);
    adapter10.Clk(Clk);

    // TL1 interface
    adapter01.ocpTL1(adapter10.ocpTL1);

    // TL0 interfaces
    initiator.MAddr(MAddr[0]);
    initiator.MCmd(MCmd[0]);
    initiator.MData(MData[0]);
    initiator.MBurstLength(MBurstLength[0]);
    initiator.MBurstSeq(MBurstSeq[0]);
    initiator.MBurstPrecise(MBurstPrecise[0]);
    initiator.MByteEn(MByteEn[0]);
    initiator.SData(SData[0]);
    initiator.SCmdAccept(SCmdAccept[0]);
    initiator.SResp(SResp[0]);
    initiator.MReset_n(MReset_n[0]);
    adapter01.MAddr(MAddr[0]);
    adapter01.MCmd(MCmd[0]);
    adapter01.MData(MData[0]);
    adapter01.MBurstLength(MBurstLength[0]);
    adapter01.MBurstSeq(MBurstSeq[0]);
    adapter01.MBurstPrecise(MBurstPrecise[0]);
    adapter01.MByteEn(MByteEn[0]);
    adapter01.SData(SData[0]);
    adapter01.SCmdAccept(SCmdAccept[0]);
    adapter01.SResp(SResp[0]);
    adapter01.MReset_n(MReset_n[0]);

    adapter10.MAddr(MAddr[1]);
    adapter10.MCmd(MCmd[1]);
    adapter10.MData(MData[1]);
    adapter10.MBurstLength(MBurstLength[1]);
    adapter10.MBurstSeq(MBurstSeq[1]);
    adapter10.MBurstPrecise(MBurstPrecise[1]);
    adapter10.MByteEn(MByteEn[1]);
    adapter10.SData(SData[1]);
    adapter10.SCmdAccept(SCmdAccept[1]);
    adapter10.SResp(SResp[1]);
    adapter10.MReset_n(MReset_n[1]);
    target.MAddr(MAddr[1]);
    target.MCmd(MCmd[1]);
    target.MData(MData[1]);
    target.MBurstLength(MBurstLength[1]);
    target.MBurstSeq(MBurstSeq[1]);
    target.MBurstPrecise(MBurstPrecise[1]);
    target.MByteEn(MByteEn[1]);
    target.SData(SData[1]);
    target.SCmdAccept(SCmdAccept[1]);
    target.SResp(SResp[1]);
    target.MReset_n(MReset_n[1]);

    // timing: nothing needed as adapters' TL0 interfaces are
    //  directly connected to TL0-native initiator and target
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;
  tl0_initiator_to_tl1_target<32, MRMDBURST_TYPES> adapter01;
  tl1_initiator_to_tl0_target<32, MRMDBURST_TYPES> adapter10;

  sc_signal<unsigned> MAddr[2];
  sc_signal<unsigned> MCmd[2];
  sc_signal<unsigned> MData[2];
  sc_signal<unsigned> MBurstLength[2];
  sc_signal<unsigned> MBurstSeq[2];
  sc_signal<bool> MBurstPrecise[2];
  sc_signal<unsigned> MByteEn[2];
  sc_signal<unsigned> SData[2];
  sc_signal<bool> SCmdAccept[2];
  sc_signal<unsigned> SResp[2];
  sc_signal<bool> MReset_n[2];
};


class testbench: public sc_module {
public:
  testbench(sc_module_name n, bool run_01 = true, bool run_10 = true):
    sc_module(n),
    tb00("00"),
    tb01("01"),
    tb10("10"),
    tb11("11"),
    tb101("101"),
    tb010("010"),
    Clk("Clk", sc_time(10.0, SC_NS))
  {
    tb00.Clk(Clk);
    tb11.Clk(Clk);

    if(run_01) tb01.Clk(Clk);
    else tb01.Clk(tieoff);

    if(run_10) tb10.Clk(Clk);
    else tb10.Clk(tieoff);

    if(run_01 && run_10) {
      tb010.Clk(Clk);
      tb101.Clk(Clk);
    } else {
      tb101.Clk(tieoff);
      tb010.Clk(tieoff);
    }
  }
private:
  testbench00 tb00;
  testbench01 tb01;
  testbench10 tb10;
  testbench11 tb11;
  testbench101 tb101;
  testbench010 tb010;
  sc_clock Clk;
  sc_signal<bool> tieoff;
};


#ifndef OMIT_SC_MAIN
int sc_main(int argc, char **argv) {
  sc_report_handler::set_actions(SC_ERROR, SC_DISPLAY | SC_ABORT);
  bool run_01 = ((argc < 2) || ('A' == *argv[1]) || ('0' == *argv[1]));
  bool run_10 = ((argc < 2) || ('A' == *argv[1]) || ('1' == *argv[1]));
  testbench theTestbench("testbench", run_01, run_10);
  sc_start();
  return 0;
}
#endif

