///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  TL1/TL0 OCP Master Adapter
//
// SystemC Module Adapting between a TLM2-based OCP-TL1 slave interface
// and a TL0 (signal-level) OCP interface
// Typical use is between a TL0 OCP initiator and a TL1 OCP target
//
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "ocpip_adapters_dev.h"
// namespace
namespace ocpip_dev { namespace tl1_tl0 {

// WARNING: do not use the character sequence "_dev" in this file
// because it will be replaced by the installation process


using namespace tlm;
using namespace sc_core;
using sc_dt::uint64;


// private namespace for support infrastructure
namespace support_01 {

// base class for OCP burst tracker classes
struct tracker_base {
  virtual ~tracker_base() {}
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) = 0;

  // Req
  virtual bool ReqDone(bool track) = 0;
  bool ReqAcc() {return XAcc(req);}
  unsigned char *ReqBEPtr() {return XBEPtr(req);}
  unsigned char *ReqDataPtr() {return XDataPtr(req);}
  unsigned ReqBeat() {return req.nr;}

  // WR data
  virtual bool WrDone(bool track) = 0;
  bool WrAcc() {return XAcc(data);}
  unsigned char *WrBEPtr() {return XBEPtr(data);}
  unsigned char *WrDataPtr() {return XDataPtr(data);}
  unsigned WrBeat() {return data.nr;}

  // RD data
  virtual bool RdDone(bool track) = 0;
  bool RdAcc() {return XAcc(resp);}
  unsigned char *RdDataPtr() {return XDataPtr(resp);}
  unsigned RdBeat() {return resp.nr;}

  // management
  virtual void return_to_pool() = 0;

  // members
  struct handshake {unsigned nr, done, acc;};
  handshake req, data, resp;
  tlm_generic_payload *txn;
  tl0_to_tl1_base *adapter;

  virtual bool XAcc(handshake &hs) {
    ++hs.acc;
    return((req.acc == req.nr) && (data.acc == data.nr) && (resp.acc == resp.nr));
  }
  virtual unsigned char *XDataPtr(handshake &hs) {
    unsigned off = adapter->byte_wdth * hs.done;
    sc_assert(off < adapter->max_burst_bytes);
    return txn->get_data_ptr() + off;
  }
  virtual unsigned char *XBEPtr(handshake &hs) {
    return txn->get_byte_enable_ptr() + adapter->byte_wdth * hs.done;
  }
};


// class for doing RDEX/WR atomic transaction pairs
struct lock_object: lock_object_base {
  void atomic_txn_completed() {if(--count == 0) pool(this);}
  unsigned count, ocp_thread;
  lock_object *next;
  static lock_object *pool(lock_object *used = 0) {
    static lock_object *available = 0;
    if(used == 0) {
      // getting something from pool
      if(available == 0) {available = new lock_object; available->next = 0;}
      lock_object *rv = available;
      available = available->next;
      rv->count = 2;
      return rv;
    } else {
      // returning something to pool
      used->next = available;
      available = used;
      return 0;
    }
  }
};


// extension class for adapter private use
class ext: public tlm_extension<ext> {
public:
  // required stuff
  void free() {delete this;}
  void copy_from(tlm_extension_base const &) {}
  tlm_extension_base *clone() const {return 0;}
  static ext &get(tlm_generic_payload &pl) {return *(pl.get_extension<ext>());}

  // construct and destruct
  ext(unsigned datasize, bool be) {
    data_buffer = new unsigned char[datasize];
    if(be) be_buffer = new unsigned char[datasize];
    else be_buffer = 0;
  }
  ~ext() {
    delete [] data_buffer;
    if(be_buffer != 0) delete [] be_buffer;
  }

  // content
  tlm_generic_payload *next, *prev;
  unsigned thread;
  int tag;  // tag -1 indicates tag-in-order
  unsigned char *data_buffer, *be_buffer;
  tracker_base *tracker;
  lock_object *is_rdex;
  semaphore *is_wrc;
};


// function for initialising a TLM transaction and extension
// during the first request phase
void init_ext(tlm_generic_payload &, tl0_to_tl1_base &);


// wrapper for processes
struct process {
  void operator()() {(adapter->*pf)();}
  tl0_to_tl1_base *adapter;
  tl0_to_tl1_base::callable pf;
  process(
    tl0_to_tl1_base::callable p, tl0_to_tl1_base *a,
    const char *n, const sc_core::sc_event &e)
    : adapter(a), pf(p)
  {
    sc_spawn_options so;
    so.spawn_method();
    so.dont_initialize();
    so.set_sensitivity(&e);
    sc_spawn(*this, n, &so);
  }
};

// end of support namespace
} using namespace support_01;


// class methods
//////////////////////////////////////////


// process function for critical sc_signal writing
void tl0_to_tl1_base::writer_process() {
  if(sresp_wr) sresp(new_sresp);
  if(scmdaccept_wr) scmdaccept(new_scmdaccept);
  if(sdataaccept_wr) sdataaccept(new_sdataaccept);
  sresp_wr = scmdaccept_wr = sdataaccept_wr = false;
}


// constructor
tl0_to_tl1_base::tl0_to_tl1_base(sc_module_name n):
  sc_module(n),
  scmdaccept_wr(false),
  sdataaccept_wr(false),
  sresp_wr(false),
  max_burst_bytes(0),
  newest_lock(0),
  request_t(sc_get_time_resolution()),
  data_t(2 * sc_get_time_resolution()),
  respaccept_t(sc_get_time_resolution()),
  threadbusy_t(sc_get_time_resolution()),
  resp_t(SC_ZERO_TIME),
  oldest_txn(0),
  newest_txn(0),
  respaccept_state(false),
  curr_tl0_cmd(0),
  curr_tl0_data(0),
  curr_tl0_resp(0),
  curr_mthreadbusy(~0)
{
}


// destructor
tl0_to_tl1_base::~tl0_to_tl1_base() {
}


void tl0_to_tl1_base::end_of_elaboration() {
  byte_wdth = ocp_config.data_wdth / 8;

  if(ocp_config.burstlength) {
    unsigned prec_burst_len = (1 << ocp_config.burstlength_wdth) * byte_wdth;
    if(ocp_config.burstseq_blck_enable)
      prec_burst_len *= (1 << ocp_config.blockheight_wdth);
    if(!ocp_config.burstprecise || (prec_burst_len > max_burst_bytes)) {
      max_burst_bytes = prec_burst_len;
    }
  } else {
    max_burst_bytes = byte_wdth;
  }

  // set up clocked process
  process(&tl0_to_tl1_base::clk_rising_edge, this, "clk_edge",
    clk_port->posedge_event());

  // signal writer process
  process(&tl0_to_tl1_base::writer_process, this, "signal_writer",
    writer_trigger);

  // set up other processes that will fire at sample times
  process(&tl0_to_tl1_base::request_sample, this, "request_sample",
    request_e);

  if(ocp_config.datahandshake) {
    process(&tl0_to_tl1_base::data_sample, this, "data_sample",
      data_e);
  } else curr_tl0_data = reinterpret_cast<tlm_generic_payload *>(-1);

  if(ocp_config.respaccept) {
    process(&tl0_to_tl1_base::respaccept_sample, this, "respaccept_sample",
      respaccept_e);
  } else respaccept_state = true;

  if(ocp_config.mthreadbusy) {
    process(&tl0_to_tl1_base::threadbusy_sample, this, "threadbusy_sample",
      threadbusy_e);
  }

  if(ocp_config.mflag) {
    process(&tl0_to_tl1_base::handle_mflag, this, "handle_mflag", mflag_event());
  }

  if(ocp_config.merror) {
    process(&tl0_to_tl1_base::handle_merror, this, "handle_merror", merror_event());
  }

  if(ocp_config.mreset) {
    process(&tl0_to_tl1_base::handle_mreset, this, "handle_mreset", mreset_event());
  }
}


// clock edge behaviour
void tl0_to_tl1_base::clk_rising_edge() {
  if(!enableclk()) return;

  // pull critical TL0 signals to IDLE
  bool do_critical_writes = ocp_config.cmdaccept || ocp_config.dataaccept;
  if(respaccept_state) {
    new_sresp = RESP_NULL;
    do_critical_writes = sresp_wr = true;
  } else sresp_wr = false;
  new_scmdaccept = false;
  scmdaccept_wr = true;
  new_sdataaccept = false;
  sdataaccept_wr = true;
  if(do_critical_writes) writer_trigger.notify();

  // trigger sampling processes
  // do not sample requests if one is stalled on the bus
  if(curr_tl0_cmd == 0) request_e.notify(request_t);

  // do not sample write data if some is stalled on the bus
  if(curr_tl0_data == 0) data_e.notify(data_t);

  if(ocp_config.respaccept) {
    respaccept_e.notify(respaccept_t);
    respaccept_state = false;
  }

  if(ocp_config.mthreadbusy) threadbusy_e.notify(threadbusy_t);
};


// TLM transport into this module
tlm_sync_enum tl0_to_tl1_base::nb_transport_bw(
  tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
{
  if(ph == END_REQ) {
    end_req(pl);
    return TLM_ACCEPTED;
  }

  if(ph == END_DATA) {
    end_data(pl);
    return TLM_ACCEPTED;
  }

  if(ph == BEGIN_RESP) {
    ext &e(ext::get(pl));
    tracker_base *t = e.tracker;
    // put the response onto the TL0 interface
    new_sresp =
      (pl.get_response_status() == TLM_OK_RESPONSE ? RESP_DVA : RESP_ERR);
    if((e.is_wrc != 0) && e.is_wrc->value) new_sresp = RESP_FAIL;
    sresp_wr = true;
    writer_trigger.notify();

    if(pl.is_read()) {
      // RD data
      unsigned char *d = t->RdDataPtr();
      for(unsigned b = 0; b < byte_wdth; b++) sdata(*d++, b);
      sdata();
      if(e.is_rdex != 0) {
        sc_assert(e.is_rdex->lock_is_understood_by_slave);
        e.is_rdex->atomic_txn_completed();
      }
    }
    staginorder(e.tag < 0);
    stagid(e.tag < 0 ? 0 : e.tag);
    sthreadid(e.thread);
    sdatainfo(&pl, t->RdBeat());
    srespinfo(&pl);
    sresplast(t->RdDone(true)); // will set SRespRowLast if needed

    // return: can only accept _after_ resp-accept has been sampled
    if(respaccept_state) {
      end_resp(pl);
      ph = END_RESP;
      return TLM_UPDATED;
    } else {
      curr_tl0_resp = &pl;
      return TLM_ACCEPTED;
    }
  }

  if(ph == CMD_THREAD_BUSY_CHANGE) {
    // extract the thread-busy
    cmd_thread_busy *tbex;
    extension_api::get_extension<cmd_thread_busy>(tbex, pl);
    // forward to TL0
    sthreadbusy(tbex->value);
    return TLM_ACCEPTED;
  }

  if(ph == DATA_THREAD_BUSY_CHANGE) {
    // extract the thread-busy
    data_thread_busy *tbex;
    extension_api::get_extension<data_thread_busy>(tbex, pl);
    // forward to TL0
    sdatathreadbusy(tbex->value);
    return TLM_ACCEPTED;
  }

  if(ph == BEGIN_ERROR) {
    serror(true);
    return TLM_ACCEPTED;
  }

  if(ph == END_ERROR) {
    serror(false);
    return TLM_ACCEPTED;
  }

  if(ph == BEGIN_INTERRUPT) {
    sinterrupt(true);
    return TLM_ACCEPTED;
  }

  if(ph == END_INTERRUPT) {
    sinterrupt(false);
    return TLM_ACCEPTED;
  }

  if(ph == BEGIN_RESET) {
    sreset_n(false);
    return TLM_ACCEPTED;
  }

  if(ph == END_RESET) {
    sreset_n(true);
    exit_reset();
    return TLM_ACCEPTED;
  }

  sc_assert(ph == SFLAG_CHANGE); {
    sflag(&pl);
    return TLM_ACCEPTED;
  }
}


// response accept may be done in return code to bw transport call or
// in fw transport call from sample process.  same behaviour in both cases
void tl0_to_tl1_base::end_resp(tlm_generic_payload &pl) {
  if(ext::get(pl).tracker->RdAcc()) remove_from_list(pl);
  curr_tl0_resp = 0;
}


// cmd accept may be seen in return from transport fw or in a dedicated
// transport bw
void tl0_to_tl1_base::end_req(tlm_generic_payload &pl) {
  curr_tl0_cmd = 0;
  new_scmdaccept = true;
  scmdaccept_wr = true;
  if(ocp_config.cmdaccept) writer_trigger.notify();
  if(ext::get(pl).tracker->ReqAcc()) remove_from_list(pl);
}


// ditto data accept
void tl0_to_tl1_base::end_data(tlm_generic_payload &pl) {
  curr_tl0_data = 0;
  new_sdataaccept = true;
  sdataaccept_wr = true;
  if(ocp_config.dataaccept) writer_trigger.notify();
  if(ext::get(pl).tracker->WrAcc()) remove_from_list(pl);
}


tlm_generic_payload &tl0_to_tl1_base::add_transaction(unsigned cmd) {
  // provide a payload object for a BEGIN_REQ
  // may be a new transaction or an ongoing MRMD, in which case we
  // need the most recent for the sampled thread-ID
  if(ocp_config.burstlength) {
    // need to search for ongoing request sequence
    tlm_generic_payload *pn = newest_txn;
    unsigned thread = mthreadid();
    while(pn != 0) {
      ext &n(ext::get(*pn));
      if(thread == n.thread) {
        if(!n.tracker->ReqDone(false)) return *pn;
        else break;  // can stop searching at the newest from the thread
      }
      pn = n.prev;
    }
  }

  // get a payload object from the OCP socket
  tlm_generic_payload &pl(*get_transaction());
  init_ext(pl, *this);

  // add the transaction to the linked-list
  ext &e(ext::get(pl));
  e.next = 0;
  e.prev = newest_txn;
  if(newest_txn == 0) oldest_txn = &pl;
  else ext::get(*newest_txn).next = &pl;
  newest_txn = &pl;

  return pl;
}


void tl0_to_tl1_base::remove_from_list(tlm_generic_payload &pl) {
  ext &e(ext::get(pl));
  if(e.prev == 0) oldest_txn = e.next;
  else ext::get(*(e.prev)).next = e.next;
  if(e.next == 0) newest_txn = e.prev;
  else ext::get(*(e.next)).prev = e.prev;
  release_transaction(&pl);
}


// sample time behaviour: request
void tl0_to_tl1_base::request_sample() {
  unsigned cmd = mcmd();
  if(cmd != CMD_IDLE) {
    // find the TLM payload for this command, which may be new
    tlm_generic_payload &pl(add_transaction(cmd));
    ext &e(ext::get(pl));
    tracker_base *t = e.tracker;

    if(pl.is_write()) {
      if(!ocp_config.datahandshake) {
        // MData
        unsigned char *d = t->ReqDataPtr();
        for(unsigned b = 0; b < byte_wdth; b++) *d++ = mdata(b);
        // MDataInfo
        mdatainfo(&pl, t->ReqBeat());
      }
    }

    // MByteEnable
    if(ocp_config.byteen && (pl.is_read()
        || !(ocp_config.datahandshake && ocp_config.mdatabyteen))) {
      uint64 be = mbyteen();
      if(mburstsinglereq()) {
        // SRMD: fan out byte enables to all beats of burst
        unsigned char *d = pl.get_byte_enable_ptr();
        unsigned len = (t->data.nr == 0 ? t->resp.nr : t->data.nr);
        for(unsigned beat = 0; beat < len; beat++)
          for(unsigned b = 0; b < byte_wdth; b++)
            *d++ = (be & (1 << b) ? TLM_BYTE_ENABLED : TLM_BYTE_DISABLED);
      } else {
        // MRMD: capture the current byte enables
        unsigned char *d = t->ReqBEPtr();
        for(unsigned b = 0; b < byte_wdth; b++)
          *d++ = (be & (1 << b) ? TLM_BYTE_ENABLED : TLM_BYTE_DISABLED);
      }
    }

    // MReqInfo
    mreqinfo(&pl);

    // other request group signals are processed by the tracker in ReqDone()
    // or are constant in a burst and read during init().
    // signals that might need to be read at every beat are MAddr and MBurstLen

    t->ReqDone(true);
    curr_tl0_cmd = &pl;
    tlm_phase ph(BEGIN_REQ);
    sc_time tm(SC_ZERO_TIME);
    if(nb_transport_fw(pl, ph, tm) == TLM_UPDATED) end_req(pl);
  }
}


// sample time behaviour: data handshake
void tl0_to_tl1_base::data_sample() {
  if(mdatavalid()) {
    // find the TLM payload for this command - oldest that has data outstanding
    tlm_generic_payload *pl = oldest_txn;
    unsigned thread = mdatathreadid();
    while(pl != 0) {
      ext &e(ext::get(*pl));
      if((thread == e.thread) && !e.tracker->WrDone(false)) {
        unsigned char *d = e.tracker->WrDataPtr();
        for(unsigned b = 0; b < byte_wdth; b++) *d++ = mdata(b);
        if(ocp_config.mdatabyteen) {
          unsigned char *d = e.tracker->WrBEPtr();
          uint64 be = mdatabyteen();
          for(unsigned b = 0; b < byte_wdth; b++)
            *d++ = (be & (1 << b) ? TLM_BYTE_ENABLED : TLM_BYTE_DISABLED);
        }
        // MDataInfo
        mdatainfo(pl, e.tracker->WrBeat());

        e.tracker->WrDone(true);
        curr_tl0_data = pl;
        tlm_phase ph(BEGIN_DATA);
        sc_time t(SC_ZERO_TIME);
        if(nb_transport_fw(*pl, ph, t) == TLM_UPDATED) end_data(*pl);
        return;
      }
      pl = e.next;
    }
    sc_assert(false); // can not find a transaction waiting for WR data
  }
}


// sample time behaviour: response accept
void tl0_to_tl1_base::respaccept_sample() {
  respaccept_state = mrespaccept();  // enables transport to accept commands
  if(respaccept_state && (curr_tl0_resp != 0)) {
    tlm_phase ph(END_RESP);
    sc_time t(SC_ZERO_TIME);
    nb_transport_fw(*curr_tl0_resp, ph, t);
    end_resp(*curr_tl0_resp);
  }
}


// sample time behaviour: m-thread-busy
void tl0_to_tl1_base::threadbusy_sample() {
  unsigned tb = mthreadbusy();
  if(tb != curr_mthreadbusy) {
    curr_mthreadbusy = tb;
    tlm_phase ph(RESP_THREAD_BUSY_CHANGE);
    sc_time t(SC_ZERO_TIME);
    resp_thread_busy *tbex;
    tlm_generic_payload *tbpl = get_tb_transaction();
    extension_api::get_extension<resp_thread_busy>(tbex, *tbpl);
    tbex->value = tb;
    nb_transport_fw(*tbpl, ph, t);
  }
}

// sideband signalling support
void tl0_to_tl1_base::handle_mflag() {
  tlm_generic_payload *pl = get_flag_transaction();
  mflag(pl);
  tlm_phase ph(MFLAG_CHANGE);
  sc_time t(SC_ZERO_TIME);
  nb_transport_fw(*pl, ph, t);
}

void tl0_to_tl1_base::handle_merror() {
  tlm_generic_payload *pl = get_error_transaction();
  tlm_phase ph;
  if(merror()) ph = BEGIN_ERROR; else ph = END_ERROR;
  sc_time t(SC_ZERO_TIME);
  nb_transport_fw(*pl, ph, t);
}

void tl0_to_tl1_base::handle_mreset() {
  tlm_generic_payload *pl = get_reset_transaction();
  tlm_phase ph;
  if(mreset_n()) ph = END_RESET; else ph = BEGIN_RESET;
  sc_time t(SC_ZERO_TIME);
  nb_transport_fw(*pl, ph, t);
  if(mreset_n()) exit_reset();
}

void tl0_to_tl1_base::exit_reset() {
  ocp_socket_reset();
  lock_object *lko = newest_lock;
  while(lko != 0) {
    lko->atomic_txn_completed();
    lko = lko->next;
  }
  newest_lock = 0;
  tlm_generic_payload *pl = oldest_txn;
  while(pl != 0) {
    tlm_generic_payload *to_release = pl;
    pl = ext::get(*pl).next;
    release_transaction(to_release);
  }
  oldest_txn = newest_txn = 0;
  new_scmdaccept = false;
  new_sdataaccept = false;
  new_sresp = RESP_NULL;
  scmdaccept_wr = sdataaccept_wr = sresp_wr = true;
  writer_trigger.notify();
  curr_tl0_cmd = curr_tl0_data = curr_tl0_resp = 0;
  curr_mthreadbusy = ~0;
  if(ocp_config.respaccept) respaccept_state = false;
}


// implementation of burst trackers and initialisation functions
namespace support_01 {

// class to implement pooling for burst tracker objects
template<class T, class BASE> struct pooled_tracker: BASE {
  static pooled_tracker *pool(pooled_tracker *used = 0) {
    static pooled_tracker *pool = 0;
    if(used == 0) {
      if(pool == 0) {pool = new T; pool->next = 0;}
      pooled_tracker *rv = pool;
      pool = pool->next;
      return rv;
    } else {
      used->next = pool;
      pool = used;
      return 0;
    }
  }
  pooled_tracker *next;
  void return_to_pool() {pool(this);}
};


struct precise_incr_tracker_base: tracker_base {
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    // address
    pl.set_address(a01.maddr());
    // burst_sequence
    extension_api::invalidate_extension<burst_sequence>(pl);
    // burst length
    extension_api::invalidate_extension<burst_length>(pl);
    // imprecise
    extension_api::invalidate_extension<imprecise>(pl);
    // initialise
    adapter = &a01;
    txn = &pl;
    // expected numbers of transfers
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    bool wresp = (a01.ocp_config.writeresp_enable || (a01.mcmd() != WR));
    unsigned len = a01.mburstlength();
    if(a01.mburstsinglereq()) {
      req.nr = 1;
      if(pl.is_read()) {resp.nr = len; data.nr = 0;}
      else {resp.nr = (wresp ? 1 : 0); data.nr = len;}
    } else {
      req.nr = len; resp.nr = (wresp ? len : 0);
      data.nr = (a01.ocp_config.datahandshake ? len : 0);
    }
    pl.set_data_length(len * a01.byte_wdth);
    if(pl.get_byte_enable_ptr() != 0)
      pl.set_byte_enable_length(len * a01.byte_wdth);
    pl.set_streaming_width(len * a01.byte_wdth);
  }

  virtual bool ReqDone(bool track) {
    // burst_length - not needed for precise burst
    // address - not needed for incr burst
    if(track) ++req.done;
    return(req.done == req.nr);
  }

  virtual bool WrDone(bool track) {
    if(track) ++data.done;
    return(data.done == data.nr);
  }

  virtual bool RdDone(bool track) {
    if(track) ++resp.done;
    bool done = (resp.done == resp.nr);
    adapter->sresprowlast(done);  // for non-BLCK same as resplast
    return(done);
  }
};


struct precise_incr_tracker:
  pooled_tracker<precise_incr_tracker, precise_incr_tracker_base>
{
};


struct precise_wrap_tracker:
  pooled_tracker<precise_wrap_tracker, precise_incr_tracker_base>
{
  unsigned offset;
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    precise_incr_tracker_base::init(pl, a01);
    uint64 a = a01.maddr();
    // burst_sequence
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, pl);
    extension_api::validate_extension<burst_sequence>(pl);
    bs->value.sequence = WRAP;
    bs->value.xor_wrap_address = a;
    // address
    unsigned l = a01.mburstlength() * adapter->byte_wdth;
    pl.set_address((a / l) * l);
    offset = (a - pl.get_address()) / adapter->byte_wdth;
  }
  virtual unsigned char *XDataPtr(handshake &hs) {
    unsigned off = ((offset + hs.done) % hs.nr) * adapter->byte_wdth;
    sc_assert(off < adapter->max_burst_bytes);
    return txn->get_data_ptr() + off;
  }
  virtual unsigned char *XBEPtr(handshake &hs) {
    unsigned off = ((offset + hs.done) % hs.nr) * adapter->byte_wdth;
    return txn->get_byte_enable_ptr() + off;
  }
};


struct precise_xor_tracker:
  pooled_tracker<precise_xor_tracker, precise_incr_tracker_base>
{
  unsigned offset;
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    precise_incr_tracker_base::init(pl, a01);
    uint64 a = a01.maddr();
    // burst_sequence
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, pl);
    extension_api::validate_extension<burst_sequence>(pl);
    bs->value.sequence = XOR;
    bs->value.xor_wrap_address = a;
    // address
    unsigned l = a01.mburstlength() * adapter->byte_wdth;
    pl.set_address((a / l) * l);
    offset = (a - pl.get_address()) / adapter->byte_wdth;
  }
  virtual unsigned char *XDataPtr(handshake &hs) {
    unsigned off = (offset ^ hs.done) * adapter->byte_wdth;
    sc_assert(off < adapter->max_burst_bytes);
    return txn->get_data_ptr() + off;
  }
  virtual unsigned char *XBEPtr(handshake &hs) {
    unsigned off = (offset ^ hs.done) * adapter->byte_wdth;
    return txn->get_byte_enable_ptr() + off;
  }
};


struct precise_strm_tracker:
  pooled_tracker<precise_strm_tracker, precise_incr_tracker_base>
{
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    precise_incr_tracker_base::init(pl, a01);
    pl.set_streaming_width(a01.byte_wdth);
    // for stream burst length 1 we need the extension as TLM is ambiguous
    if(pl.get_data_length() == a01.byte_wdth) {
      burst_sequence *bs;
      extension_api::get_extension<burst_sequence>(bs, pl);
      extension_api::validate_extension<burst_sequence>(pl);
      bs->value.sequence = STRM;
    }
  }
};


struct precise_dflt_tracker:
  pooled_tracker<precise_dflt_tracker, precise_incr_tracker_base>
{
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    precise_incr_tracker_base::init(pl, a01);
    // burst_sequence
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, pl);
    extension_api::validate_extension<burst_sequence>(pl);
    bs->value.unkn_dflt_addresses_valid = true;
    bs->value.unkn_dflt_bytes_per_address = a01.byte_wdth;
    unsigned l = pl.get_data_length() / a01.byte_wdth;
    if(bs->value.unkn_dflt_addresses.size() < l)
      bs->value.unkn_dflt_addresses.resize(l);
    bs->value.sequence = burst_seqs(a01.mburstseq());
  }
  virtual bool ReqDone(bool track) {
    if(track) {
      burst_sequence *bs;
      extension_api::get_extension<burst_sequence>(bs, *txn);
      bs->value.unkn_dflt_addresses[req.done] = adapter->maddr();
      ++req.done;
    }
    return(req.done == req.nr);
  }
};


struct precise_blck_tracker:
  pooled_tracker<precise_blck_tracker, precise_incr_tracker_base>
{
  unsigned width;
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    // address
    pl.set_address(a01.maddr());
    // burst length
    extension_api::invalidate_extension<burst_length>(pl);
    // imprecise
    extension_api::invalidate_extension<imprecise>(pl);
    // burst_sequence
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, pl);
    extension_api::validate_extension<burst_sequence>(pl);
    bs->value.sequence = BLCK;
    bs->value.block_height = a01.mblockheight();
    bs->value.block_stride = a01.mblockstride();
    width = a01.mburstlength();
    bs->value.blck_row_length_in_bytes = width * a01.byte_wdth;
    // initialise
    adapter = &a01;
    txn = &pl;
    // expected numbers of transfers
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    bool wresp = (a01.ocp_config.writeresp_enable || (a01.mcmd() != WR));
    unsigned len = width * bs->value.block_height;
    if(a01.mburstsinglereq()) {
      req.nr = 1;
      if(pl.is_read()) {resp.nr = len; data.nr = 0;}
      else {resp.nr = (wresp ? 1 : 0); data.nr = len;}
    } else {
      req.nr = len; resp.nr = (wresp ? len : 0);
      data.nr = (a01.ocp_config.datahandshake ? len : 0);
    }
    pl.set_data_length(len * a01.byte_wdth);
    if(pl.get_byte_enable_ptr() != 0)
      pl.set_byte_enable_length(len * a01.byte_wdth);
    pl.set_streaming_width(len * a01.byte_wdth);
  }
  virtual bool RdDone(bool track) {
    if(track) ++resp.done;
    adapter->sresprowlast((resp.done % width) == 0);
    return(resp.done == resp.nr);
  }
};


struct imprecise_incr_tracker_base: tracker_base {
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    // address
    pl.set_address(a01.maddr());
    // burst_sequence
    extension_api::invalidate_extension<burst_sequence>(pl);
    // burst length
    burst_length *bl;
    extension_api::get_extension<burst_length>(bl, pl);
    extension_api::validate_extension<burst_length>(pl);
    // imprecise
    extension_api::validate_extension<imprecise>(pl);
    // initialise
    adapter = &a01;
    txn = &pl;
    // expected numbers of transfers
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    bool wresp = (a01.ocp_config.writeresp_enable || (a01.mcmd() != WR));
    unsigned len = unsigned(-1);
    if(a01.mburstsinglereq()) {
      req.nr = 1;
      if(pl.is_read()) {resp.nr = len; data.nr = 0;}
      else {resp.nr = (wresp ? 1 : 0); data.nr = len;}
    } else {
      req.nr = len; resp.nr = (wresp ? len : 0);
      data.nr = (a01.ocp_config.datahandshake ? len : 0);
    }
    pl.set_data_length(a01.max_burst_bytes);
    if(pl.get_byte_enable_ptr() != 0)
      pl.set_byte_enable_length(a01.max_burst_bytes);
    pl.set_streaming_width(a01.max_burst_bytes);
  }

  virtual bool ReqDone(bool track) {
    // burst_length
    burst_length *bl;
    extension_api::get_extension<burst_length>(bl, *txn);
    extension_api::validate_extension<burst_length>(*txn);
    unsigned cbl = adapter->mburstlength();
    bl->value = cbl;
    // address - not needed for incr burst
    if(track) {
      ++req.done;
      sc_assert(req.done * adapter->byte_wdth <= adapter->max_burst_bytes);
      if(cbl == 1) {
        req.nr = req.done;
        if(resp.nr > 1) resp.nr = req.done;
        if(data.nr > 0) data.nr = req.done;
      }
    }
    return(req.nr == req.done);
  }

  virtual bool WrDone(bool track) {
    if(track) ++data.done;
    return(data.done == data.nr);
  }

  virtual bool RdDone(bool track) {
    if(track) ++resp.done;
    bool done = (resp.done == resp.nr);
    adapter->sresprowlast(done);  // for non-BLCK same as resplast
    return(done);
  }
};


struct imprecise_incr_tracker:
  pooled_tracker<imprecise_incr_tracker, imprecise_incr_tracker_base>
{
};


struct imprecise_strm_tracker:
  pooled_tracker<imprecise_strm_tracker, imprecise_incr_tracker_base>
{
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    imprecise_incr_tracker_base::init(pl, a01);
    pl.set_streaming_width(a01.byte_wdth);
    // for stream burst length 1 we need the extension as TLM is ambiguous
    if(1 == a01.mburstlength()) {
      burst_sequence *bs;
      extension_api::get_extension<burst_sequence>(bs, pl);
      extension_api::validate_extension<burst_sequence>(pl);
      bs->value.sequence = STRM;
    }
  }
};


struct imprecise_dflt_unkn_tracker:
  pooled_tracker<imprecise_dflt_unkn_tracker, imprecise_incr_tracker_base>
{
  virtual void init(tlm_generic_payload &pl, tl0_to_tl1_base &a01) {
    imprecise_incr_tracker_base::init(pl, a01);
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, pl);
    extension_api::validate_extension<burst_sequence>(pl);
    bs->value.sequence = burst_seqs(a01.mburstseq());
    bs->value.unkn_dflt_bytes_per_address = a01.byte_wdth;
    bs->value.unkn_dflt_addresses_valid = true;
    unsigned l = a01.max_burst_bytes / a01.byte_wdth;
    if(bs->value.unkn_dflt_addresses.size() < l)
      bs->value.unkn_dflt_addresses.resize(l);
  }
  virtual bool ReqDone(bool track) {
    // burst_length
    burst_length *bl;
    extension_api::get_extension<burst_length>(bl, *txn);
    extension_api::validate_extension<burst_length>(*txn);
    unsigned cbl = adapter->mburstlength();
    bl->value = cbl;
    // address
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, *txn);
    if(track) {
      bs->value.unkn_dflt_addresses[req.done] = adapter->maddr();
      ++req.done;
      sc_assert(req.done * adapter->byte_wdth <= adapter->max_burst_bytes);
      if(cbl == 1) {
        req.nr = req.done;
        if(resp.nr > 1) resp.nr = req.done;
        if(data.nr > 0) data.nr = req.done;
      }
    }
    return(req.nr == req.done);
  }
};


void init_ext(tlm_generic_payload &pl, tl0_to_tl1_base &adapter) {
  ext *e = pl.get_extension<ext>();
  if(e == 0) {
    e = new ext(adapter.max_burst_bytes,
      adapter.ocp_config.byteen || adapter.ocp_config.mdatabyteen);
    pl.set_extension<ext>(e);
  } else {
    e->tracker->return_to_pool();
  }

  // other generic payload fields
  pl.set_data_ptr(e->data_buffer);
  pl.set_byte_enable_ptr(e->be_buffer);
  pl.set_response_status(TLM_INCOMPLETE_RESPONSE);

  e->is_wrc = 0;
  e->is_rdex = 0;
  extension_api::invalidate_extension<semaphore>(pl);
  extension_api::invalidate_extension<posted>(pl);
  extension_api::invalidate_extension<lock>(pl);

  // ordering
  e->tag = (adapter.mtaginorder() ? -1 : adapter.mtagid());
  e->thread = adapter.mthreadid();
  if((adapter.ocp_config.tags > 1) && (e->tag >= 0)) {
    tag_id *tgid;
    extension_api::get_extension<tag_id>(tgid, pl);
    tgid->value = e->tag;
    extension_api::validate_extension<tag_id>(pl);
  }
  if(adapter.ocp_config.threads > 1) {
    thread_id *thid;
    extension_api::get_extension<thread_id>(thid, pl);
    thid->value = e->thread;
    extension_api::validate_extension<thread_id>(pl);
  }

  switch(adapter.mcmd()) {
    case CMD_WR:
      extension_api::validate_extension<posted>(pl);
    case CMD_WRNP:
      {
        // check if waiting for an unlocking write on this thread
        lock_object *lko = adapter.newest_lock;
        lock_object **prevlko = &adapter.newest_lock;
        while(lko != 0) {
          if(lko->ocp_thread == e->thread) {
             // lock the WR to the previous RDEX and delete from to-do list
            lock *lk;
            extension_api::get_extension<lock>(lk, pl);
            extension_api::validate_extension<lock>(pl);
            lk->value = lko;
            *prevlko = lko->next;
            lko = 0;
          } else {
            prevlko = &(lko->next);
            lko = lko->next;
          }
        }
      }
      pl.set_write();
      break;
    case CMD_WRC:
      {
        semaphore *sf;
        extension_api::get_extension<semaphore>(sf, pl);
        extension_api::validate_extension<semaphore>(pl);
        sf->value = false;
        e->is_wrc = sf;
      }
      pl.set_write();
      break;
    case CMD_RDEX:
      {
        lock *lk;
        extension_api::get_extension<lock>(lk, pl);
        extension_api::validate_extension<lock>(pl);
        lock_object *lko = lock_object::pool();
        lko->ocp_thread = e->thread;
        lko->lock_is_understood_by_slave = false;
        lko->number_of_txns = 2;
        // insert in linked list to enable next WR to be locked to it
        lko->next = adapter.newest_lock;
        adapter.newest_lock = lko;
        lk->value = lko;
        e->is_rdex = lko;
      }
    case CMD_RD:
      pl.set_read();
      break;
    case CMD_RDL:
      extension_api::validate_extension<semaphore>(pl);
      pl.set_read();
      break;
  }

  if(adapter.mburstprecise()) {
    switch(adapter.mburstseq()) {
      case INCR:  e->tracker = new precise_incr_tracker; break;
      case WRAP:  e->tracker = new precise_wrap_tracker; break;
      case XOR:  e->tracker = new precise_xor_tracker; break;
      case STRM:  e->tracker = new precise_strm_tracker; break;
      case DFLT1:
      case DFLT2:  e->tracker = new precise_dflt_tracker; break;
      case BLCK:  e->tracker = new precise_blck_tracker; break;
    }
  } else {
    switch(adapter.mburstseq()) {
      case INCR:  e->tracker = new imprecise_incr_tracker; break;
      case STRM:  e->tracker = new imprecise_strm_tracker; break;
      case DFLT1:
      case DFLT2:
      case UNKN:  e->tracker = new imprecise_dflt_unkn_tracker; break;
    }
  }
  e->tracker->init(pl, adapter);

  // other burst-invariant OCP extensions
  if(adapter.ocp_config.addrspace) {
    address_space *as;
    extension_api::get_extension<address_space>(as, pl);
    extension_api::validate_extension<address_space>(pl);
    as->value = adapter.maddrspace();
  }
  if(adapter.ocp_config.atomiclength) {
    atomic_length *al;
    extension_api::get_extension<atomic_length>(al, pl);
    extension_api::validate_extension<atomic_length>(pl);
    al->value = adapter.matomiclength();
  }
  if(adapter.ocp_config.connid) {
    conn_id *ci;
    extension_api::get_extension<conn_id>(ci, pl);
    extension_api::validate_extension<conn_id>(pl);
    ci->value = adapter.mconnid();
  }
  if(adapter.mburstsinglereq()) extension_api::validate_extension<srmd>(pl);
}


// end of namespaces
}}}
