///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  Common definitions for master and slave TL0-TL1 adapters
//
//
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef OCPIP_VERSION
  #error ocp_tl1_tl0_adapter_common.h may not be included directly. Use #include "ocpip_adapters.h" or #include "ocpip_adapters_X_X_X.h" (where desired ocp version is X.X.X)
#endif


// namespace
namespace OCPIP_VERSION { namespace tl1_tl0 {


// enums
enum ocp_cmd {
  CMD_IDLE = 0, CMD_WR, CMD_RD, CMD_RDEX, CMD_RDL, CMD_WRNP, CMD_WRC, CMD_BCST
};
enum ocp_resp {
  RESP_NULL = 0, RESP_DVA, RESP_FAIL, RESP_ERR
};
enum ocp_burst {
  BURST_INCR = 0, BURST_DFLT1, BURST_WRAP, BURST_DFLT2,
  BURST_XOR, BURST_STRM, BURST_UNKN, BURST_BLCK
};


// suppression of non-existent TL0 ports
template<typename T> T from_unsigned(unsigned def) {return T(def);}

template<unsigned DEF> class TIEOFF {
public:
  template<typename T> operator T() const {return from_unsigned<T>(DEF);}
  template<typename T> TIEOFF(const T &t) {}
  TIEOFF() {}
};

}} namespace sc_core {

template<unsigned DEF> class sc_in<OCPIP_VERSION::tl1_tl0::TIEOFF<DEF> > {
public:
  // constructor
  sc_in(const char *n = 0) {}
  // allow conversion to anything.  types without a constructor from unsigned
  // need to specialise from_unsigned<>()
  template<typename T>
    operator T() const {return OCPIP_VERSION::tl1_tl0::from_unsigned<T>(DEF);}
  // because OCP resets can be asynchronously asserted we also need
  const sc_event &default_event() const {return event;}
  sc_in *operator->() {return this;}
private:
  sc_event event;
};

template<unsigned DEF> class sc_out<OCPIP_VERSION::tl1_tl0::TIEOFF<DEF> > {
public:
  // constructor
  sc_out(const char *n = 0) {}
  // template operator = taking any type doing nothing
  template<typename T> void operator=(const T &t) {}
  // because OCP critical signals need to be zeroed at start of simulation
  template<typename T> void initialize(const T &t) {}
};

} namespace OCPIP_VERSION { namespace tl1_tl0 {


// replaceable classes for getting TL0 timing from the adapters
class get_TL0_timing_base {
public:
  virtual void
    slave(sc_core::sc_module *m, const ocp_tl1_slave_timing &t) = 0;
  virtual void
    master(sc_core::sc_module *m, const ocp_tl1_master_timing &t) = 0;
  virtual ~get_TL0_timing_base() {}
};

template<class OS> class get_TL0_timing_example:
  public get_TL0_timing_base, public sc_core::sc_module {
public:
  get_TL0_timing_example(OS *osp, sc_core::sc_module_name n = "TL0_T_Printer"):
    sc_core::sc_module(n), first(0), os(*osp) {}
  ~get_TL0_timing_example() {
    report *r = first;
    while(r != 0) {
      report *rd = r;
      r = r->next;
      delete rd;
    }
  }
  void slave(sc_core::sc_module *m, const ocp_tl1_slave_timing &t) {
    report *r = find(m);
    r->gotS = true;
    r->sT = t;
  }
  void master(sc_core::sc_module *m, const ocp_tl1_master_timing &t) {
    report *r = find(m);
    r->gotM = true;
    r->mT = t;
  }
  void start_of_simulation() {
    for(report *r = first; r != 0; r = r->next) {
      os << r->module->name() << " - current TL0 timing:" << '\n';
      if(r->gotS) {
        os << "  Resp: " << r->sT.ResponseGrpStartTime << '\n';
        os << "  CmdAccept: " << r->sT.SCmdAcceptStartTime << '\n';
        os << "  DataAccept: " << r->sT.SDataAcceptStartTime << '\n';
        os << "  ThreadBusy: " << r->sT.SThreadBusyStartTime << '\n';
        os << "  DataThreadBusy: " << r->sT.SDataThreadBusyStartTime << '\n';
      }
      if(r->gotM) {
        os << "  Cmd: " << r->mT.RequestGrpStartTime << '\n';
        os << "  DataHS: " << r->mT.DataHSGrpStartTime << '\n';
        os << "  RespAccept: " << r->mT.MRespAcceptStartTime << '\n';
        os << "  RespThreadBusy: " << r->mT.MThreadBusyStartTime << '\n';
      }
    }
  }
private:
  struct report {
    report *next;
    ocp_tl1_slave_timing sT;
    ocp_tl1_master_timing mT;
    bool gotS, gotM;
    sc_core::sc_module *module;
  };
  report *find(sc_core::sc_module *m) {
    for(report *r = first; r != 0; r = r->next) if(r->module == m) return r;
    report *s = new report;
    s->module = m;
    s->gotS = s->gotM = false;
    s->next = first;
    first = s;
    return s;
  }
  report *first;
  OS &os;
};


// traits classes for defining types of signal-level interfaces for TL0
//   side of adapters
// used as a template parameter to the adapter class (master or slave)
// these class can be used as a base class for rapid creation of
//   traits classes for other OCP configurations.  user only needs to define
//   types that do not match the defaults
// use of other tie-offs than those specified in the POD_DEFAULT_TL0_TYPES
//  is not supported.  If an OCP configuration does not include a signal,
//  use the type from POD_DEFAULT_TL0_TYPES, or the following:
//    MCMD_T - can never be a tie-off
//    MADDR_T, DATA_T - TIEOFF<x> where x is a don't care
//    SCMDACCEPT_T - TIEOFF<1>
//    MRESET_N_T - TIEOFF<1>
//    SRESP_T - TIEOFF<0>

// 1. based on POD types
template<
  typename DATA_TYPE=unsigned,
  typename ADDR_TYPE=unsigned
>
class POD_DEFAULT_TL0_TYPES
{
public:
  // clk-enable
  typedef TIEOFF<1> ENABLE_CLK_T;

  // basic
  typedef ADDR_TYPE MADDR_T;
  typedef unsigned MCMD_T;
  typedef DATA_TYPE DATA_T;
  typedef TIEOFF<1> MDATAVALID_T;
  typedef TIEOFF<1> MRESPACCEPT_T;
  typedef bool SCMDACCEPT_T;
  typedef TIEOFF<1> SDATAACCEPT_T;
  typedef unsigned SRESP_T;

  // simple
  typedef TIEOFF<0> MADDRSPACE_T;
  typedef TIEOFF<0xFFFFFFFF> MBYTEEN_T;
  typedef TIEOFF<0xFFFFFFFF> MDATABYTEEN_T;
  typedef TIEOFF<0> MDATAINFO_T;
  typedef TIEOFF<0> MREQINFO_T;
  typedef TIEOFF<0> SDATAINFO_T;
  typedef TIEOFF<0> SRESPINFO_T;

  // burst
  typedef TIEOFF<0> MATOMICLENGTH_T;
  typedef TIEOFF<1> MBLOCKHEIGHT_T;
  typedef TIEOFF<0> MBLOCKSTRIDE_T;
  typedef TIEOFF<1> MBURSTLENGTH_T;
  typedef TIEOFF<1> MBURSTPRECISE_T;
  typedef TIEOFF<0> MBURSTSEQ_T;
  typedef TIEOFF<0> MBURSTSINGLEREQ_T;
  typedef TIEOFF<0> MDATALAST_T;
  typedef TIEOFF<0> MDATAROWLAST_T;
  typedef TIEOFF<0> MREQLAST_T;
  typedef TIEOFF<0> MREQROWLAST_T;
  typedef TIEOFF<0> SRESPLAST_T;
  typedef TIEOFF<0> SRESPROWLAST_T;

  // tag
  typedef TIEOFF<0> MDATATAGID_T;
  typedef TIEOFF<0> MTAGID_T;
  typedef TIEOFF<0> MTAGINORDER_T;
  typedef TIEOFF<0> STAGID_T;
  typedef TIEOFF<0> STAGINORDER_T;

  // thread
  typedef TIEOFF<0> MCONNID_T;
  typedef TIEOFF<0> MDATATHREADID_T;
  typedef TIEOFF<0xFFFFFFFF> MTHREADBUSY_T;
  typedef TIEOFF<0> MTHREADID_T;
  typedef TIEOFF<0xFFFFFFFF> SDATATHREADBUSY_T;
  typedef TIEOFF<0xFFFFFFFF> STHREADBUSY_T;
  typedef TIEOFF<0> STHREADID_T;

  // sideband
  typedef TIEOFF<0> MERROR_T;
  typedef TIEOFF<0> MFLAG_T;
  typedef bool MRESET_N_T;
  typedef TIEOFF<0> SERROR_T;
  typedef TIEOFF<0> SFLAG_T;
  typedef TIEOFF<0> SINTERRUPT_T;
  typedef TIEOFF<1> SRESET_N_T;

  // functions for moving data bytes between user data type and unsigned char
  static void data_from_uchar(unsigned char b, unsigned lane, DATA_T &d) {
    DATA_T mask = ~(0xFF << (lane * 8));
    d = (d & mask) | (DATA_T(b) << (lane *8));
  }

  static unsigned char data_to_uchar(const DATA_T &d, unsigned lane) {
    return (unsigned char)(d >> (lane * 8));
  }

  // functions for user-defined in-band information conversion
  static void conv_mreqinfo(MREQINFO_T &info, tlm::tlm_generic_payload &pl) {}
  static void conv_mdatainfo(MDATAINFO_T &info, tlm::tlm_generic_payload &pl, unsigned beat) {}
  static void conv_srespinfo(SRESPINFO_T &info, tlm::tlm_generic_payload &pl) {}
  static void conv_sdatainfo(SDATAINFO_T &info, tlm::tlm_generic_payload &pl, unsigned beat) {}
  static void conv_mflag(MFLAG_T &flag, tlm::tlm_generic_payload &pl) {}
  static void conv_sflag(SFLAG_T &flag, tlm::tlm_generic_payload &pl) {}
};


// 2. based on SystemC vectorised integer types
// should work for all of sc_int, sc_uint, sc_bigint, sc_biguint, sc_bv, sc_lv
// a boolean type is also available so that for example sc_logic can be used
//   instead of sc_lv<1> if necessary
template<
  int DATA_WDTH=32,
  int ADDR_WDTH=32,
  template<int T> class SIGNAL_TEMPLATE = sc_dt::sc_bv,
  typename BOOL_TYPE = SIGNAL_TEMPLATE<1>
>
class SIGNAL_DEFAULT_TL0_TYPES: public POD_DEFAULT_TL0_TYPES<>
{
public:

  // basic
  typedef SIGNAL_TEMPLATE<ADDR_WDTH> MADDR_T;
  typedef SIGNAL_TEMPLATE<3> MCMD_T;
  typedef SIGNAL_TEMPLATE<DATA_WDTH> DATA_T;
  typedef BOOL_TYPE SCMDACCEPT_T;
  typedef SIGNAL_TEMPLATE<2> SRESP_T;

  // sideband
  typedef BOOL_TYPE MRESET_N_T;

  // functions for moving data bytes between user data type and unsigned char
  static void data_from_uchar(unsigned char b, unsigned lane, DATA_T &d) {
    d.range(lane*8+7,lane*8) = b;
  }

  static unsigned char data_to_uchar(const DATA_T &d, unsigned lane) {
    return (unsigned char)(d.range(lane*8+7,lane*8).to_uint());
  }
};


// conversion function templates for reading from ports with arbitrary
// types.  need only bool, unsigned and uint64 return types.
// specialise as required
template<typename T>
bool port_to_bool(const sc_core::sc_in<T> &port) {
  return (0 != port->read());
}
template<typename T>
unsigned port_to_unsigned(const sc_core::sc_in<T> &port) {
  return (port->read()).to_uint();
}
template<typename T>
sc_dt::uint64 port_to_uint64(const sc_core::sc_in<T> &port) {
  return (port->read()).to_uint64();
}
// specialised implementations for TIEOFF<>
template<unsigned DEF> inline
bool port_to_bool(const sc_core::sc_in<TIEOFF<DEF> > &port) {
  return bool(DEF);
}
template<unsigned DEF> inline
static unsigned port_to_unsigned(const sc_core::sc_in<TIEOFF<DEF> > &port) {
  return DEF;
}
template<unsigned DEF> inline
sc_dt::uint64 port_to_uint64(const sc_core::sc_in<TIEOFF<DEF> > &port) {
  return sc_dt::uint64(DEF);
}
// specialised implementations for POD types
// bool
template<> inline
bool port_to_bool(const sc_core::sc_in<bool> &port) {
  return port->read();
}
template<> inline
bool port_to_bool(const sc_core::sc_in<unsigned> &port) {
  return (0 != port->read());
}
template<> inline
bool port_to_bool(const sc_core::sc_in<int> &port) {
  return (0 != port->read());
}
// unsigned
template<> inline
unsigned port_to_unsigned(const sc_core::sc_in<unsigned> &port) {
  return port->read();
}
template<> inline
unsigned port_to_unsigned(const sc_core::sc_in<int> &port) {
  return unsigned(port->read());
}
// uint64
template<> inline
sc_dt::uint64 port_to_uint64(const sc_core::sc_in<unsigned> &port) {
  return sc_dt::uint64(port->read());
}
template<> inline
sc_dt::uint64 port_to_uint64(const sc_core::sc_in<int> &port) {
  return sc_dt::uint64(port->read());
}
template<> inline
sc_dt::uint64 port_to_uint64(const sc_core::sc_in<sc_dt::uint64> &port) {
  return port->read();
}


// end namespaces
}}

