///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2010
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Herve Alexanian - Sonics, inc.
//
//          $Id:
//
//  Description :  This file defines utility classes designed to be used when
//                 writing adapters for TL1 or TL2 sockets
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef OCPIP_VERSION
  #error ocp_adapter_utils.h may not be included directly. Use #include "ocpip.h" or #include "ocpip_X_X_X.h" (where desired ocp version is X.X.X)
#endif

namespace OCPIP_VERSION {
struct adapter_error : public std::exception {
    std::string m_msg;
    adapter_error( const std::string& msg ) :
        m_msg( msg )
        {}
    ~adapter_error() throw() {};
    const char* what() const throw() {
        return m_msg.c_str();
    }
};

template <typename index_t>
struct arb_interleave_policy_none {
    void filter( std::set<index_t>& candidates ) {}
    void winner( const index_t&, bool ) {}
};

template <typename item_t> class arbiter_filter_if;
template <typename item_t, template <typename item_t> class algo_pol_t>
struct simple_arb {
    simple_arb() { m_min = m_max = 0; } // invalid starting condition until set_range() is called
    ~simple_arb();
    void set_range( const item_t& min, const item_t& max );
    bool operator()( const std::set<item_t> candidates, item_t& winner );
    void winner( const item_t& item, bool end_of_block=false );
    template <typename filter_t>
    void add_filter( filter_t f ) {
	m_filters.push_back( new filter_t( f ) );
    }
private:
    algo_pol_t<item_t> m_algo;
    std::deque<arbiter_filter_if<item_t>*> m_filters;
    item_t m_min, m_max;
};
    
template <typename item_t>
struct simple_rr_arb_algo {
    void set_range( const item_t& min, const item_t& max );
    bool operator()( const std::set<item_t>& candidates, item_t& result );
    void winner( const item_t& item, bool end_of_block=false );
private:
    std::vector<item_t> m_rr_order;
};

template <typename item_t>
struct arbiter_filter_if {
public:
    virtual ~arbiter_filter_if() {}
    virtual void operator()( std::set<item_t>& candidates ) = 0;
    virtual void winner( const item_t& item, bool end_of_block=false ) = 0;
};

template <typename item_t, typename busy_t=uint32_t>
struct arbiter_threadbusy_filter : public arbiter_filter_if<item_t> {
    arbiter_threadbusy_filter( const busy_t& word );
    arbiter_threadbusy_filter();
    virtual void operator()( std::set<item_t>& candidates );
    virtual void winner( const item_t& item, bool end_of_block=false ) {}
    void set_busy_word( const busy_t& word );
    bool is_busy( const item_t& ) const;
private:
    const busy_t* m_p_busy_word;
};


template <typename item_t>
struct arbiter_interleave_filter : public arbiter_filter_if<item_t> {
    arbiter_interleave_filter( uint32_t size=1, bool optional=false );
    virtual void operator()( std::set<item_t>& candidates );
    virtual void winner( const item_t& item, bool end_of_block=false );
private:
    uint32_t m_size;
    uint32_t m_num_interleaves;
    bool     m_optional;
    bool     m_locked;
    item_t   m_locked_to;
};

template <typename item_t,template <typename item_t, typename alloc_t=std::allocator<item_t> > class container_t=std::deque>
class simple_phase_queue : public tlm::tlm_nonblocking_put_if<item_t>, public tlm::tlm_nonblocking_get_peek_if<item_t>,
                           public container_t<item_t>
{
public:
    typedef tlm::tlm_nonblocking_get_peek_if<item_t>     nb_get_type;
    typedef container_t<item_t, std::allocator<item_t> > container_type;
    simple_phase_queue( size_t size_ ); // size_ = 0 means unlimited queuing
    virtual bool nb_put( const item_t& );
    virtual bool nb_can_put( tlm::tlm_tag<item_t> *t = 0 ) const;
    virtual const sc_core::sc_event& ok_to_put( tlm::tlm_tag<item_t> *t = 0 ) const;
    virtual bool nb_get( item_t& );
    virtual bool nb_can_get( tlm::tlm_tag<item_t> *t = 0 ) const;
    virtual const sc_core::sc_event& ok_to_get( tlm::tlm_tag<item_t> *t = 0 ) const;
    virtual bool nb_peek( item_t& ) const;
    virtual bool nb_can_peek( tlm::tlm_tag<item_t> *t = 0 ) const;
    virtual const sc_core::sc_event& ok_to_peek( tlm::tlm_tag<item_t> *t = 0 ) const;
private:
    size_t m_max_size;
    std::back_insert_iterator<container_type> m_insert;
    sc_core::sc_event m_not_full_ev;
    sc_core::sc_event m_not_empty_ev;
};

template <template <typename item_t, typename alloc_t=std::allocator<item_t> > class container_t, typename queue_t>
std::set<typename container_t<queue_t*>::iterator>
collect_valid_items( container_t<queue_t*>& cont ) {
    std::set<typename container_t<queue_t*>::iterator> ret;
    for ( typename container_t<queue_t*>::iterator it = cont.begin(); it != cont.end(); ++it ) {
	if ( (*it)->nb_can_peek() )
	    ret.insert( it );
    }
    return ret;
}
template <template <typename item_t, typename alloc_t=std::allocator<item_t> > class container_t, typename queue_t>
std::set<size_t>
collect_valid_item_numbers( container_t<queue_t*>& cont ) {
    std::set<size_t> ret;
    for ( typename container_t<queue_t*>::iterator it = cont.begin(); it != cont.end(); ++it ) {
	if ( (*it)->nb_can_peek() )
	    ret.insert( it - cont.begin() );
    }
    return ret;
}


template <typename extension_t>
inline
extension_t* check_ispec_extension(
    tlm::tlm_generic_payload& txn, tlm_utils::instance_specific_extension_accessor acc )
{
    extension_t* p_ext = NULL;
    acc(txn).get_extension( p_ext );
    return p_ext; // will stay NULL if extension is absent
}

template <typename extension_t>
inline
extension_t* require_ispec_extension(
    tlm::tlm_generic_payload& txn, tlm_utils::instance_specific_extension_accessor acc )
{
    extension_t* p_ext = NULL;
    acc(txn).get_extension( p_ext );
    assert( p_ext != NULL && "transaction was assumed to have been fitted with an instance specific extension" );
    return p_ext;
}

template <typename payload_t, typename delay_t=uint32_t>
class ocp_cycle_based_peq : public sc_core::sc_module
{
public:
    typedef payload_t transaction_type;
    SC_HAS_PROCESS( ocp_cycle_based_peq );
    ocp_cycle_based_peq( sc_core::sc_module_name nm );
    sc_core::sc_in_clk m_clk;

    void notify( payload_t&, delay_t );
    void reset() { m_scheduled.clear(); m_cycle_count = 0; }
    payload_t* get_next_transaction();
    const sc_core::sc_event& get_event() { return m_event; }

private:
    void process();
    sc_core::sc_event                           m_new_scheduling_ev;
    sc_core::sc_event                           m_event;
    delay_t                                     m_cycle_count;
    
    std::multimap<uint32_t, payload_t*> m_scheduled;
};

struct ocp_tl1_phase_pos {
    tlm::tlm_generic_payload*    txn;
    ocp_txn_track::txn_position  position;
    int                          delay;
    ocp_tl1_phase_pos() { txn = NULL; delay = 0; }
};

template <typename delay_t>
struct phase_delay_calc_if {
    virtual ~phase_delay_calc_if() {}
    virtual delay_t calc( ocp_txn_track_base::txn_position, tlm::tlm_generic_payload* ) = 0;
};

template <typename delay_t>
struct phase_delay_gen_if {
    virtual delay_t operator()() = 0;
    virtual ~phase_delay_gen_if() {}
};

template <typename delay_t>
struct phase_delay_basic_burst_calc : public phase_delay_calc_if<delay_t> {
    phase_delay_basic_burst_calc( delay_t init_delay, delay_t beat_delay ) :
	m_init_delay( init_delay ),
	m_beat_delay( beat_delay )
	{}
    virtual delay_t calc( ocp_txn_track_base::txn_position pos, tlm::tlm_generic_payload* ) {
	// Note this is meant to be called BEFORE a phase is position tracked
	//assert( pos.count > 0 );
	if ( pos.count == 1 ) {
	    return m_init_delay;
	} else {
	    return m_beat_delay;
	}
    }
    private:
    delay_t m_init_delay, m_beat_delay;
};

#ifdef SCV_RANDOM_H
template <typename delay_t>
struct phase_delay_scv_range_random_gen : public phase_delay_gen_if<delay_t> {
    phase_delay_scv_range_random_gen( delay_t min_, delay_t max_ ) :
	m_min_delay( min_ ),
	m_max_delay( max_ )
	{}
    virtual delay_t operator()() {
	delay_t range = m_max_delay - m_min_delay;
	range++;
	return m_min_delay + ( m_gen.next() % range );
    }
    private:
    delay_t    m_min_delay, m_max_delay;
    scv_random m_gen;
};
#endif

template <typename delay_t,
#ifdef SCV_RANDOM_H
    typename gen_t=phase_delay_scv_range_random_gen<delay_t> >
#else
    typename gen_t>
#endif
struct phase_delay_range_random_calc : public phase_delay_calc_if<delay_t> {
    phase_delay_range_random_calc( delay_t min_, delay_t max_ ) :
	m_gen( min_, max_ )
	{}
    virtual delay_t calc( ocp_txn_track_base::txn_position, tlm::tlm_generic_payload* ) {
	return m_gen();
    }
    private:
    gen_t m_gen;
};

template <typename module_t, typename delay_t=uint32_t, typename txn_t=tlm::tlm_generic_payload,
	  template <typename payload_t, typename delay_t> class peq_t=ocp_cycle_based_peq>
struct phase_delayed_accept : public sc_core::sc_module
{
public:
    typedef  peq_t<txn_t, delay_t> peq_type;
    typedef void (module_t::*callback_type)( txn_t& );
    SC_HAS_PROCESS( phase_delayed_accept );

    phase_delayed_accept( sc_core::sc_module_name nm );
    sc_core::sc_in_clk m_clk;
    void register_callback  ( module_t* p_mod, callback_type cb )
	{ m_p_mod = p_mod; m_cb = cb; }
    void accept( txn_t&, delay_t );

private:
    peq_type         m_peq;
    void             on_peq();
    callback_type    m_cb;
    module_t*        m_p_mod;
};
}

#include __MACRO_STRINGYFY__(../src/OCPIP_VERSION/ocp_adapter_utils.tpp)
